import ts from "typescript";
import { ChangeAction } from "../../../utils/textChanges.js";
import AccessExpressionBaseFix from "./AccessExpressionBaseFix.js";
/**
 * Fix a property access. This could also be the property access of a call expression, allowing for a more general
 * replacement in cases where the arguments or other conditions of the call expression do not matter.
 */
export default class AccessExpressionFix extends AccessExpressionBaseFix {
    params;
    constructor(params, ui5TypeInfo) {
        super(params, ui5TypeInfo);
        this.params = params;
    }
    visitAutofixNode(node, position, sourceFile) {
        if (!super.visitAutofixNode(node, position, sourceFile)) {
            return false;
        }
        if (!ts.isPropertyAccessExpression(node) && !ts.isElementAccessExpression(node)) {
            return false;
        }
        let relevantNode = node;
        for (let i = 0; i < (this.params.scope ?? 0); i++) {
            if (!ts.isPropertyAccessExpression(relevantNode) &&
                !ts.isElementAccessExpression(relevantNode)) {
                return false;
            }
            relevantNode = relevantNode.expression;
        }
        if (!this.requestsModuleOrGlobal) {
            // If no module or global is requested, we assume the current property access should stay.
            // Therefore, ignore the expression of the "relevant node" and start at the name
            if (!ts.isPropertyAccessExpression(relevantNode)) {
                // We can't replace an element access expression like this
                return false;
            }
            this.startPos = relevantNode.name.getStart(sourceFile);
        }
        else {
            this.startPos = relevantNode.getStart(sourceFile);
        }
        this.endPos = relevantNode.getEnd();
        return true;
    }
    generateChanges() {
        if (this.startPos === undefined || this.endPos === undefined) {
            throw new Error("Start and end position are not defined");
        }
        let value;
        if (this.requestsModuleOrGlobal) {
            const identifier = this.getIdentifiersForSingleRequest(this.params.moduleName, this.params.globalName);
            if (!identifier) {
                // Requests could not be fulfilled, do not generate a change
                return;
            }
            value = identifier;
            if (this.params.propertyAccess) {
                // If a property is defined, we need to access it on the identifier
                value = `${value}.${this.params.propertyAccess}`;
            }
        }
        else {
            value = this.params.propertyAccess ?? "";
        }
        return {
            action: ChangeAction.REPLACE,
            start: this.startPos,
            end: this.endPos,
            value,
        };
    }
}
//# sourceMappingURL=AccessExpressionFix.js.map