'use strict';

var eslintVisitorKeys = require('eslint-visitor-keys');
var espree = require('espree');

const anyFunctionPattern = /^(?:Function(?:Declaration|Expression)|ArrowFunctionExpression)$/u;
const COMMENTS_IGNORE_PATTERN = /^\s*(?:eslint|jshint\s+|jslint\s+|istanbul\s+|globals?\s+|exported\s+|jscs)/u;
const LINEBREAKS = /* @__PURE__ */ new Set(["\r\n", "\r", "\n", "\u2028", "\u2029"]);
const LINEBREAK_MATCHER = /\r\n|[\r\n\u2028\u2029]/u;
const STATEMENT_LIST_PARENTS = /* @__PURE__ */ new Set(["Program", "BlockStatement", "StaticBlock", "SwitchCase"]);
const DECIMAL_INTEGER_PATTERN = /^(?:0|0[0-7]*[89]\d*|[1-9](?:_?\d)*)$/u;
const OCTAL_OR_NON_OCTAL_DECIMAL_ESCAPE_PATTERN = /^(?:[^\\]|\\.)*\\(?:[1-9]|0[0-9])/su;
function createGlobalLinebreakMatcher() {
  return new RegExp(LINEBREAK_MATCHER.source, "gu");
}
function getUpperFunction(node) {
  for (let currentNode = node; currentNode; currentNode = currentNode.parent) {
    if (anyFunctionPattern.test(currentNode.type))
      return currentNode;
  }
  return null;
}
function isFunction(node) {
  return Boolean(node && anyFunctionPattern.test(node.type));
}
function isNullLiteral(node) {
  return node.type === "Literal" && node.value === null && !("regex" in node) && !("bigint" in node);
}
function getStaticStringValue(node) {
  switch (node.type) {
    case "Literal":
      if (node.value === null) {
        if (isNullLiteral(node))
          return String(node.value);
        if ("regex" in node && node.regex)
          return `/${node.regex.pattern}/${node.regex.flags}`;
        if ("bigint" in node && node.bigint)
          return node.bigint;
      } else {
        return String(node.value);
      }
      break;
    case "TemplateLiteral":
      if (node.expressions.length === 0 && node.quasis.length === 1)
        return node.quasis[0].value.cooked;
      break;
  }
  return null;
}
function getStaticPropertyName(node) {
  let prop;
  if (node) {
    switch (node.type) {
      case "ChainExpression":
        return getStaticPropertyName(node.expression);
      case "Property":
      case "PropertyDefinition":
      case "MethodDefinition":
        prop = node.key;
        break;
      case "MemberExpression":
        prop = node.property;
        break;
    }
  }
  if (prop) {
    if (prop.type === "Identifier" && !("computed" in node && node.computed))
      return prop.name;
    return getStaticStringValue(prop);
  }
  return null;
}
function skipChainExpression(node) {
  return node && node.type === "ChainExpression" ? node.expression : node;
}
function negate$1(f) {
  return (token) => !f(token);
}
function isParenthesised(sourceCode, node) {
  const previousToken = sourceCode.getTokenBefore(node);
  const nextToken = sourceCode.getTokenAfter(node);
  return !!previousToken && !!nextToken && previousToken.value === "(" && previousToken.range[1] <= node.range[0] && nextToken.value === ")" && nextToken.range[0] >= node.range[1];
}
function isEqToken(token) {
  return token.value === "=" && token.type === "Punctuator";
}
function isArrowToken$1(token) {
  return token.value === "=>" && token.type === "Punctuator";
}
function isCommaToken$1(token) {
  return token.value === "," && token.type === "Punctuator";
}
function isQuestionDotToken(token) {
  return token.value === "?." && token.type === "Punctuator";
}
function isSemicolonToken$1(token) {
  return token.value === ";" && token.type === "Punctuator";
}
function isColonToken$1(token) {
  return token.value === ":" && token.type === "Punctuator";
}
function isOpeningParenToken$1(token) {
  return token.value === "(" && token.type === "Punctuator";
}
function isClosingParenToken$1(token) {
  return token.value === ")" && token.type === "Punctuator";
}
function isOpeningBracketToken$1(token) {
  return token.value === "[" && token.type === "Punctuator";
}
function isClosingBracketToken$1(token) {
  return token.value === "]" && token.type === "Punctuator";
}
function isOpeningBraceToken$1(token) {
  return token.value === "{" && token.type === "Punctuator";
}
function isClosingBraceToken$1(token) {
  return token.value === "}" && token.type === "Punctuator";
}
function isCommentToken$1(token) {
  if (!token)
    return false;
  return token.type === "Line" || token.type === "Block" || token.type === "Shebang";
}
function isKeywordToken(token) {
  return token.type === "Keyword";
}
function isLogicalExpression(node) {
  return node.type === "LogicalExpression" && (node.operator === "&&" || node.operator === "||");
}
function isCoalesceExpression(node) {
  return node.type === "LogicalExpression" && node.operator === "??";
}
function isMixedLogicalAndCoalesceExpressions(left, right) {
  return isLogicalExpression(left) && isCoalesceExpression(right) || isCoalesceExpression(left) && isLogicalExpression(right);
}
function getSwitchCaseColonToken(node, sourceCode) {
  if ("test" in node && node.test)
    return sourceCode.getTokenAfter(node.test, (token) => isColonToken$1(token));
  return sourceCode.getFirstToken(node, 1);
}
function isTopLevelExpressionStatement(node) {
  if (node.type !== "ExpressionStatement")
    return false;
  const parent = node.parent;
  return parent.type === "Program" || parent.type === "BlockStatement" && isFunction(parent.parent);
}
function isDirective(node) {
  return node.type === "ExpressionStatement" && typeof node.directive === "string";
}
function isTokenOnSameLine(left, right) {
  return left?.loc?.end.line === right?.loc?.start.line;
}
const isNotClosingParenToken = negate$1(isClosingParenToken$1);
const isNotCommaToken = negate$1(isCommaToken$1);
const isNotQuestionDotToken = negate$1(isQuestionDotToken);
const isNotOpeningParenToken = negate$1(isOpeningParenToken$1);
const isNotSemicolonToken = negate$1(isSemicolonToken$1);
function isStringLiteral(node) {
  return node.type === "Literal" && typeof node.value === "string" || node.type === "TemplateLiteral";
}
function isSurroundedBy(val, character) {
  return val[0] === character && val[val.length - 1] === character;
}
function getPrecedence(node) {
  switch (node.type) {
    case "SequenceExpression":
      return 0;
    case "AssignmentExpression":
    case "ArrowFunctionExpression":
    case "YieldExpression":
      return 1;
    case "ConditionalExpression":
      return 3;
    case "LogicalExpression":
      switch (node.operator) {
        case "||":
        case "??":
          return 4;
        case "&&":
          return 5;
      }
    case "BinaryExpression":
      switch (node.operator) {
        case "|":
          return 6;
        case "^":
          return 7;
        case "&":
          return 8;
        case "==":
        case "!=":
        case "===":
        case "!==":
          return 9;
        case "<":
        case "<=":
        case ">":
        case ">=":
        case "in":
        case "instanceof":
          return 10;
        case "<<":
        case ">>":
        case ">>>":
          return 11;
        case "+":
        case "-":
          return 12;
        case "*":
        case "/":
        case "%":
          return 13;
        case "**":
          return 15;
      }
    case "UnaryExpression":
    case "AwaitExpression":
      return 16;
    case "UpdateExpression":
      return 17;
    case "CallExpression":
    case "ChainExpression":
    case "ImportExpression":
      return 18;
    case "NewExpression":
      return 19;
    default:
      if (node.type in eslintVisitorKeys.KEYS)
        return 20;
      return -1;
  }
}
function isDecimalInteger(node) {
  return node.type === "Literal" && typeof node.value === "number" && DECIMAL_INTEGER_PATTERN.test(node.raw);
}
function isDecimalIntegerNumericToken(token) {
  return token.type === "Numeric" && DECIMAL_INTEGER_PATTERN.test(token.value);
}
function getNextLocation(sourceCode, { column, line }) {
  if (column < sourceCode.lines[line - 1].length) {
    return {
      column: column + 1,
      line
    };
  }
  if (line < sourceCode.lines.length) {
    return {
      column: 0,
      line: line + 1
    };
  }
  return null;
}
function isNumericLiteral(node) {
  return node.type === "Literal" && (typeof node.value === "number" || Boolean("bigint" in node && node.bigint));
}
function canTokensBeAdjacent(leftValue, rightValue) {
  const espreeOptions = {
    comment: true,
    ecmaVersion: espree.latestEcmaVersion,
    range: true
  };
  let leftToken;
  if (typeof leftValue === "string") {
    let tokens;
    try {
      tokens = espree.tokenize(leftValue, espreeOptions);
    } catch {
      return false;
    }
    const comments = tokens.comments;
    leftToken = tokens[tokens.length - 1];
    if (comments.length) {
      const lastComment = comments[comments.length - 1];
      if (!leftToken || lastComment.range[0] > leftToken.range[0])
        leftToken = lastComment;
    }
  } else {
    leftToken = leftValue;
  }
  if (leftToken.type === "Shebang" || leftToken.type === "Hashbang")
    return false;
  let rightToken;
  if (typeof rightValue === "string") {
    let tokens;
    try {
      tokens = espree.tokenize(rightValue, espreeOptions);
    } catch {
      return false;
    }
    const comments = tokens.comments;
    rightToken = tokens[0];
    if (comments.length) {
      const firstComment = comments[0];
      if (!rightToken || firstComment.range[0] < rightToken.range[0])
        rightToken = firstComment;
    }
  } else {
    rightToken = rightValue;
  }
  if (leftToken.type === "Punctuator" || rightToken.type === "Punctuator") {
    if (leftToken.type === "Punctuator" && rightToken.type === "Punctuator") {
      const PLUS_TOKENS = /* @__PURE__ */ new Set(["+", "++"]);
      const MINUS_TOKENS = /* @__PURE__ */ new Set(["-", "--"]);
      return !(PLUS_TOKENS.has(leftToken.value) && PLUS_TOKENS.has(rightToken.value) || MINUS_TOKENS.has(leftToken.value) && MINUS_TOKENS.has(rightToken.value));
    }
    if (leftToken.type === "Punctuator" && leftToken.value === "/")
      return !["Block", "Line", "RegularExpression"].includes(rightToken.type);
    return true;
  }
  if (leftToken.type === "String" || rightToken.type === "String" || leftToken.type === "Template" || rightToken.type === "Template")
    return true;
  if (leftToken.type !== "Numeric" && rightToken.type === "Numeric" && rightToken.value.startsWith("."))
    return true;
  if (leftToken.type === "Block" || rightToken.type === "Block" || rightToken.type === "Line")
    return true;
  if (rightToken.type === "PrivateIdentifier")
    return true;
  return false;
}
function hasOctalOrNonOctalDecimalEscapeSequence(rawString) {
  return OCTAL_OR_NON_OCTAL_DECIMAL_ESCAPE_PATTERN.test(rawString);
}

function createRule(rule) {
  return rule;
}

const ASCII_REGEX = /^[\u0000-\u007F]*$/u;
let segmenter;
function getGraphemeCount(value) {
  if (ASCII_REGEX.test(value))
    return value.length;
  if (!segmenter)
    segmenter = new Intl.Segmenter();
  return [...segmenter.segment(value)].length;
}

var keywords = [
  "abstract",
  "boolean",
  "break",
  "byte",
  "case",
  "catch",
  "char",
  "class",
  "const",
  "continue",
  "debugger",
  "default",
  "delete",
  "do",
  "double",
  "else",
  "enum",
  "export",
  "extends",
  "false",
  "final",
  "finally",
  "float",
  "for",
  "function",
  "goto",
  "if",
  "implements",
  "import",
  "in",
  "instanceof",
  "int",
  "interface",
  "long",
  "native",
  "new",
  "null",
  "package",
  "private",
  "protected",
  "public",
  "return",
  "short",
  "static",
  "super",
  "switch",
  "synchronized",
  "this",
  "throw",
  "throws",
  "transient",
  "true",
  "try",
  "typeof",
  "var",
  "void",
  "volatile",
  "while",
  "with"
];

/**
 * Negate the result of `this` calling.
 * @param {Token} token The token to check.
 * @returns {boolean} `true` if the result of `this(token)` is `false`.
 */
function negate0(token) {
    return !this(token) //eslint-disable-line no-invalid-this
}

/**
 * Creates the negate function of the given function.
 * @param {function(Token):boolean} f - The function to negate.
 * @returns {function(Token):boolean} Negated function.
 */
function negate(f) {
    return negate0.bind(f)
}

/**
 * Checks if the given token is a PunctuatorToken with the given value
 * @param {Token} token - The token to check.
 * @param {string} value - The value to check.
 * @returns {boolean} `true` if the token is a PunctuatorToken with the given value.
 */
function isPunctuatorTokenWithValue(token, value) {
    return token.type === "Punctuator" && token.value === value
}

/**
 * Checks if the given token is an arrow token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is an arrow token.
 */
function isArrowToken(token) {
    return isPunctuatorTokenWithValue(token, "=>")
}

/**
 * Checks if the given token is a comma token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is a comma token.
 */
function isCommaToken(token) {
    return isPunctuatorTokenWithValue(token, ",")
}

/**
 * Checks if the given token is a semicolon token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is a semicolon token.
 */
function isSemicolonToken(token) {
    return isPunctuatorTokenWithValue(token, ";")
}

/**
 * Checks if the given token is a colon token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is a colon token.
 */
function isColonToken(token) {
    return isPunctuatorTokenWithValue(token, ":")
}

/**
 * Checks if the given token is an opening parenthesis token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is an opening parenthesis token.
 */
function isOpeningParenToken(token) {
    return isPunctuatorTokenWithValue(token, "(")
}

/**
 * Checks if the given token is a closing parenthesis token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is a closing parenthesis token.
 */
function isClosingParenToken(token) {
    return isPunctuatorTokenWithValue(token, ")")
}

/**
 * Checks if the given token is an opening square bracket token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is an opening square bracket token.
 */
function isOpeningBracketToken(token) {
    return isPunctuatorTokenWithValue(token, "[")
}

/**
 * Checks if the given token is a closing square bracket token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is a closing square bracket token.
 */
function isClosingBracketToken(token) {
    return isPunctuatorTokenWithValue(token, "]")
}

/**
 * Checks if the given token is an opening brace token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is an opening brace token.
 */
function isOpeningBraceToken(token) {
    return isPunctuatorTokenWithValue(token, "{")
}

/**
 * Checks if the given token is a closing brace token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is a closing brace token.
 */
function isClosingBraceToken(token) {
    return isPunctuatorTokenWithValue(token, "}")
}

/**
 * Checks if the given token is a comment token or not.
 * @param {Token} token - The token to check.
 * @returns {boolean} `true` if the token is a comment token.
 */
function isCommentToken(token) {
    return ["Block", "Line", "Shebang"].includes(token.type)
}

negate(isArrowToken);
negate(isCommaToken);
negate(isSemicolonToken);
negate(isColonToken);
negate(isOpeningParenToken);
negate(isClosingParenToken);
negate(isOpeningBracketToken);
negate(isClosingBracketToken);
negate(isOpeningBraceToken);
negate(isClosingBraceToken);
negate(isCommentToken);

Object.freeze(
    new Set([
        "Array",
        "ArrayBuffer",
        "BigInt",
        "BigInt64Array",
        "BigUint64Array",
        "Boolean",
        "DataView",
        "Date",
        "decodeURI",
        "decodeURIComponent",
        "encodeURI",
        "encodeURIComponent",
        "escape",
        "Float32Array",
        "Float64Array",
        "Function",
        "Infinity",
        "Int16Array",
        "Int32Array",
        "Int8Array",
        "isFinite",
        "isNaN",
        "isPrototypeOf",
        "JSON",
        "Map",
        "Math",
        "NaN",
        "Number",
        "Object",
        "parseFloat",
        "parseInt",
        "Promise",
        "Proxy",
        "Reflect",
        "RegExp",
        "Set",
        "String",
        "Symbol",
        "Uint16Array",
        "Uint32Array",
        "Uint8Array",
        "Uint8ClampedArray",
        "undefined",
        "unescape",
        "WeakMap",
        "WeakSet",
    ]),
);
new Set(
    [
        Array.isArray,
        Array.of,
        Array.prototype.at,
        Array.prototype.concat,
        Array.prototype.entries,
        Array.prototype.every,
        Array.prototype.filter,
        Array.prototype.find,
        Array.prototype.findIndex,
        Array.prototype.flat,
        Array.prototype.includes,
        Array.prototype.indexOf,
        Array.prototype.join,
        Array.prototype.keys,
        Array.prototype.lastIndexOf,
        Array.prototype.slice,
        Array.prototype.some,
        Array.prototype.toString,
        Array.prototype.values,
        typeof BigInt === "function" ? BigInt : undefined,
        Boolean,
        Date,
        Date.parse,
        decodeURI,
        decodeURIComponent,
        encodeURI,
        encodeURIComponent,
        escape,
        isFinite,
        isNaN,
        isPrototypeOf,
        Map,
        Map.prototype.entries,
        Map.prototype.get,
        Map.prototype.has,
        Map.prototype.keys,
        Map.prototype.values,
        ...Object.getOwnPropertyNames(Math)
            .filter((k) => k !== "random")
            .map((k) => Math[k])
            .filter((f) => typeof f === "function"),
        Number,
        Number.isFinite,
        Number.isNaN,
        Number.parseFloat,
        Number.parseInt,
        Number.prototype.toExponential,
        Number.prototype.toFixed,
        Number.prototype.toPrecision,
        Number.prototype.toString,
        Object,
        Object.entries,
        Object.is,
        Object.isExtensible,
        Object.isFrozen,
        Object.isSealed,
        Object.keys,
        Object.values,
        parseFloat,
        parseInt,
        RegExp,
        Set,
        Set.prototype.entries,
        Set.prototype.has,
        Set.prototype.keys,
        Set.prototype.values,
        String,
        String.fromCharCode,
        String.fromCodePoint,
        String.raw,
        String.prototype.at,
        String.prototype.charAt,
        String.prototype.charCodeAt,
        String.prototype.codePointAt,
        String.prototype.concat,
        String.prototype.endsWith,
        String.prototype.includes,
        String.prototype.indexOf,
        String.prototype.lastIndexOf,
        String.prototype.normalize,
        String.prototype.padEnd,
        String.prototype.padStart,
        String.prototype.slice,
        String.prototype.startsWith,
        String.prototype.substr,
        String.prototype.substring,
        String.prototype.toLowerCase,
        String.prototype.toString,
        String.prototype.toUpperCase,
        String.prototype.trim,
        String.prototype.trimEnd,
        String.prototype.trimLeft,
        String.prototype.trimRight,
        String.prototype.trimStart,
        Symbol.for,
        Symbol.keyFor,
        unescape,
    ].filter((f) => typeof f === "function"),
);

const typeConversionBinaryOps = Object.freeze(
    new Set([
        "==",
        "!=",
        "<",
        "<=",
        ">",
        ">=",
        "<<",
        ">>",
        ">>>",
        "+",
        "-",
        "*",
        "/",
        "%",
        "|",
        "^",
        "&",
        "in",
    ]),
);
const typeConversionUnaryOps = Object.freeze(new Set(["-", "+", "!", "~"]));

/**
 * Check whether the given value is an ASTNode or not.
 * @param {any} x The value to check.
 * @returns {boolean} `true` if the value is an ASTNode.
 */
function isNode(x) {
    return x !== null && typeof x === "object" && typeof x.type === "string"
}

Object.freeze(
    Object.assign(Object.create(null), {
        $visit(node, options, visitorKeys) {
            const { type } = node;

            if (typeof this[type] === "function") {
                return this[type](node, options, visitorKeys)
            }

            return this.$visitChildren(node, options, visitorKeys)
        },

        $visitChildren(node, options, visitorKeys) {
            const { type } = node;

            for (const key of visitorKeys[type] || eslintVisitorKeys.getKeys(node)) {
                const value = node[key];

                if (Array.isArray(value)) {
                    for (const element of value) {
                        if (
                            isNode(element) &&
                            this.$visit(element, options, visitorKeys)
                        ) {
                            return true
                        }
                    }
                } else if (
                    isNode(value) &&
                    this.$visit(value, options, visitorKeys)
                ) {
                    return true
                }
            }

            return false
        },

        ArrowFunctionExpression() {
            return false
        },
        AssignmentExpression() {
            return true
        },
        AwaitExpression() {
            return true
        },
        BinaryExpression(node, options, visitorKeys) {
            if (
                options.considerImplicitTypeConversion &&
                typeConversionBinaryOps.has(node.operator) &&
                (node.left.type !== "Literal" || node.right.type !== "Literal")
            ) {
                return true
            }
            return this.$visitChildren(node, options, visitorKeys)
        },
        CallExpression() {
            return true
        },
        FunctionExpression() {
            return false
        },
        ImportExpression() {
            return true
        },
        MemberExpression(node, options, visitorKeys) {
            if (options.considerGetters) {
                return true
            }
            if (
                options.considerImplicitTypeConversion &&
                node.computed &&
                node.property.type !== "Literal"
            ) {
                return true
            }
            return this.$visitChildren(node, options, visitorKeys)
        },
        MethodDefinition(node, options, visitorKeys) {
            if (
                options.considerImplicitTypeConversion &&
                node.computed &&
                node.key.type !== "Literal"
            ) {
                return true
            }
            return this.$visitChildren(node, options, visitorKeys)
        },
        NewExpression() {
            return true
        },
        Property(node, options, visitorKeys) {
            if (
                options.considerImplicitTypeConversion &&
                node.computed &&
                node.key.type !== "Literal"
            ) {
                return true
            }
            return this.$visitChildren(node, options, visitorKeys)
        },
        PropertyDefinition(node, options, visitorKeys) {
            if (
                options.considerImplicitTypeConversion &&
                node.computed &&
                node.key.type !== "Literal"
            ) {
                return true
            }
            return this.$visitChildren(node, options, visitorKeys)
        },
        UnaryExpression(node, options, visitorKeys) {
            if (node.operator === "delete") {
                return true
            }
            if (
                options.considerImplicitTypeConversion &&
                typeConversionUnaryOps.has(node.operator) &&
                node.argument.type !== "Literal"
            ) {
                return true
            }
            return this.$visitChildren(node, options, visitorKeys)
        },
        UpdateExpression() {
            return true
        },
        YieldExpression() {
            return true
        },
    }),
);

/**
 * Get the left parenthesis of the parent node syntax if it exists.
 * E.g., `if (a) {}` then the `(`.
 * @param {Node} node The AST node to check.
 * @param {SourceCode} sourceCode The source code object to get tokens.
 * @returns {Token|null} The left parenthesis of the parent node syntax
 */
function getParentSyntaxParen(node, sourceCode) {
    const parent = node.parent;

    switch (parent.type) {
        case "CallExpression":
        case "NewExpression":
            if (parent.arguments.length === 1 && parent.arguments[0] === node) {
                return sourceCode.getTokenAfter(
                    parent.callee,
                    isOpeningParenToken,
                )
            }
            return null

        case "DoWhileStatement":
            if (parent.test === node) {
                return sourceCode.getTokenAfter(
                    parent.body,
                    isOpeningParenToken,
                )
            }
            return null

        case "IfStatement":
        case "WhileStatement":
            if (parent.test === node) {
                return sourceCode.getFirstToken(parent, 1)
            }
            return null

        case "ImportExpression":
            if (parent.source === node) {
                return sourceCode.getFirstToken(parent, 1)
            }
            return null

        case "SwitchStatement":
            if (parent.discriminant === node) {
                return sourceCode.getFirstToken(parent, 1)
            }
            return null

        case "WithStatement":
            if (parent.object === node) {
                return sourceCode.getFirstToken(parent, 1)
            }
            return null

        default:
            return null
    }
}

/**
 * Check whether a given node is parenthesized or not.
 * @param {number} times The number of parantheses.
 * @param {Node} node The AST node to check.
 * @param {SourceCode} sourceCode The source code object to get tokens.
 * @returns {boolean} `true` if the node is parenthesized the given times.
 */
/**
 * Check whether a given node is parenthesized or not.
 * @param {Node} node The AST node to check.
 * @param {SourceCode} sourceCode The source code object to get tokens.
 * @returns {boolean} `true` if the node is parenthesized.
 */
function isParenthesized(
    timesOrNode,
    nodeOrSourceCode,
    optionalSourceCode,
) {
    let times, node, sourceCode, maybeLeftParen, maybeRightParen;
    if (typeof timesOrNode === "number") {
        times = timesOrNode | 0;
        node = nodeOrSourceCode;
        sourceCode = optionalSourceCode;
        if (!(times >= 1)) {
            throw new TypeError("'times' should be a positive integer.")
        }
    } else {
        times = 1;
        node = timesOrNode;
        sourceCode = nodeOrSourceCode;
    }

    if (
        node == null ||
        // `Program` can't be parenthesized
        node.parent == null ||
        // `CatchClause.param` can't be parenthesized, example `try {} catch (error) {}`
        (node.parent.type === "CatchClause" && node.parent.param === node)
    ) {
        return false
    }

    maybeLeftParen = maybeRightParen = node;
    do {
        maybeLeftParen = sourceCode.getTokenBefore(maybeLeftParen);
        maybeRightParen = sourceCode.getTokenAfter(maybeRightParen);
    } while (
        maybeLeftParen != null &&
        maybeRightParen != null &&
        isOpeningParenToken(maybeLeftParen) &&
        isClosingParenToken(maybeRightParen) &&
        // Avoid false positive such as `if (a) {}`
        maybeLeftParen !== getParentSyntaxParen(node, sourceCode) &&
        --times > 0
    )

    return times === 0
}
Function.call.bind(Object.hasOwnProperty);

var __defProp = Object.defineProperty;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __publicField = (obj, key, value) => {
  __defNormalProp(obj, typeof key !== "symbol" ? key + "" : key, value);
  return value;
};
class FixTracker {
  /**
   * Create a new FixTracker.
   * @param fixer A ruleFixer instance.
   * @param sourceCode A SourceCode object for the current code.
   */
  constructor(fixer, sourceCode) {
    this.fixer = fixer;
    this.sourceCode = sourceCode;
    __publicField(this, "retainedRange");
    this.retainedRange = null;
  }
  /**
   * Mark the given range as "retained", meaning that other fixes may not
   * may not modify this region in the same pass.
   * @param range The range to retain.
   * @returns The same RuleFixer, for chained calls.
   */
  retainRange(range) {
    this.retainedRange = range;
    return this;
  }
  /**
   * Given a node, find the function containing it (or the entire program) and
   * mark it as retained, meaning that other fixes may not modify it in this
   * pass. This is useful for avoiding conflicts in fixes that modify control
   * flow.
   * @param node The node to use as a starting point.
   * @returns The same RuleFixer, for chained calls.
   */
  retainEnclosingFunction(node) {
    const functionNode = getUpperFunction(node);
    return this.retainRange(functionNode ? functionNode.range : this.sourceCode.ast.range);
  }
  /**
   * Given a node or token, find the token before and afterward, and mark that
   * range as retained, meaning that other fixes may not modify it in this
   * pass. This is useful for avoiding conflicts in fixes that make a small
   * change to the code where the AST should not be changed.
   * @param nodeOrToken The node or token to use as a starting
   *      point. The token to the left and right are use in the range.
   * @returns The same RuleFixer, for chained calls.
   */
  retainSurroundingTokens(nodeOrToken) {
    const tokenBefore = this.sourceCode.getTokenBefore(nodeOrToken) || nodeOrToken;
    const tokenAfter = this.sourceCode.getTokenAfter(nodeOrToken) || nodeOrToken;
    return this.retainRange([tokenBefore.range[0], tokenAfter.range[1]]);
  }
  /**
   * Create a fix command that replaces the given range with the given text,
   * accounting for any retained ranges.
   * @param range The range to remove in the fix.
   * @param text The text to insert in place of the range.
   * @returns The fix command.
   */
  replaceTextRange(range, text) {
    let actualRange;
    if (this.retainedRange) {
      actualRange = [
        Math.min(this.retainedRange[0], range[0]),
        Math.max(this.retainedRange[1], range[1])
      ];
    } else {
      actualRange = range;
    }
    return this.fixer.replaceTextRange(
      actualRange,
      this.sourceCode.text.slice(actualRange[0], range[0]) + text + this.sourceCode.text.slice(range[1], actualRange[1])
    );
  }
  /**
   * Create a fix command that removes the given node or token, accounting for
   * any retained ranges.
   * @param nodeOrToken The node or token to remove.
   * @returns The fix command.
   */
  remove(nodeOrToken) {
    return this.replaceTextRange(nodeOrToken.range, "");
  }
}

exports.COMMENTS_IGNORE_PATTERN = COMMENTS_IGNORE_PATTERN;
exports.FixTracker = FixTracker;
exports.LINEBREAKS = LINEBREAKS;
exports.LINEBREAK_MATCHER = LINEBREAK_MATCHER;
exports.STATEMENT_LIST_PARENTS = STATEMENT_LIST_PARENTS;
exports.canTokensBeAdjacent = canTokensBeAdjacent;
exports.createGlobalLinebreakMatcher = createGlobalLinebreakMatcher;
exports.createRule = createRule;
exports.getGraphemeCount = getGraphemeCount;
exports.getNextLocation = getNextLocation;
exports.getPrecedence = getPrecedence;
exports.getStaticPropertyName = getStaticPropertyName;
exports.getSwitchCaseColonToken = getSwitchCaseColonToken;
exports.hasOctalOrNonOctalDecimalEscapeSequence = hasOctalOrNonOctalDecimalEscapeSequence;
exports.isArrowToken = isArrowToken$1;
exports.isClosingBraceToken = isClosingBraceToken$1;
exports.isClosingBracketToken = isClosingBracketToken$1;
exports.isClosingParenToken = isClosingParenToken$1;
exports.isColonToken = isColonToken$1;
exports.isCommaToken = isCommaToken$1;
exports.isCommentToken = isCommentToken$1;
exports.isDecimalInteger = isDecimalInteger;
exports.isDecimalIntegerNumericToken = isDecimalIntegerNumericToken;
exports.isDirective = isDirective;
exports.isEqToken = isEqToken;
exports.isFunction = isFunction;
exports.isKeywordToken = isKeywordToken;
exports.isMixedLogicalAndCoalesceExpressions = isMixedLogicalAndCoalesceExpressions;
exports.isNotClosingParenToken = isNotClosingParenToken;
exports.isNotCommaToken = isNotCommaToken;
exports.isNotOpeningParenToken = isNotOpeningParenToken;
exports.isNotQuestionDotToken = isNotQuestionDotToken;
exports.isNotSemicolonToken = isNotSemicolonToken;
exports.isNumericLiteral = isNumericLiteral;
exports.isOpeningBraceToken = isOpeningBraceToken$1;
exports.isOpeningBracketToken = isOpeningBracketToken$1;
exports.isOpeningParenToken = isOpeningParenToken$1;
exports.isParenthesised = isParenthesised;
exports.isParenthesized = isParenthesized;
exports.isQuestionDotToken = isQuestionDotToken;
exports.isSemicolonToken = isSemicolonToken$1;
exports.isStringLiteral = isStringLiteral;
exports.isSurroundedBy = isSurroundedBy;
exports.isTokenOnSameLine = isTokenOnSameLine;
exports.isTopLevelExpressionStatement = isTopLevelExpressionStatement;
exports.keywords = keywords;
exports.skipChainExpression = skipChainExpression;
