/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { hasInSkipHydrationBlockFlag } from '../hydration/skip_hydration';
import { assertDefined } from '../util/assert';
import { assertLContainer, assertLView, assertTNodeForLView } from './assert';
import { renderView } from './instructions/render';
import { createLView } from './instructions/shared';
import { CONTAINER_HEADER_OFFSET, NATIVE } from './interfaces/container';
import { DECLARATION_LCONTAINER, FLAGS, HYDRATION, QUERIES, RENDERER, T_HOST, TVIEW } from './interfaces/view';
import { addViewToDOM, destroyLView, detachView, getBeforeNodeForView, insertView, nativeParentNode } from './node_manipulation';
export function createAndRenderEmbeddedLView(declarationLView, templateTNode, context, options) {
    const embeddedTView = templateTNode.tView;
    ngDevMode && assertDefined(embeddedTView, 'TView must be defined for a template node.');
    ngDevMode && assertTNodeForLView(templateTNode, declarationLView);
    // Embedded views follow the change detection strategy of the view they're declared in.
    const isSignalView = declarationLView[FLAGS] & 4096 /* LViewFlags.SignalView */;
    const viewFlags = isSignalView ? 4096 /* LViewFlags.SignalView */ : 16 /* LViewFlags.CheckAlways */;
    const embeddedLView = createLView(declarationLView, embeddedTView, context, viewFlags, null, templateTNode, null, null, null, options?.injector ?? null, options?.dehydratedView ?? null);
    const declarationLContainer = declarationLView[templateTNode.index];
    ngDevMode && assertLContainer(declarationLContainer);
    embeddedLView[DECLARATION_LCONTAINER] = declarationLContainer;
    const declarationViewLQueries = declarationLView[QUERIES];
    if (declarationViewLQueries !== null) {
        embeddedLView[QUERIES] = declarationViewLQueries.createEmbeddedView(embeddedTView);
    }
    // execute creation mode of a view
    renderView(embeddedTView, embeddedLView, context);
    return embeddedLView;
}
export function getLViewFromLContainer(lContainer, index) {
    const adjustedIndex = CONTAINER_HEADER_OFFSET + index;
    // avoid reading past the array boundaries
    if (adjustedIndex < lContainer.length) {
        const lView = lContainer[adjustedIndex];
        ngDevMode && assertLView(lView);
        return lView;
    }
    return undefined;
}
/**
 * Returns whether an elements that belong to a view should be
 * inserted into the DOM. For client-only cases, DOM elements are
 * always inserted. For hydration cases, we check whether serialized
 * info is available for a view and the view is not in a "skip hydration"
 * block (in which case view contents was re-created, thus needing insertion).
 */
export function shouldAddViewToDom(tNode, dehydratedView) {
    return !dehydratedView || dehydratedView.firstChild === null ||
        hasInSkipHydrationBlockFlag(tNode);
}
export function addLViewToLContainer(lContainer, lView, index, addToDOM = true) {
    const tView = lView[TVIEW];
    // Insert into the view tree so the new view can be change-detected
    insertView(tView, lView, lContainer, index);
    // Insert elements that belong to this view into the DOM tree
    if (addToDOM) {
        const beforeNode = getBeforeNodeForView(index, lContainer);
        const renderer = lView[RENDERER];
        const parentRNode = nativeParentNode(renderer, lContainer[NATIVE]);
        if (parentRNode !== null) {
            addViewToDOM(tView, lContainer[T_HOST], renderer, lView, parentRNode, beforeNode);
        }
    }
    // When in hydration mode, reset the pointer to the first child in
    // the dehydrated view. This indicates that the view was hydrated and
    // further attaching/detaching should work with this view as normal.
    const hydrationInfo = lView[HYDRATION];
    if (hydrationInfo !== null && hydrationInfo.firstChild !== null) {
        hydrationInfo.firstChild = null;
    }
}
export function removeLViewFromLContainer(lContainer, index) {
    const lView = detachView(lContainer, index);
    if (lView !== undefined) {
        destroyLView(lView[TVIEW], lView);
    }
    return lView;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlld19tYW5pcHVsYXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi9wYWNrYWdlcy9jb3JlL3NyYy9yZW5kZXIzL3ZpZXdfbWFuaXB1bGF0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQUlILE9BQU8sRUFBQywyQkFBMkIsRUFBQyxNQUFNLDZCQUE2QixDQUFDO0FBQ3hFLE9BQU8sRUFBQyxhQUFhLEVBQUMsTUFBTSxnQkFBZ0IsQ0FBQztBQUU3QyxPQUFPLEVBQUMsZ0JBQWdCLEVBQUUsV0FBVyxFQUFFLG1CQUFtQixFQUFDLE1BQU0sVUFBVSxDQUFDO0FBQzVFLE9BQU8sRUFBQyxVQUFVLEVBQUMsTUFBTSx1QkFBdUIsQ0FBQztBQUNqRCxPQUFPLEVBQUMsV0FBVyxFQUFDLE1BQU0sdUJBQXVCLENBQUM7QUFDbEQsT0FBTyxFQUFDLHVCQUF1QixFQUFjLE1BQU0sRUFBQyxNQUFNLHdCQUF3QixDQUFDO0FBR25GLE9BQU8sRUFBQyxzQkFBc0IsRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFxQixPQUFPLEVBQUUsUUFBUSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUMsTUFBTSxtQkFBbUIsQ0FBQztBQUNoSSxPQUFPLEVBQUMsWUFBWSxFQUFFLFlBQVksRUFBRSxVQUFVLEVBQUUsb0JBQW9CLEVBQUUsVUFBVSxFQUFFLGdCQUFnQixFQUFDLE1BQU0scUJBQXFCLENBQUM7QUFFL0gsTUFBTSxVQUFVLDRCQUE0QixDQUN4QyxnQkFBZ0MsRUFBRSxhQUFvQixFQUFFLE9BQVUsRUFDbEUsT0FBOEU7SUFDaEYsTUFBTSxhQUFhLEdBQUcsYUFBYSxDQUFDLEtBQU0sQ0FBQztJQUMzQyxTQUFTLElBQUksYUFBYSxDQUFDLGFBQWEsRUFBRSw0Q0FBNEMsQ0FBQyxDQUFDO0lBQ3hGLFNBQVMsSUFBSSxtQkFBbUIsQ0FBQyxhQUFhLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztJQUVsRSx1RkFBdUY7SUFDdkYsTUFBTSxZQUFZLEdBQUcsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLG1DQUF3QixDQUFDO0lBQ3JFLE1BQU0sU0FBUyxHQUFHLFlBQVksQ0FBQyxDQUFDLGtDQUF1QixDQUFDLGdDQUF1QixDQUFDO0lBQ2hGLE1BQU0sYUFBYSxHQUFHLFdBQVcsQ0FDN0IsZ0JBQWdCLEVBQUUsYUFBYSxFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLGFBQWEsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFDMUYsT0FBTyxFQUFFLFFBQVEsSUFBSSxJQUFJLEVBQUUsT0FBTyxFQUFFLGNBQWMsSUFBSSxJQUFJLENBQUMsQ0FBQztJQUVoRSxNQUFNLHFCQUFxQixHQUFHLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNwRSxTQUFTLElBQUksZ0JBQWdCLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUNyRCxhQUFhLENBQUMsc0JBQXNCLENBQUMsR0FBRyxxQkFBcUIsQ0FBQztJQUU5RCxNQUFNLHVCQUF1QixHQUFHLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQzFELElBQUksdUJBQXVCLEtBQUssSUFBSSxFQUFFLENBQUM7UUFDckMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxHQUFHLHVCQUF1QixDQUFDLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFFRCxrQ0FBa0M7SUFDbEMsVUFBVSxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFFbEQsT0FBTyxhQUFhLENBQUM7QUFDdkIsQ0FBQztBQUVELE1BQU0sVUFBVSxzQkFBc0IsQ0FBSSxVQUFzQixFQUFFLEtBQWE7SUFFN0UsTUFBTSxhQUFhLEdBQUcsdUJBQXVCLEdBQUcsS0FBSyxDQUFDO0lBQ3RELDBDQUEwQztJQUMxQyxJQUFJLGFBQWEsR0FBRyxVQUFVLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDdEMsTUFBTSxLQUFLLEdBQUcsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3hDLFNBQVMsSUFBSSxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDaEMsT0FBTyxLQUFpQixDQUFDO0lBQzNCLENBQUM7SUFDRCxPQUFPLFNBQVMsQ0FBQztBQUNuQixDQUFDO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsTUFBTSxVQUFVLGtCQUFrQixDQUM5QixLQUFZLEVBQUUsY0FBNkM7SUFDN0QsT0FBTyxDQUFDLGNBQWMsSUFBSSxjQUFjLENBQUMsVUFBVSxLQUFLLElBQUk7UUFDeEQsMkJBQTJCLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDekMsQ0FBQztBQUVELE1BQU0sVUFBVSxvQkFBb0IsQ0FDaEMsVUFBc0IsRUFBRSxLQUFxQixFQUFFLEtBQWEsRUFBRSxRQUFRLEdBQUcsSUFBSTtJQUMvRSxNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0IsbUVBQW1FO0lBQ25FLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUU1Qyw2REFBNkQ7SUFDN0QsSUFBSSxRQUFRLEVBQUUsQ0FBQztRQUNiLE1BQU0sVUFBVSxHQUFHLG9CQUFvQixDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMzRCxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakMsTUFBTSxXQUFXLEdBQUcsZ0JBQWdCLENBQUMsUUFBUSxFQUFFLFVBQVUsQ0FBQyxNQUFNLENBQXdCLENBQUMsQ0FBQztRQUMxRixJQUFJLFdBQVcsS0FBSyxJQUFJLEVBQUUsQ0FBQztZQUN6QixZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRSxRQUFRLEVBQUUsS0FBSyxFQUFFLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUNwRixDQUFDO0lBQ0gsQ0FBQztJQUVELGtFQUFrRTtJQUNsRSxxRUFBcUU7SUFDckUsb0VBQW9FO0lBQ3BFLE1BQU0sYUFBYSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUN2QyxJQUFJLGFBQWEsS0FBSyxJQUFJLElBQUksYUFBYSxDQUFDLFVBQVUsS0FBSyxJQUFJLEVBQUUsQ0FBQztRQUNoRSxhQUFhLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQztJQUNsQyxDQUFDO0FBQ0gsQ0FBQztBQUVELE1BQU0sVUFBVSx5QkFBeUIsQ0FBQyxVQUFzQixFQUFFLEtBQWE7SUFFN0UsTUFBTSxLQUFLLEdBQUcsVUFBVSxDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM1QyxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUUsQ0FBQztRQUN4QixZQUFZLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFDRCxPQUFPLEtBQUssQ0FBQztBQUNmLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIExMQyBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cblxuaW1wb3J0IHtJbmplY3Rvcn0gZnJvbSAnLi4vZGkvaW5qZWN0b3InO1xuaW1wb3J0IHtEZWh5ZHJhdGVkQ29udGFpbmVyVmlld30gZnJvbSAnLi4vaHlkcmF0aW9uL2ludGVyZmFjZXMnO1xuaW1wb3J0IHtoYXNJblNraXBIeWRyYXRpb25CbG9ja0ZsYWd9IGZyb20gJy4uL2h5ZHJhdGlvbi9za2lwX2h5ZHJhdGlvbic7XG5pbXBvcnQge2Fzc2VydERlZmluZWR9IGZyb20gJy4uL3V0aWwvYXNzZXJ0JztcblxuaW1wb3J0IHthc3NlcnRMQ29udGFpbmVyLCBhc3NlcnRMVmlldywgYXNzZXJ0VE5vZGVGb3JMVmlld30gZnJvbSAnLi9hc3NlcnQnO1xuaW1wb3J0IHtyZW5kZXJWaWV3fSBmcm9tICcuL2luc3RydWN0aW9ucy9yZW5kZXInO1xuaW1wb3J0IHtjcmVhdGVMVmlld30gZnJvbSAnLi9pbnN0cnVjdGlvbnMvc2hhcmVkJztcbmltcG9ydCB7Q09OVEFJTkVSX0hFQURFUl9PRkZTRVQsIExDb250YWluZXIsIE5BVElWRX0gZnJvbSAnLi9pbnRlcmZhY2VzL2NvbnRhaW5lcic7XG5pbXBvcnQge1ROb2RlfSBmcm9tICcuL2ludGVyZmFjZXMvbm9kZSc7XG5pbXBvcnQge1JDb21tZW50LCBSRWxlbWVudH0gZnJvbSAnLi9pbnRlcmZhY2VzL3JlbmRlcmVyX2RvbSc7XG5pbXBvcnQge0RFQ0xBUkFUSU9OX0xDT05UQUlORVIsIEZMQUdTLCBIWURSQVRJT04sIExWaWV3LCBMVmlld0ZsYWdzLCBRVUVSSUVTLCBSRU5ERVJFUiwgVF9IT1NULCBUVklFV30gZnJvbSAnLi9pbnRlcmZhY2VzL3ZpZXcnO1xuaW1wb3J0IHthZGRWaWV3VG9ET00sIGRlc3Ryb3lMVmlldywgZGV0YWNoVmlldywgZ2V0QmVmb3JlTm9kZUZvclZpZXcsIGluc2VydFZpZXcsIG5hdGl2ZVBhcmVudE5vZGV9IGZyb20gJy4vbm9kZV9tYW5pcHVsYXRpb24nO1xuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlQW5kUmVuZGVyRW1iZWRkZWRMVmlldzxUPihcbiAgICBkZWNsYXJhdGlvbkxWaWV3OiBMVmlldzx1bmtub3duPiwgdGVtcGxhdGVUTm9kZTogVE5vZGUsIGNvbnRleHQ6IFQsXG4gICAgb3B0aW9ucz86IHtpbmplY3Rvcj86IEluamVjdG9yLCBkZWh5ZHJhdGVkVmlldz86IERlaHlkcmF0ZWRDb250YWluZXJWaWV3fG51bGx9KTogTFZpZXc8VD4ge1xuICBjb25zdCBlbWJlZGRlZFRWaWV3ID0gdGVtcGxhdGVUTm9kZS50VmlldyE7XG4gIG5nRGV2TW9kZSAmJiBhc3NlcnREZWZpbmVkKGVtYmVkZGVkVFZpZXcsICdUVmlldyBtdXN0IGJlIGRlZmluZWQgZm9yIGEgdGVtcGxhdGUgbm9kZS4nKTtcbiAgbmdEZXZNb2RlICYmIGFzc2VydFROb2RlRm9yTFZpZXcodGVtcGxhdGVUTm9kZSwgZGVjbGFyYXRpb25MVmlldyk7XG5cbiAgLy8gRW1iZWRkZWQgdmlld3MgZm9sbG93IHRoZSBjaGFuZ2UgZGV0ZWN0aW9uIHN0cmF0ZWd5IG9mIHRoZSB2aWV3IHRoZXkncmUgZGVjbGFyZWQgaW4uXG4gIGNvbnN0IGlzU2lnbmFsVmlldyA9IGRlY2xhcmF0aW9uTFZpZXdbRkxBR1NdICYgTFZpZXdGbGFncy5TaWduYWxWaWV3O1xuICBjb25zdCB2aWV3RmxhZ3MgPSBpc1NpZ25hbFZpZXcgPyBMVmlld0ZsYWdzLlNpZ25hbFZpZXcgOiBMVmlld0ZsYWdzLkNoZWNrQWx3YXlzO1xuICBjb25zdCBlbWJlZGRlZExWaWV3ID0gY3JlYXRlTFZpZXc8VD4oXG4gICAgICBkZWNsYXJhdGlvbkxWaWV3LCBlbWJlZGRlZFRWaWV3LCBjb250ZXh0LCB2aWV3RmxhZ3MsIG51bGwsIHRlbXBsYXRlVE5vZGUsIG51bGwsIG51bGwsIG51bGwsXG4gICAgICBvcHRpb25zPy5pbmplY3RvciA/PyBudWxsLCBvcHRpb25zPy5kZWh5ZHJhdGVkVmlldyA/PyBudWxsKTtcblxuICBjb25zdCBkZWNsYXJhdGlvbkxDb250YWluZXIgPSBkZWNsYXJhdGlvbkxWaWV3W3RlbXBsYXRlVE5vZGUuaW5kZXhdO1xuICBuZ0Rldk1vZGUgJiYgYXNzZXJ0TENvbnRhaW5lcihkZWNsYXJhdGlvbkxDb250YWluZXIpO1xuICBlbWJlZGRlZExWaWV3W0RFQ0xBUkFUSU9OX0xDT05UQUlORVJdID0gZGVjbGFyYXRpb25MQ29udGFpbmVyO1xuXG4gIGNvbnN0IGRlY2xhcmF0aW9uVmlld0xRdWVyaWVzID0gZGVjbGFyYXRpb25MVmlld1tRVUVSSUVTXTtcbiAgaWYgKGRlY2xhcmF0aW9uVmlld0xRdWVyaWVzICE9PSBudWxsKSB7XG4gICAgZW1iZWRkZWRMVmlld1tRVUVSSUVTXSA9IGRlY2xhcmF0aW9uVmlld0xRdWVyaWVzLmNyZWF0ZUVtYmVkZGVkVmlldyhlbWJlZGRlZFRWaWV3KTtcbiAgfVxuXG4gIC8vIGV4ZWN1dGUgY3JlYXRpb24gbW9kZSBvZiBhIHZpZXdcbiAgcmVuZGVyVmlldyhlbWJlZGRlZFRWaWV3LCBlbWJlZGRlZExWaWV3LCBjb250ZXh0KTtcblxuICByZXR1cm4gZW1iZWRkZWRMVmlldztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldExWaWV3RnJvbUxDb250YWluZXI8VD4obENvbnRhaW5lcjogTENvbnRhaW5lciwgaW5kZXg6IG51bWJlcik6IExWaWV3PFQ+fFxuICAgIHVuZGVmaW5lZCB7XG4gIGNvbnN0IGFkanVzdGVkSW5kZXggPSBDT05UQUlORVJfSEVBREVSX09GRlNFVCArIGluZGV4O1xuICAvLyBhdm9pZCByZWFkaW5nIHBhc3QgdGhlIGFycmF5IGJvdW5kYXJpZXNcbiAgaWYgKGFkanVzdGVkSW5kZXggPCBsQ29udGFpbmVyLmxlbmd0aCkge1xuICAgIGNvbnN0IGxWaWV3ID0gbENvbnRhaW5lclthZGp1c3RlZEluZGV4XTtcbiAgICBuZ0Rldk1vZGUgJiYgYXNzZXJ0TFZpZXcobFZpZXcpO1xuICAgIHJldHVybiBsVmlldyBhcyBMVmlldzxUPjtcbiAgfVxuICByZXR1cm4gdW5kZWZpbmVkO1xufVxuXG4vKipcbiAqIFJldHVybnMgd2hldGhlciBhbiBlbGVtZW50cyB0aGF0IGJlbG9uZyB0byBhIHZpZXcgc2hvdWxkIGJlXG4gKiBpbnNlcnRlZCBpbnRvIHRoZSBET00uIEZvciBjbGllbnQtb25seSBjYXNlcywgRE9NIGVsZW1lbnRzIGFyZVxuICogYWx3YXlzIGluc2VydGVkLiBGb3IgaHlkcmF0aW9uIGNhc2VzLCB3ZSBjaGVjayB3aGV0aGVyIHNlcmlhbGl6ZWRcbiAqIGluZm8gaXMgYXZhaWxhYmxlIGZvciBhIHZpZXcgYW5kIHRoZSB2aWV3IGlzIG5vdCBpbiBhIFwic2tpcCBoeWRyYXRpb25cIlxuICogYmxvY2sgKGluIHdoaWNoIGNhc2UgdmlldyBjb250ZW50cyB3YXMgcmUtY3JlYXRlZCwgdGh1cyBuZWVkaW5nIGluc2VydGlvbikuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzaG91bGRBZGRWaWV3VG9Eb20oXG4gICAgdE5vZGU6IFROb2RlLCBkZWh5ZHJhdGVkVmlldz86IERlaHlkcmF0ZWRDb250YWluZXJWaWV3fG51bGwpOiBib29sZWFuIHtcbiAgcmV0dXJuICFkZWh5ZHJhdGVkVmlldyB8fCBkZWh5ZHJhdGVkVmlldy5maXJzdENoaWxkID09PSBudWxsIHx8XG4gICAgICBoYXNJblNraXBIeWRyYXRpb25CbG9ja0ZsYWcodE5vZGUpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYWRkTFZpZXdUb0xDb250YWluZXIoXG4gICAgbENvbnRhaW5lcjogTENvbnRhaW5lciwgbFZpZXc6IExWaWV3PHVua25vd24+LCBpbmRleDogbnVtYmVyLCBhZGRUb0RPTSA9IHRydWUpOiB2b2lkIHtcbiAgY29uc3QgdFZpZXcgPSBsVmlld1tUVklFV107XG5cbiAgLy8gSW5zZXJ0IGludG8gdGhlIHZpZXcgdHJlZSBzbyB0aGUgbmV3IHZpZXcgY2FuIGJlIGNoYW5nZS1kZXRlY3RlZFxuICBpbnNlcnRWaWV3KHRWaWV3LCBsVmlldywgbENvbnRhaW5lciwgaW5kZXgpO1xuXG4gIC8vIEluc2VydCBlbGVtZW50cyB0aGF0IGJlbG9uZyB0byB0aGlzIHZpZXcgaW50byB0aGUgRE9NIHRyZWVcbiAgaWYgKGFkZFRvRE9NKSB7XG4gICAgY29uc3QgYmVmb3JlTm9kZSA9IGdldEJlZm9yZU5vZGVGb3JWaWV3KGluZGV4LCBsQ29udGFpbmVyKTtcbiAgICBjb25zdCByZW5kZXJlciA9IGxWaWV3W1JFTkRFUkVSXTtcbiAgICBjb25zdCBwYXJlbnRSTm9kZSA9IG5hdGl2ZVBhcmVudE5vZGUocmVuZGVyZXIsIGxDb250YWluZXJbTkFUSVZFXSBhcyBSRWxlbWVudCB8IFJDb21tZW50KTtcbiAgICBpZiAocGFyZW50Uk5vZGUgIT09IG51bGwpIHtcbiAgICAgIGFkZFZpZXdUb0RPTSh0VmlldywgbENvbnRhaW5lcltUX0hPU1RdLCByZW5kZXJlciwgbFZpZXcsIHBhcmVudFJOb2RlLCBiZWZvcmVOb2RlKTtcbiAgICB9XG4gIH1cblxuICAvLyBXaGVuIGluIGh5ZHJhdGlvbiBtb2RlLCByZXNldCB0aGUgcG9pbnRlciB0byB0aGUgZmlyc3QgY2hpbGQgaW5cbiAgLy8gdGhlIGRlaHlkcmF0ZWQgdmlldy4gVGhpcyBpbmRpY2F0ZXMgdGhhdCB0aGUgdmlldyB3YXMgaHlkcmF0ZWQgYW5kXG4gIC8vIGZ1cnRoZXIgYXR0YWNoaW5nL2RldGFjaGluZyBzaG91bGQgd29yayB3aXRoIHRoaXMgdmlldyBhcyBub3JtYWwuXG4gIGNvbnN0IGh5ZHJhdGlvbkluZm8gPSBsVmlld1tIWURSQVRJT05dO1xuICBpZiAoaHlkcmF0aW9uSW5mbyAhPT0gbnVsbCAmJiBoeWRyYXRpb25JbmZvLmZpcnN0Q2hpbGQgIT09IG51bGwpIHtcbiAgICBoeWRyYXRpb25JbmZvLmZpcnN0Q2hpbGQgPSBudWxsO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiByZW1vdmVMVmlld0Zyb21MQ29udGFpbmVyKGxDb250YWluZXI6IExDb250YWluZXIsIGluZGV4OiBudW1iZXIpOiBMVmlldzx1bmtub3duPnxcbiAgICB1bmRlZmluZWQge1xuICBjb25zdCBsVmlldyA9IGRldGFjaFZpZXcobENvbnRhaW5lciwgaW5kZXgpO1xuICBpZiAobFZpZXcgIT09IHVuZGVmaW5lZCkge1xuICAgIGRlc3Ryb3lMVmlldyhsVmlld1tUVklFV10sIGxWaWV3KTtcbiAgfVxuICByZXR1cm4gbFZpZXc7XG59XG4iXX0=