package org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.action.rev140714.nx.action.multipath.grouping;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.openflowjava.nx.action.rev140421.OfjNxHashFields;
import org.opendaylight.yang.gen.v1.urn.opendaylight.openflowjava.nx.action.rev140421.OfjNxMpAlgorithm;
import org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.action.rev140714.nx.action.multipath.grouping.nx.multipath.Dst;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link NxMultipathBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     NxMultipathBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new NxMultipathBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of NxMultipathBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see NxMultipathBuilder
 * @see Builder
 *
 */
public class NxMultipathBuilder implements Builder<NxMultipath> {

    private OfjNxMpAlgorithm _algorithm;
    private Long _arg;
    private Integer _basis;
    private Dst _dst;
    private OfjNxHashFields _fields;
    private Integer _maxLink;


    Map<Class<? extends Augmentation<NxMultipath>>, Augmentation<NxMultipath>> augmentation = Collections.emptyMap();

    public NxMultipathBuilder() {
    }

    public NxMultipathBuilder(NxMultipath base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<NxMultipath>>, Augmentation<NxMultipath>> aug =((AugmentationHolder<NxMultipath>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._algorithm = base.getAlgorithm();
        this._arg = base.getArg();
        this._basis = base.getBasis();
        this._dst = base.getDst();
        this._fields = base.getFields();
        this._maxLink = base.getMaxLink();
    }


    public OfjNxMpAlgorithm getAlgorithm() {
        return _algorithm;
    }
    
    public Long getArg() {
        return _arg;
    }
    
    public Integer getBasis() {
        return _basis;
    }
    
    public Dst getDst() {
        return _dst;
    }
    
    public OfjNxHashFields getFields() {
        return _fields;
    }
    
    public Integer getMaxLink() {
        return _maxLink;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<NxMultipath>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public NxMultipathBuilder setAlgorithm(final OfjNxMpAlgorithm value) {
        this._algorithm = value;
        return this;
    }
    private static void checkArgRange(final long value) {
        if (value >= 0L && value <= 4294967295L) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..4294967295]]", value);
    }
    
    public NxMultipathBuilder setArg(final Long value) {
    if (value != null) {
        checkArgRange(value);
        
    }
        this._arg = value;
        return this;
    }
    private static void checkBasisRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxMultipathBuilder setBasis(final Integer value) {
    if (value != null) {
        checkBasisRange(value);
        
    }
        this._basis = value;
        return this;
    }
    
    public NxMultipathBuilder setDst(final Dst value) {
        this._dst = value;
        return this;
    }
    
    public NxMultipathBuilder setFields(final OfjNxHashFields value) {
        this._fields = value;
        return this;
    }
    private static void checkMaxLinkRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxMultipathBuilder setMaxLink(final Integer value) {
    if (value != null) {
        checkMaxLinkRange(value);
        
    }
        this._maxLink = value;
        return this;
    }
    
    public NxMultipathBuilder addAugmentation(Class<? extends Augmentation<NxMultipath>> augmentationType, Augmentation<NxMultipath> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public NxMultipathBuilder removeAugmentation(Class<? extends Augmentation<NxMultipath>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public NxMultipath build() {
        return new NxMultipathImpl(this);
    }

    private static final class NxMultipathImpl
        extends AbstractAugmentable<NxMultipath>
        implements NxMultipath {
    
        private final OfjNxMpAlgorithm _algorithm;
        private final Long _arg;
        private final Integer _basis;
        private final Dst _dst;
        private final OfjNxHashFields _fields;
        private final Integer _maxLink;
    
        NxMultipathImpl(NxMultipathBuilder base) {
            super(base.augmentation);
            this._algorithm = base.getAlgorithm();
            this._arg = base.getArg();
            this._basis = base.getBasis();
            this._dst = base.getDst();
            this._fields = base.getFields();
            this._maxLink = base.getMaxLink();
        }
    
        @Override
        public Class<NxMultipath> getImplementedInterface() {
            return NxMultipath.class;
        }
    
        @Override
        public OfjNxMpAlgorithm getAlgorithm() {
            return _algorithm;
        }
        
        @Override
        public Long getArg() {
            return _arg;
        }
        
        @Override
        public Integer getBasis() {
            return _basis;
        }
        
        @Override
        public Dst getDst() {
            return _dst;
        }
        
        @Override
        public OfjNxHashFields getFields() {
            return _fields;
        }
        
        @Override
        public Integer getMaxLink() {
            return _maxLink;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_algorithm);
            result = prime * result + Objects.hashCode(_arg);
            result = prime * result + Objects.hashCode(_basis);
            result = prime * result + Objects.hashCode(_dst);
            result = prime * result + Objects.hashCode(_fields);
            result = prime * result + Objects.hashCode(_maxLink);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NxMultipath.class.equals(((DataObject)obj).getImplementedInterface())) {
                return false;
            }
            NxMultipath other = (NxMultipath)obj;
            if (!Objects.equals(_algorithm, other.getAlgorithm())) {
                return false;
            }
            if (!Objects.equals(_arg, other.getArg())) {
                return false;
            }
            if (!Objects.equals(_basis, other.getBasis())) {
                return false;
            }
            if (!Objects.equals(_dst, other.getDst())) {
                return false;
            }
            if (!Objects.equals(_fields, other.getFields())) {
                return false;
            }
            if (!Objects.equals(_maxLink, other.getMaxLink())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                NxMultipathImpl otherImpl = (NxMultipathImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<NxMultipath>>, Augmentation<NxMultipath>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NxMultipath");
            CodeHelpers.appendValue(helper, "_algorithm", _algorithm);
            CodeHelpers.appendValue(helper, "_arg", _arg);
            CodeHelpers.appendValue(helper, "_basis", _basis);
            CodeHelpers.appendValue(helper, "_dst", _dst);
            CodeHelpers.appendValue(helper, "_fields", _fields);
            CodeHelpers.appendValue(helper, "_maxLink", _maxLink);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
