package org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.action.rev140714.nx.action.conntrack.grouping;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.Short;
import java.lang.String;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.action.rev140714.nx.action.conntrack.grouping.nx.conntrack.CtActions;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link NxConntrackBuilder} instances.
 *
 * @see NxConntrackBuilder
 *
 */
public class NxConntrackBuilder implements Builder<NxConntrack> {

    private Integer _conntrackZone;
    private List<CtActions> _ctActions;
    private Integer _flags;
    private Short _recircTable;
    private Long _zoneSrc;


    Map<Class<? extends Augmentation<NxConntrack>>, Augmentation<NxConntrack>> augmentation = Collections.emptyMap();

    public NxConntrackBuilder() {
    }

    public NxConntrackBuilder(NxConntrack base) {
        this._conntrackZone = base.getConntrackZone();
        this._ctActions = base.getCtActions();
        this._flags = base.getFlags();
        this._recircTable = base.getRecircTable();
        this._zoneSrc = base.getZoneSrc();
        if (base instanceof NxConntrackImpl) {
            NxConntrackImpl impl = (NxConntrackImpl) base;
            if (!impl.augmentation.isEmpty()) {
                this.augmentation = new HashMap<>(impl.augmentation);
            }
        } else if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            AugmentationHolder<NxConntrack> casted =(AugmentationHolder<NxConntrack>) base;
            if (!casted.augmentations().isEmpty()) {
                this.augmentation = new HashMap<>(casted.augmentations());
            }
        }
    }


    public Integer getConntrackZone() {
        return _conntrackZone;
    }
    
    public List<CtActions> getCtActions() {
        return _ctActions;
    }
    
    public Integer getFlags() {
        return _flags;
    }
    
    public Short getRecircTable() {
        return _recircTable;
    }
    
    public Long getZoneSrc() {
        return _zoneSrc;
    }
    
    @SuppressWarnings("unchecked")
    public <E extends Augmentation<NxConntrack>> E augmentation(Class<E> augmentationType) {
        return (E) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static void checkConntrackZoneRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxConntrackBuilder setConntrackZone(final Integer value) {
    if (value != null) {
        checkConntrackZoneRange(value);
        
    }
        this._conntrackZone = value;
        return this;
    }
    public NxConntrackBuilder setCtActions(final List<CtActions> values) {
        this._ctActions = values;
        return this;
    }
    
    private static void checkFlagsRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxConntrackBuilder setFlags(final Integer value) {
    if (value != null) {
        checkFlagsRange(value);
        
    }
        this._flags = value;
        return this;
    }
    private static void checkRecircTableRange(final short value) {
        if (value >= (short)0 && value <= (short)255) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..255]]", value);
    }
    
    public NxConntrackBuilder setRecircTable(final Short value) {
    if (value != null) {
        checkRecircTableRange(value);
        
    }
        this._recircTable = value;
        return this;
    }
    private static void checkZoneSrcRange(final long value) {
        if (value >= 0L && value <= 4294967295L) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..4294967295]]", value);
    }
    
    public NxConntrackBuilder setZoneSrc(final Long value) {
    if (value != null) {
        checkZoneSrcRange(value);
        
    }
        this._zoneSrc = value;
        return this;
    }
    
    public NxConntrackBuilder addAugmentation(Class<? extends Augmentation<NxConntrack>> augmentationType, Augmentation<NxConntrack> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public NxConntrackBuilder removeAugmentation(Class<? extends Augmentation<NxConntrack>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public NxConntrack build() {
        return new NxConntrackImpl(this);
    }

    private static final class NxConntrackImpl implements NxConntrack {
    
        private final Integer _conntrackZone;
        private final List<CtActions> _ctActions;
        private final Integer _flags;
        private final Short _recircTable;
        private final Long _zoneSrc;
    
        private Map<Class<? extends Augmentation<NxConntrack>>, Augmentation<NxConntrack>> augmentation = Collections.emptyMap();
    
        private NxConntrackImpl(NxConntrackBuilder base) {
            this._conntrackZone = base.getConntrackZone();
            this._ctActions = base.getCtActions();
            this._flags = base.getFlags();
            this._recircTable = base.getRecircTable();
            this._zoneSrc = base.getZoneSrc();
            this.augmentation = ImmutableMap.copyOf(base.augmentation);
        }
    
        @Override
        public Class<NxConntrack> getImplementedInterface() {
            return NxConntrack.class;
        }
    
        @Override
        public Integer getConntrackZone() {
            return _conntrackZone;
        }
        
        @Override
        public List<CtActions> getCtActions() {
            return _ctActions;
        }
        
        @Override
        public Integer getFlags() {
            return _flags;
        }
        
        @Override
        public Short getRecircTable() {
            return _recircTable;
        }
        
        @Override
        public Long getZoneSrc() {
            return _zoneSrc;
        }
        
        @SuppressWarnings("unchecked")
        @Override
        public <E extends Augmentation<NxConntrack>> E augmentation(Class<E> augmentationType) {
            return (E) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_conntrackZone);
            result = prime * result + Objects.hashCode(_ctActions);
            result = prime * result + Objects.hashCode(_flags);
            result = prime * result + Objects.hashCode(_recircTable);
            result = prime * result + Objects.hashCode(_zoneSrc);
            result = prime * result + Objects.hashCode(augmentation);
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NxConntrack.class.equals(((DataObject)obj).getImplementedInterface())) {
                return false;
            }
            NxConntrack other = (NxConntrack)obj;
            if (!Objects.equals(_conntrackZone, other.getConntrackZone())) {
                return false;
            }
            if (!Objects.equals(_ctActions, other.getCtActions())) {
                return false;
            }
            if (!Objects.equals(_flags, other.getFlags())) {
                return false;
            }
            if (!Objects.equals(_recircTable, other.getRecircTable())) {
                return false;
            }
            if (!Objects.equals(_zoneSrc, other.getZoneSrc())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                NxConntrackImpl otherImpl = (NxConntrackImpl) obj;
                if (!Objects.equals(augmentation, otherImpl.augmentation)) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<NxConntrack>>, Augmentation<NxConntrack>> e : augmentation.entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NxConntrack");
            CodeHelpers.appendValue(helper, "_conntrackZone", _conntrackZone);
            CodeHelpers.appendValue(helper, "_ctActions", _ctActions);
            CodeHelpers.appendValue(helper, "_flags", _flags);
            CodeHelpers.appendValue(helper, "_recircTable", _recircTable);
            CodeHelpers.appendValue(helper, "_zoneSrc", _zoneSrc);
            CodeHelpers.appendValue(helper, "augmentation", augmentation.values());
            return helper.toString();
        }
    }
}
