/*
 * This library is part of OpenCms -
 * the Open Source Content Management System
 *
 * Copyright (c) Alkacon Software GmbH (http://www.alkacon.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * For further information about Alkacon Software GmbH, please see the
 * company website: http://www.alkacon.com
 *
 * For further information about OpenCms, please see the
 * project website: http://www.opencms.org
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.opencms.workplace.commons;

import org.opencms.file.CmsFile;
import org.opencms.file.CmsResource;
import org.opencms.i18n.CmsEncoder;
import org.opencms.i18n.CmsLocaleManager;
import org.opencms.jsp.CmsJspActionElement;
import org.opencms.security.CmsPermissionSet;
import org.opencms.util.CmsStringUtil;
import org.opencms.workplace.CmsDialog;
import org.opencms.workplace.CmsWorkplaceSettings;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.PageContext;

/**
 * The edit pointer dialog changes the link target of a pointer resource.<p>
 * 
 * The following files use this class:
 * <ul>
 * <li>/commons/editpointer.jsp
 * </ul>
 * <p>
 * 
 * @since 6.0.0 
 */
public class CmsEditPointer extends CmsDialog {

    /** The dialog type.<p> */
    public static final String DIALOG_TYPE = "newlink";

    /** Request parameter name for the link target.<p> */
    public static final String PARAM_LINKTARGET = "linktarget";

    /** Stores the value of the link target.<p> */
    private String m_paramLinkTarget;

    /**
     * Public constructor with JSP action element.<p>
     * 
     * @param jsp an initialized JSP action element
     */
    public CmsEditPointer(CmsJspActionElement jsp) {

        super(jsp);
    }

    /**
     * Public constructor with JSP variables.<p>
     * 
     * @param context the JSP page context
     * @param req the JSP request
     * @param res the JSP response
     */
    public CmsEditPointer(PageContext context, HttpServletRequest req, HttpServletResponse res) {

        this(new CmsJspActionElement(context, req, res));
    }

    /**
     * Changes the link target of the pointer.<p>
     * 
     * @throws JspException if inclusion of error dialog fails
     */
    public void actionChangeLinkTarget() throws JspException {

        try {
            // check the resource lock state
            checkLock(getParamResource());
            // change the link target
            CmsFile editFile = getCms().readFile(getParamResource());
            String encoding = CmsLocaleManager.getResourceEncoding(getCms(), editFile);
            editFile.setContents(getParamLinkTarget().getBytes(encoding));
            getCms().writeFile(editFile);
            // close the dialog window
            actionCloseDialog();
        } catch (Throwable e) {
            // error changing link target, show error dialog
            setParamMessage(Messages.get().getBundle(getLocale()).key(Messages.ERR_CHANGE_LINK_TARGET_0));
            includeErrorpage(this, e);
        }
    }

    /**
     * Returns the old link target value of the pointer resource to edit.<p>
     * 
     * @return the old link target value
     * @throws JspException if problems including sub-elements occur 
     * 
     */
    public String getOldTargetValue() throws JspException {

        String linkTarget = "";
        if (CmsStringUtil.isEmpty(getParamLinkTarget())) {
            // this is the initial dialog call, get link target value
            try {
                // get pointer contents
                CmsFile file = getCms().readFile(getParamResource());
                linkTarget = new String(file.getContents(), CmsLocaleManager.getResourceEncoding(getCms(), file));
            } catch (Throwable e1) {
                // error reading file, show error dialog
                setParamMessage(Messages.get().getBundle(getLocale()).key(
                    Messages.ERR_GET_LINK_TARGET_1,
                    getParamResource()));
                includeErrorpage(this, e1);

            }
        }
        return CmsEncoder.escapeXml(linkTarget);
    }

    /**
     * Returns the link target request parameter value.<p>
     * 
     * @return the link target request parameter value
     */
    public String getParamLinkTarget() {

        return m_paramLinkTarget;
    }

    /**
     * Sets the link target request parameter value.<p>
     * 
     * @param linkTarget the link target request parameter value
     */
    public void setParamLinkTarget(String linkTarget) {

        m_paramLinkTarget = linkTarget;
    }

    /**
     * @see org.opencms.workplace.CmsWorkplace#decodeParamValue(java.lang.String, java.lang.String)
     */
    @Override
    protected String decodeParamValue(String paramName, String paramValue) {

        // We DON'T want to decode the URL
        if (PARAM_LINKTARGET.equals(paramName)) {
            return paramValue;
        } else {
            return super.decodeParamValue(paramName, paramValue);
        }
    }

    /**
     * @see org.opencms.workplace.CmsWorkplace#initWorkplaceRequestValues(org.opencms.workplace.CmsWorkplaceSettings, javax.servlet.http.HttpServletRequest)
     */
    @Override
    protected void initWorkplaceRequestValues(CmsWorkplaceSettings settings, HttpServletRequest request) {

        // fill the parameter values in the get/set methods
        fillParamValues(request);

        // check the required permissions to edit the pointer    
        if (!checkResourcePermissions(CmsPermissionSet.ACCESS_WRITE, false)) {
            // no write permissions for the resource, set cancel action to close dialog
            setParamAction(DIALOG_CANCEL);
        }

        // set the dialog type
        setParamDialogtype(DIALOG_TYPE);
        // set the action for the JSP switch 
        if (DIALOG_OK.equals(getParamAction())) {
            // ok button pressed, change link target
            setAction(ACTION_OK);
        } else if (DIALOG_CANCEL.equals(getParamAction())) {
            // cancel button pressed
            setAction(ACTION_CANCEL);
        } else {
            // first call of dialog
            setAction(ACTION_DEFAULT);
            // build title for change link target dialog     
            setParamTitle(key(Messages.GUI_CHLINK_1, new Object[] {CmsResource.getName(getParamResource())}));
        }
    }

}
