/**
 * Licensed to The Apereo Foundation under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 *
 * The Apereo Foundation licenses this file to you under the Educational
 * Community License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License
 * at:
 *
 *   http://opensource.org/licenses/ecl2.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package org.opencastproject.workflow.handler.workflow;

import static java.lang.String.format;

import org.opencastproject.job.api.JobContext;
import org.opencastproject.mediapackage.MediaPackage;
import org.opencastproject.mediapackage.MediaPackageElement;
import org.opencastproject.mediapackage.MediaPackageElementFlavor;
import org.opencastproject.mediapackage.selector.AbstractMediaPackageElementSelector;
import org.opencastproject.mediapackage.selector.SimpleElementSelector;
import org.opencastproject.serviceregistry.api.ServiceRegistry;
import org.opencastproject.util.FileSupport;
import org.opencastproject.util.NotFoundException;
import org.opencastproject.util.UrlSupport;
import org.opencastproject.util.data.Option;
import org.opencastproject.workflow.api.AbstractWorkflowOperationHandler;
import org.opencastproject.workflow.api.ConfiguredTagsAndFlavors;
import org.opencastproject.workflow.api.WorkflowInstance;
import org.opencastproject.workflow.api.WorkflowOperationException;
import org.opencastproject.workflow.api.WorkflowOperationHandler;
import org.opencastproject.workflow.api.WorkflowOperationInstance;
import org.opencastproject.workflow.api.WorkflowOperationResult;
import org.opencastproject.workflow.api.WorkflowOperationResult.Action;
import org.opencastproject.workspace.api.Workspace;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.List;

/**
 * Workflow operation handler for copying video data through NFS
 */
@Component(
    immediate = true,
    service = WorkflowOperationHandler.class,
    property = {
        "service.description=Copy Workflow Operation Handler",
        "workflow.operation=copy"
    }
)
public class CopyWorkflowOperationHandler extends AbstractWorkflowOperationHandler {
  /** Configuration key for the \"tag\" of the track to use as a source input */
  public static final String OPT_SOURCE_TAGS = "source-tags";

  /** Configuration key for the \"flavor\" of the track to use as a source input */
  public static final String OPT_SOURCE_FLAVORS = "source-flavors";

  /** Configuration key for the directory where the file must be delivered */
  public static final String OPT_TARGET_DIRECTORY = "target-directory";

  /** Configuration key for the name of the target file */
  public static final String OPT_TARGET_FILENAME = "target-filename";

  /** The logging facility */
  private static final Logger logger = LoggerFactory.getLogger(CopyWorkflowOperationHandler.class);

  /** The workspace reference */
  protected Workspace workspace = null;

  /**
   * Callback for the OSGi environment to set the workspace reference.
   *
   * @param workspace
   *          the workspace
   */
  @Reference
  protected void setWorkspace(Workspace workspace) {
    this.workspace = workspace;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public WorkflowOperationResult start(WorkflowInstance workflowInstance, JobContext context)
          throws WorkflowOperationException {
    logger.debug("Running copy workflow operation on workflow {}", workflowInstance.getId());

    MediaPackage mediaPackage = workflowInstance.getMediaPackage();
    WorkflowOperationInstance currentOperation = workflowInstance.getCurrentOperation();

    // Check which tags have been configured
    ConfiguredTagsAndFlavors tagsAndFlavors = getTagsAndFlavors(workflowInstance,
        Configuration.many, Configuration.many, Configuration.none, Configuration.none);
    List<String> sourceTagsOption = tagsAndFlavors.getSrcTags();
    List<MediaPackageElementFlavor> sourceFlavorsOption = tagsAndFlavors.getSrcFlavors();
    String targetDirectoryOption = StringUtils.trimToNull(currentOperation.getConfiguration(OPT_TARGET_DIRECTORY));
    Option<String> targetFilenameOption = Option.option(StringUtils.trimToNull(currentOperation
            .getConfiguration(OPT_TARGET_FILENAME)));

    StringBuilder sb = new StringBuilder();
    sb.append("Parameters passed to copy workflow operation:");
    sb.append("\n source-tags: ").append(sourceTagsOption);
    sb.append("\n source-flavors: ").append(sourceFlavorsOption);
    sb.append("\n target-directory: ").append(targetDirectoryOption);
    sb.append("\n target-filename: ").append(targetFilenameOption);
    logger.debug(sb.toString());

    AbstractMediaPackageElementSelector<MediaPackageElement> elementSelector = new SimpleElementSelector();

    // Make sure either one of tags or flavors are provided
    if (sourceTagsOption.isEmpty() && sourceFlavorsOption.isEmpty()) {
      logger.info("No source tags or flavors have been specified, not matching anything");
      return createResult(mediaPackage, Action.CONTINUE);
    }

    // Make the target filename and directory are provided
    if (StringUtils.isBlank(targetDirectoryOption))
      throw new WorkflowOperationException("No target directory has been set for the copy operation!");

    // Select the source flavors
    for (MediaPackageElementFlavor flavor : sourceFlavorsOption) {
      try {
        elementSelector.addFlavor(flavor);
      } catch (IllegalArgumentException e) {
        throw new WorkflowOperationException("Source flavor '" + flavor + "' is malformed");
      }
    }

    // Select the source tags
    for (String tag : sourceTagsOption) {
      elementSelector.addTag(tag);
    }

    // Look for elements matching the tag
    Collection<MediaPackageElement> elements = elementSelector.select(mediaPackage, true);

    // Check the the number of element returned
    if (elements.size() == 0) {
      // If no one found, we skip the operation
      return createResult(workflowInstance.getMediaPackage(), Action.SKIP);
    } else if (elements.size() == 1) {
      for (MediaPackageElement element : elements) {
        logger.debug("Copy single element to: {}", targetDirectoryOption);
        final String fileName;
        if (targetFilenameOption.isSome()) {
          fileName = targetFilenameOption.get();
        } else {
          fileName = FilenameUtils.getBaseName(element.getURI().toString());
        }

        String ext = FilenameUtils.getExtension(element.getURI().toString());
        ext = ext.length() > 0 ? ".".concat(ext) : "";
        File targetFile = new File(UrlSupport.concat(targetDirectoryOption, fileName.concat(ext)));

        copyElement(element, targetFile);
      }
    } else {
      logger.debug("Copy multiple elements to: {}", targetDirectoryOption);
      int i = 1;
      for (MediaPackageElement element : elements) {
        final String fileName;
        if (targetFilenameOption.isSome()) {
          fileName = String.format(targetFilenameOption.get(), i);
        } else {
          fileName = FilenameUtils.getBaseName(element.getURI().toString());
        }

        String ext = FilenameUtils.getExtension(element.getURI().toString());
        ext = ext.length() > 0 ? ".".concat(ext) : "";
        File targetFile = new File(UrlSupport.concat(targetDirectoryOption, fileName + ext));

        copyElement(element, targetFile);
        i++;
      }
    }

    return createResult(workflowInstance.getMediaPackage(), Action.CONTINUE);
  }

  private void copyElement(MediaPackageElement element, File targetFile) throws WorkflowOperationException {
    File sourceFile;
    try {
      sourceFile = workspace.get(element.getURI());
    } catch (NotFoundException e) {
      throw new WorkflowOperationException("Unable to find " + element.getURI() + " in the workspace", e);
    } catch (IOException e) {
      throw new WorkflowOperationException("Error loading " + element.getURI() + " from the workspace", e);
    }

    logger.debug("Start copying element {} to target {}.", sourceFile.getPath(), targetFile.getPath());
    try {
      FileSupport.copy(sourceFile, targetFile);
    } catch (IOException e) {
      throw new WorkflowOperationException(format("Unable to copy %s to %s: %s", sourceFile.getPath(),
              targetFile.getPath(), e.getMessage()));
    }
    logger.debug("Element {} copied to target {}.", sourceFile.getPath(), targetFile.getPath());
  }

  @Reference
  @Override
  public void setServiceRegistry(ServiceRegistry serviceRegistry) {
    super.setServiceRegistry(serviceRegistry);
  }

}
