/*
 * Decompiled with CFR 0.152.
 */
package org.opencastproject.composer.impl;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.InputStreamReader;
import java.lang.invoke.CallSite;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.activation.MimetypesFileTypeMap;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.codehaus.plexus.util.cli.CommandLineUtils;
import org.opencastproject.composer.api.EncoderException;
import org.opencastproject.composer.api.EncodingProfile;
import org.opencastproject.composer.api.VideoClip;
import org.opencastproject.mediapackage.AdaptivePlaylist;
import org.opencastproject.mediapackage.identifier.IdImpl;
import org.opencastproject.util.IoSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EncoderEngine
implements AutoCloseable {
    static final String CMD_SUFFIX = "ffmpeg.command";
    static final String ADAPTIVE_TYPE_SUFFIX = "adaptive.type";
    static final String PROP_TRIMMING_START_TIME = "trim.start";
    static final String PROP_TRIMMING_DURATION = "trim.duration";
    private static final boolean REDIRECT_ERROR_STREAM = true;
    private static Logger logger = LoggerFactory.getLogger(EncoderEngine.class);
    private String binary = "ffmpeg";
    private Set<Process> processes = new HashSet<Process>();
    private final Pattern outputPattern = Pattern.compile("Output .* (\\S+) to '(.*)':");
    private final Pattern outputPatternHLS = Pattern.compile("Opening '([^']+)\\.tmp'|([^']+)' for writing");
    private static List<String> mappableOptions = Stream.of("-bf", "-b_strategy", "-bitrate", "-bufsize", "-crf", "-f", "-flags", "-force_key_frames", "-g", "-level", "-keyint", "-keyint_min", "-maxrate", "-minrate", "-pix_fmt", "-preset", "-profile", "-r", "-refs", "-s", "-sc_threshold", "-tune", "-x264opts", "-x264-params").collect(Collectors.toList());

    EncoderEngine(String binary) {
        this.binary = binary;
    }

    File encode(File mediaSource, EncodingProfile format, Map<String, String> properties) throws EncoderException {
        List<File> output = this.process(Collections.singletonMap("video", mediaSource), format, properties);
        if (output.size() != 1) {
            throw new EncoderException(String.format("Encode expects one output file (%s found)", output.size()));
        }
        return output.get(0);
    }

    List<File> extract(File mediaSource, EncodingProfile format, Map<String, String> properties, double ... times) throws EncoderException {
        LinkedList<File> extractedImages = new LinkedList<File>();
        try {
            if (times.length == 0) {
                extractedImages.add(this.encode(mediaSource, format, properties));
            }
            for (double time : times) {
                HashMap<String, String> params = new HashMap<String, String>();
                if (properties != null) {
                    params.putAll(properties);
                }
                DecimalFormatSymbols ffmpegFormat = new DecimalFormatSymbols();
                ffmpegFormat.setDecimalSeparator('.');
                DecimalFormat df = new DecimalFormat("0.00000", ffmpegFormat);
                params.put("time", df.format(time));
                extractedImages.add(this.encode(mediaSource, format, params));
            }
        }
        catch (Exception e) {
            this.cleanup(extractedImages);
            if (e instanceof EncoderException) {
                throw (EncoderException)((Object)e);
            }
            throw new EncoderException("Image extraction failed", (Throwable)e);
        }
        return extractedImages;
    }

    List<File> process(Map<String, File> source, EncodingProfile profile, Map<String, String> properties) throws EncoderException {
        ArrayList<File> arrayList;
        HashMap<String, String> params = new HashMap<String, String>();
        if (properties != null) {
            params.putAll(properties);
        }
        if (source.isEmpty()) {
            throw new IllegalArgumentException("At least one track must be specified.");
        }
        for (Map.Entry<String, File> f : source.entrySet()) {
            String input = FilenameUtils.normalize((String)f.getValue().getAbsolutePath());
            String pre = "in." + f.getKey();
            params.put(pre + ".path", input);
            params.put(pre + ".name", FilenameUtils.getBaseName((String)input));
            params.put(pre + ".suffix", FilenameUtils.getExtension((String)input));
            params.put(pre + ".filename", FilenameUtils.getName((String)input));
            params.put(pre + ".mimetype", MimetypesFileTypeMap.getDefaultFileTypeMap().getContentType(input));
        }
        File parentFile = source.getOrDefault("video", source.getOrDefault("audio", source.values().stream().findFirst().get()));
        String outDir = parentFile.getAbsoluteFile().getParent();
        String outFileName = FilenameUtils.getBaseName((String)parentFile.getName()) + "_" + UUID.randomUUID().toString();
        params.put("out.dir", outDir);
        params.put("out.name", outFileName);
        if (profile.getSuffix() != null) {
            String outSuffix = this.processParameters(profile.getSuffix(), params);
            params.put("out.suffix", outSuffix);
        }
        for (String tag : profile.getTags()) {
            String suffix = this.processParameters(profile.getSuffix(tag), params);
            params.put("out.suffix." + tag, suffix);
        }
        List<String> command = this.buildCommand(profile, params);
        logger.info("Executing encoding command: {}", command);
        ArrayList<File> outFiles = new ArrayList<File>();
        BufferedReader in = null;
        Process encoderProcess = null;
        try {
            String line;
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            encoderProcess = processBuilder.start();
            this.processes.add(encoderProcess);
            in = new BufferedReader(new InputStreamReader(encoderProcess.getInputStream()));
            while ((line = in.readLine()) != null) {
                this.handleEncoderOutput(outFiles, line);
            }
            int exitCode = encoderProcess.waitFor();
            if (exitCode != 0) {
                throw new EncoderException("Encoder exited abnormally with status " + exitCode);
            }
            logger.info("Tracks {} successfully encoded using profile '{}'", source, (Object)profile.getIdentifier());
            arrayList = outFiles;
        }
        catch (Exception e) {
            try {
                logger.warn("Error while encoding {}  using profile '{}'", new Object[]{source, profile.getIdentifier(), e});
                for (File outFile : outFiles) {
                    if (!FileUtils.deleteQuietly((File)outFile)) continue;
                    logger.debug("Removed output file of failed encoding process: {}", (Object)outFile);
                }
                throw new EncoderException((Throwable)e);
            }
            catch (Throwable throwable) {
                IoSupport.closeQuietly(in);
                IoSupport.closeQuietly(encoderProcess);
                throw throwable;
            }
        }
        IoSupport.closeQuietly((Closeable)in);
        IoSupport.closeQuietly((Process)encoderProcess);
        return arrayList;
    }

    protected List<File> process(List<String> commandopts) throws EncoderException {
        ArrayList<File> arrayList;
        logger.trace("Process raw command -  {}", commandopts);
        Process encoderProcess = null;
        BufferedReader in = null;
        ArrayList<File> outFiles = new ArrayList<File>();
        try {
            String line;
            ArrayList<String> command = new ArrayList<String>();
            command.add(this.binary);
            command.addAll(commandopts);
            logger.info("Executing encoding command: {}", (Object)StringUtils.join(command, (String)" "));
            ProcessBuilder pbuilder = new ProcessBuilder(command);
            pbuilder.redirectErrorStream(true);
            encoderProcess = pbuilder.start();
            in = new BufferedReader(new InputStreamReader(encoderProcess.getInputStream()));
            while ((line = in.readLine()) != null) {
                this.handleEncoderOutput(outFiles, line);
            }
            encoderProcess.waitFor();
            int exitCode = encoderProcess.exitValue();
            if (exitCode != 0) {
                throw new EncoderException("Encoder exited abnormally with status " + exitCode);
            }
            logger.info("Video track successfully encoded '{}'", new Object[]{StringUtils.join(commandopts, (String)" ")});
            arrayList = outFiles;
        }
        catch (Exception e) {
            try {
                logger.warn("Error while encoding video tracks using '{}': {}", new Object[]{StringUtils.join(commandopts, (String)" "), e.getMessage()});
                for (File outFile : outFiles) {
                    if (!FileUtils.deleteQuietly((File)outFile)) continue;
                    logger.debug("Removed output file of failed encoding process: {}", (Object)outFile);
                }
                throw new EncoderException((Throwable)e);
            }
            catch (Throwable throwable) {
                IoSupport.closeQuietly(in);
                IoSupport.closeQuietly(encoderProcess);
                throw throwable;
            }
        }
        IoSupport.closeQuietly((Closeable)in);
        IoSupport.closeQuietly((Process)encoderProcess);
        return arrayList;
    }

    private void cleanup(List<File> outputFiles) {
        for (File file : outputFiles) {
            if (file == null || !file.isFile()) continue;
            String path = file.getAbsolutePath();
            if (file.delete()) {
                logger.info("Deleted file {}", (Object)path);
                continue;
            }
            logger.warn("Could not delete file {}", (Object)path);
        }
    }

    private List<String> buildCommand(EncodingProfile profile, Map<String, String> argumentReplacements) throws EncoderException {
        ArrayList<String> command = new ArrayList<String>();
        command.add(this.binary);
        command.add("-nostdin");
        command.add("-nostats");
        String commandline = profile.getExtension(CMD_SUFFIX);
        for (String key : argumentReplacements.keySet()) {
            if (!key.startsWith("ffmpeg.command.")) continue;
            String shortKey = key.substring(CMD_SUFFIX.length() + 1);
            commandline = commandline.replace("#{" + shortKey + "}", argumentReplacements.get(key));
        }
        String processedCommandLine = this.processParameters(commandline, argumentReplacements);
        try {
            command.addAll(Arrays.asList(CommandLineUtils.translateCommandline(processedCommandLine)));
        }
        catch (Exception e) {
            throw new EncoderException("Could not process encoding profile command line", (Throwable)e);
        }
        return command;
    }

    File trim(File mediaSource, EncodingProfile format, long start, long duration, Map<String, String> properties) throws EncoderException {
        if (properties == null) {
            properties = new HashMap<String, String>();
        }
        double startD = (double)start / 1000.0;
        double durationD = (double)duration / 1000.0;
        DecimalFormatSymbols ffmpegFormat = new DecimalFormatSymbols();
        ffmpegFormat.setDecimalSeparator('.');
        DecimalFormat df = new DecimalFormat("00.00000", ffmpegFormat);
        properties.put(PROP_TRIMMING_START_TIME, df.format(startD));
        properties.put(PROP_TRIMMING_DURATION, df.format(durationD));
        return this.encode(mediaSource, format, properties);
    }

    private String processParameters(String cmd, Map<String, String> args) {
        String cmdBefore = null;
        while (!cmd.equals(cmdBefore)) {
            cmdBefore = cmd;
            for (Map.Entry<String, String> e : args.entrySet()) {
                cmd = cmd.replace("#{" + e.getKey() + "}", e.getValue());
            }
        }
        cmd = cmd.replace("#{space}", " ");
        return cmd.replaceAll("#\\{.*?\\}", "");
    }

    @Override
    public void close() {
        for (Process process : this.processes) {
            if (!process.isAlive()) continue;
            logger.debug("Destroying encoding process {}", (Object)process);
            process.destroy();
        }
    }

    private void handleEncoderOutput(List<File> output, String message) {
        if ("".equals(message = message.trim())) {
            return;
        }
        if (StringUtils.startsWithAny((CharSequence)message.toLowerCase(), (CharSequence[])new CharSequence[]{"ffmpeg version", "configuration", "lib", "size=", "frame=", "built with"})) {
            logger.trace(message);
        } else if (StringUtils.startsWith((CharSequence)message, (CharSequence)"Output #")) {
            logger.debug(message);
            Matcher matcher = this.outputPattern.matcher(message);
            if (matcher.find()) {
                String type = matcher.group(1);
                String outputPath = matcher.group(2);
                if (!(StringUtils.equals((CharSequence)"NUL", (CharSequence)outputPath) || StringUtils.equals((CharSequence)"/dev/null", (CharSequence)outputPath) || StringUtils.equals((CharSequence)"/dev/null", (CharSequence)outputPath) || StringUtils.startsWith((CharSequence)"pipe:", (CharSequence)outputPath))) {
                    File outputFile = new File(outputPath);
                    if (!type.startsWith("hls")) {
                        logger.info("Identified output file {}", (Object)outputFile);
                        output.add(outputFile);
                    }
                }
            }
        } else if (StringUtils.startsWith((CharSequence)message, (CharSequence)"[hls @ ")) {
            File outputFile;
            String outputPath;
            logger.debug(message);
            Matcher matcher = this.outputPatternHLS.matcher(message);
            if (!(!matcher.find() || StringUtils.equals((CharSequence)"NUL", (CharSequence)(outputPath = Objects.toString(matcher.group(1), matcher.group(2)))) || StringUtils.equals((CharSequence)"/dev/null", (CharSequence)outputPath) || StringUtils.startsWith((CharSequence)"pipe:", (CharSequence)outputPath) || output.contains(outputFile = new File(outputPath)))) {
                logger.info("Identified HLS output file {}", (Object)outputFile);
                output.add(outputFile);
            }
        } else if (StringUtils.startsWithAny((CharSequence)message.toLowerCase(), (CharSequence[])new CharSequence[]{"artist", "compatible_brands", "copyright", "creation_time", "description", "composer", "date", "duration", "encoder", "handler_name", "input #", "last message repeated", "major_brand", "metadata", "minor_version", "output #", "program", "side data:", "stream #", "stream mapping", "title", "video:", "[libx264 @ ", "Press ["})) {
            logger.debug(message);
        } else {
            logger.info(message);
        }
    }

    public List<String> commandSplit(String str) {
        ArrayList<String> al = new ArrayList<String>();
        Pattern regex = Pattern.compile("\"([^\"]*)\"|'([^']*)'|\\S+");
        Matcher m = regex.matcher(str);
        while (m.find()) {
            if (m.group(1) != null) {
                al.add(m.group(1));
                continue;
            }
            if (m.group(2) != null) {
                al.add(m.group(2));
                continue;
            }
            al.add(m.group());
        }
        return al;
    }

    public String joinNonNullString(String[] srlist, String separator) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < srlist.length; ++i) {
            if (srlist[i] == null || srlist[i].isEmpty()) continue;
            if (sb.length() > 0) {
                sb.append(separator);
            }
            sb.append(srlist[i]);
        }
        return sb.toString();
    }

    private static List<VideoClip> sortSegments(List<VideoClip> edits, double gap) {
        VideoClip clip2;
        LinkedList<VideoClip> ll = new LinkedList<VideoClip>();
        Iterator<VideoClip> it = edits.iterator();
        int lastSrc = -1;
        while (it.hasNext()) {
            clip2 = it.next();
            if (lastSrc < 0) {
                lastSrc = clip2.getSrc();
                continue;
            }
            if (lastSrc == clip2.getSrc()) continue;
            return edits;
        }
        Collections.sort(edits);
        ArrayList<VideoClip> clips = new ArrayList<VideoClip>();
        for (VideoClip clip2 : edits) {
            if (!(clip2.getDuration() > gap)) continue;
            ll.add(clip2);
        }
        clip2 = (VideoClip)ll.pop();
        while (!ll.isEmpty()) {
            if (ll.peek() == null) continue;
            VideoClip nextclip = (VideoClip)ll.pop();
            if (nextclip.getSrc() == clip2.getSrc() && nextclip.getStart() - clip2.getEnd() < gap) {
                clip2.setEnd(nextclip.getEnd());
                continue;
            }
            clips.add(clip2);
            clip2 = nextclip;
        }
        clips.add(clip2);
        return clips;
    }

    private List<String> makeEdits(List<VideoClip> clips, int transitionDuration, Boolean hasVideo, Boolean hasAudio) throws Exception {
        double vfade;
        double afade = vfade = (double)(transitionDuration / 1000);
        DecimalFormatSymbols ffmpegFormat = new DecimalFormatSymbols();
        ffmpegFormat.setDecimalSeparator('.');
        DecimalFormat f = new DecimalFormat("0.00", ffmpegFormat);
        ArrayList<CallSite> vpads = new ArrayList<CallSite>();
        ArrayList<CallSite> apads = new ArrayList<CallSite>();
        ArrayList<String> clauses = new ArrayList<String>();
        int n = 0;
        if (clips != null) {
            n = clips.size();
        }
        String outmap = "o";
        if (n > 1) {
            int i;
            for (i = 0; i < n; ++i) {
                vpads.add((CallSite)((Object)("[v" + i + "]")));
                apads.add((CallSite)((Object)("[a" + i + "]")));
            }
            outmap = "";
            for (i = 0; i < n; ++i) {
                VideoClip vclip = clips.get(i);
                int fileindx = vclip.getSrc();
                double inpt = vclip.getStart();
                double duration = vclip.getDuration();
                double vend = Math.max(duration - vfade, 0.0);
                double aend = Math.max(duration - afade, 0.0);
                if (hasVideo.booleanValue()) {
                    String vvclip = "[" + fileindx + ":v]trim=" + f.format(inpt) + ":duration=" + f.format(duration) + ",setpts=PTS-STARTPTS" + (String)(vfade > 0.0 ? ",fade=t=in:st=0:d=" + vfade + ",fade=t=out:st=" + f.format(vend) + ":d=" + vfade : "") + "[" + outmap + "v" + i + "]";
                    clauses.add(vvclip);
                }
                if (!hasAudio.booleanValue()) continue;
                String aclip = "[" + fileindx + ":a]atrim=" + f.format(inpt) + ":duration=" + f.format(duration) + ",asetpts=PTS-STARTPTS" + (String)(afade > 0.0 ? ",afade=t=in:st=0:d=" + afade + ",afade=t=out:st=" + f.format(aend) + ":d=" + afade : "") + "[" + outmap + "a" + i + "]";
                clauses.add(aclip);
            }
            if (hasVideo.booleanValue()) {
                clauses.add(StringUtils.join(vpads, (String)"") + "concat=n=" + n + ":unsafe=1[ov]");
            }
            if (hasAudio.booleanValue()) {
                clauses.add(StringUtils.join(apads, (String)"") + "concat=n=" + n + ":v=0:a=1[oa]");
            }
        } else if (n == 1) {
            VideoClip vclip = clips.get(0);
            int fileindx = vclip.getSrc();
            double inpt = vclip.getStart();
            double duration = vclip.getDuration();
            double vend = Math.max(duration - vfade, 0.0);
            double aend = Math.max(duration - afade, 0.0);
            if (hasVideo.booleanValue()) {
                String vvclip = "[" + fileindx + ":v]trim=" + f.format(inpt) + ":duration=" + f.format(duration) + ",setpts=PTS-STARTPTS" + (String)(vfade > 0.0 ? ",fade=t=in:st=0:d=" + vfade + ",fade=t=out:st=" + f.format(vend) + ":d=" + vfade : "") + "[ov]";
                clauses.add(vvclip);
            }
            if (hasAudio.booleanValue()) {
                String aclip = "[" + fileindx + ":a]atrim=" + f.format(inpt) + ":duration=" + f.format(duration) + ",asetpts=PTS-STARTPTS" + (String)(afade > 0.0 ? ",afade=t=in:st=0:d=" + afade + ",afade=t=out:st=" + f.format(aend) + ":d=" + afade : "") + "[oa]";
                clauses.add(aclip);
            }
        }
        return clauses;
    }

    private Map<String, String> getParamsFromFile(File parentFile) {
        HashMap<String, String> params = new HashMap<String, String>();
        String videoInput = FilenameUtils.normalize((String)parentFile.getAbsolutePath());
        params.put("in.video.path", videoInput);
        params.put("in.video.name", FilenameUtils.getBaseName((String)videoInput));
        params.put("in.name", FilenameUtils.getBaseName((String)videoInput));
        params.put("in.video.suffix", FilenameUtils.getExtension((String)videoInput));
        params.put("in.video.filename", FilenameUtils.getName((String)videoInput));
        params.put("in.video.mimetype", MimetypesFileTypeMap.getDefaultFileTypeMap().getContentType(videoInput));
        String outDir = parentFile.getAbsoluteFile().getParent();
        params.put("out.dir", outDir);
        String outFileName = FilenameUtils.getBaseName((String)parentFile.getName());
        params.put("out.name.base", outFileName);
        params.put("out.name", outFileName);
        return params;
    }

    public List<File> multiTrimConcat(List<File> inputs, List<Long> edits, List<EncodingProfile> profiles, int transitionDuration) throws EncoderException {
        return this.multiTrimConcat(inputs, edits, profiles, transitionDuration, true, true);
    }

    public List<File> multiTrimConcat(List<File> inputs, List<Long> edits, List<EncodingProfile> profiles, int transitionDuration, boolean hasVideo, boolean hasAudio) throws EncoderException, IllegalArgumentException {
        if (inputs == null || inputs.size() < 1) {
            throw new IllegalArgumentException("At least one track must be specified.");
        }
        if (edits == null && inputs.size() > 1) {
            throw new IllegalArgumentException("If there is no editing, only one track can be specified.");
        }
        List<VideoClip> clips = null;
        if (edits != null) {
            clips = new ArrayList<VideoClip>(edits.size() / 3);
            int adjust = 0;
            for (int i = 0; i < edits.size(); i += 3) {
                adjust = edits.get(i + 1) < (long)transitionDuration ? transitionDuration / 2000 : 0;
                clips.add(new VideoClip(edits.get(i).intValue(), (double)edits.get(i + 1).longValue() / 1000.0 + (double)adjust, (double)edits.get(i + 2).longValue() / 1000.0));
            }
            try {
                clips = EncoderEngine.sortSegments(clips, transitionDuration / 1000);
            }
            catch (Exception e) {
                logger.error("Illegal edits, cannot sort segment", (Throwable)e);
                throw new EncoderException("Cannot understand the edit points", (Throwable)e);
            }
        }
        Map<String, String> params = null;
        if (inputs.size() > 0) {
            params = this.getParamsFromFile(inputs.get(0));
        }
        if (profiles == null || profiles.size() == 0) {
            logger.error("Missing encoding profiles");
            throw new EncoderException("Missing encoding profile(s)");
        }
        try {
            ArrayList<String> command = new ArrayList<String>();
            List<String> clauses = this.makeEdits(clips, transitionDuration, hasVideo, hasAudio);
            String videoOut = clips == null ? "[0:v]" : "[ov]";
            String audioOut = clips == null ? "[0:a]" : "[oa]";
            OutputAggregate outmaps = new OutputAggregate(profiles, params, hasVideo ? videoOut : null, hasAudio ? audioOut : null);
            if (hasAudio) {
                clauses.add(outmaps.getAsplit());
                clauses.add(outmaps.getAudioFilter());
            }
            if (hasVideo) {
                clauses.add(outmaps.getVsplit());
                clauses.add(outmaps.getVideoFilter());
            }
            clauses.removeIf(Objects::isNull);
            command.add("-nostats");
            command.add("-hide_banner");
            for (File o : inputs) {
                command.add("-i");
                command.add(o.getCanonicalPath());
            }
            if (!clauses.isEmpty()) {
                command.add("-filter_complex");
                command.add(StringUtils.join(clauses, (String)";"));
            }
            for (String outpad : outmaps.getOutput()) {
                command.addAll(this.commandSplit(outpad));
            }
            if (outmaps.hasAdaptivePlaylist()) {
                List<File> results = this.process(command);
                List segments = results.stream().filter(AdaptivePlaylist.isHLSFilePred.negate()).collect(Collectors.toList());
                segments.sort((f1, f2) -> f1.getName().compareTo(f2.getName()));
                List<String> suffixes = outmaps.getSegmentOutputSuffixes();
                HashMap<File, File> renames = new HashMap<File, File>();
                results.forEach(f -> renames.put((File)f, (File)f));
                for (int i = 0; i < segments.size(); ++i) {
                    File file = (File)segments.get(i);
                    String newname = FilenameUtils.concat((String)file.getParent(), (String)(FilenameUtils.getBaseName((String)file.getName()) + suffixes.get(i)));
                    renames.put(file, new File(newname));
                }
                return AdaptivePlaylist.hlsRenameAllFiles(results, renames);
            }
            return this.process(command);
        }
        catch (Exception e) {
            logger.error("MultiTrimConcat failed to run command {} ", (Object)e.getMessage());
            throw new EncoderException("Cannot encode the inputs", (Throwable)e);
        }
    }

    protected class OutputAggregate {
        private final List<EncodingProfile> pf;
        private final ArrayList<String> outputs = new ArrayList();
        private final ArrayList<String> outputFiles = new ArrayList();
        private final ArrayList<String> outputSuffixes = new ArrayList();
        private boolean hasAdaptiveProfile = false;
        private final ArrayList<String> vpads;
        private final ArrayList<String> apads;
        private final ArrayList<String> vfilter;
        private final ArrayList<String> afilter;
        private String vInputPad = "";
        private String aInputPad = "";
        private String vsplit = "";
        private String asplit = "";
        private final ArrayList<String> vstream;
        private final ArrayList<String> astream;

        public OutputAggregate(List<EncodingProfile> profiles, Map<String, String> params, String vInputPad, String aInputPad) throws EncoderException {
            ArrayList<EncodingProfile> deliveryProfiles = new ArrayList<EncodingProfile>(profiles.size());
            EncodingProfile groupProfile = null;
            for (EncodingProfile ep : profiles) {
                String adaptiveType = ep.getExtension(EncoderEngine.ADAPTIVE_TYPE_SUFFIX);
                if (adaptiveType == null) {
                    deliveryProfiles.add(ep);
                    continue;
                }
                if ("HLS".equalsIgnoreCase(adaptiveType)) {
                    groupProfile = ep;
                    this.hasAdaptiveProfile = true;
                    continue;
                }
                throw new EncoderException("Only HLS is supported" + ep.getIdentifier() + " ffmpeg command");
            }
            this.pf = deliveryProfiles;
            int size = this.pf.size();
            if (vInputPad == null && aInputPad == null) {
                throw new EncoderException("At least one of video or audio input must be specified");
            }
            this.vfilter = new ArrayList<Object>(Collections.nCopies(size, null));
            this.afilter = new ArrayList<Object>(Collections.nCopies(size, null));
            this.apads = new ArrayList<Object>(Collections.nCopies(size, null));
            this.vpads = new ArrayList<Object>(Collections.nCopies(size, null));
            this.vstream = new ArrayList<Object>(Collections.nCopies(size, null));
            this.astream = new ArrayList<Object>(Collections.nCopies(size, null));
            this.vsplit = size > 1 ? vInputPad + "split=" + size : null;
            this.asplit = size > 1 ? aInputPad + "asplit=" + size : null;
            this.vInputPad = vInputPad;
            this.aInputPad = aInputPad;
            if (groupProfile != null) {
                this.outputAggregateReal(deliveryProfiles, groupProfile, params, vInputPad, aInputPad);
            } else {
                this.outputAggregateReal(deliveryProfiles, params, vInputPad, aInputPad);
            }
        }

        private void setAudioFilters() {
            if (this.pf.size() == 1) {
                if (this.afilter.get(0) != null) {
                    this.afilter.set(0, this.aInputPad + this.afilter.get(0) + this.apads.get(0));
                }
                this.astream.set(0, this.apads.get(0));
            } else {
                for (int i = 0; i < this.pf.size(); ++i) {
                    if (this.afilter.get(i) != null) {
                        this.afilter.set(i, "[oa0" + i + "]" + this.afilter.get(i) + this.apads.get(i));
                        this.asplit = this.asplit + "[oa0" + i + "]";
                        this.astream.set(i, "[oa0" + i + "]");
                        continue;
                    }
                    this.asplit = this.asplit + this.apads.get(i);
                    this.astream.set(i, this.apads.get(i));
                }
            }
            this.afilter.removeAll(Arrays.asList(new String[]{null}));
        }

        private void setVideoFilters() {
            if (this.pf.size() == 1) {
                if (this.vfilter.get(0) != null) {
                    this.vfilter.set(0, this.vInputPad + this.vfilter.get(0) + this.vpads.get(0));
                }
                this.vstream.set(0, this.vpads.get(0));
            } else {
                for (int i = 0; i < this.pf.size(); ++i) {
                    if (this.vfilter.get(i) != null) {
                        this.vfilter.set(i, "[ov0" + i + "]" + this.vfilter.get(i) + this.vpads.get(i));
                        this.vsplit = this.vsplit + "[ov0" + i + "]";
                        this.vstream.set(i, "[ov0" + i + "]");
                        continue;
                    }
                    this.vsplit = this.vsplit + this.vpads.get(i);
                    this.vstream.set(i, this.vpads.get(i));
                }
            }
            this.vfilter.removeAll(Arrays.asList(new String[]{null}));
        }

        public List<String> getOutFiles() {
            return this.outputFiles;
        }

        public List<String> getOutput() {
            return this.outputs;
        }

        public List<String> getSegmentOutputSuffixes() {
            return this.outputSuffixes;
        }

        public boolean hasAdaptivePlaylist() {
            return this.hasAdaptiveProfile;
        }

        public String getVsplit() {
            return this.vsplit;
        }

        public String getAsplit() {
            return this.asplit;
        }

        public String getVideoFilter() {
            if (this.vfilter.isEmpty()) {
                return null;
            }
            return StringUtils.join(this.vfilter, (String)";");
        }

        public String getAudioFilter() {
            if (this.afilter.isEmpty()) {
                return null;
            }
            return StringUtils.join(this.afilter, (String)";");
        }

        public String adjustForNoComplexFilter(String pad) {
            Pattern outpad = Pattern.compile("\\[(\\d+:[av\\d{1,2}])\\]");
            try {
                Matcher matcher = outpad.matcher(pad);
                if (matcher.matches()) {
                    return matcher.group(1);
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            return pad;
        }

        protected String processParameters(String cmd, Map<String, String> params) {
            String r = cmd;
            for (Map.Entry<String, String> e : params.entrySet()) {
                r = r.replace("#{" + e.getKey() + "}", e.getValue());
            }
            return r;
        }

        public void outputAggregateReal(List<EncodingProfile> profiles, EncodingProfile groupProfile, Map<String, String> params, String vInputPad, String aInputPad) throws EncoderException {
            int size = profiles.size();
            try {
                String outSuffix = this.processParameters(groupProfile.getSuffix(), params);
                params.put("out.suffix", outSuffix);
            }
            catch (Exception e) {
                throw new EncoderException("Missing Encoding Profiles");
            }
            String ffmpgGCmd = groupProfile.getExtension(EncoderEngine.CMD_SUFFIX);
            if (ffmpgGCmd == null) {
                throw new EncoderException("Missing ffmpeg Encoding Profile " + groupProfile.getIdentifier() + " ffmpeg command");
            }
            for (Map.Entry<String, String> e : params.entrySet()) {
                ffmpgGCmd = ffmpgGCmd.replace("#{" + e.getKey() + "}", e.getValue());
            }
            ffmpgGCmd = ffmpgGCmd.replace("#{space}", " ");
            int indx = 0;
            for (EncodingProfile profile : profiles) {
                int i;
                List<String> cmdToken;
                Object cmd = "";
                this.outputSuffixes.add(this.processParameters(profile.getSuffix(), params));
                String ffmpgCmd = profile.getExtension(EncoderEngine.CMD_SUFFIX);
                if (ffmpgCmd == null) {
                    throw new EncoderException("Missing Encoding Profile " + profile.getIdentifier() + " ffmpeg command");
                }
                params.remove("out.dir");
                params.remove("out.name");
                params.remove("out.suffix");
                for (Map.Entry<String, String> e : params.entrySet()) {
                    ffmpgCmd = ffmpgCmd.replace("#{" + e.getKey() + "}", e.getValue());
                }
                ffmpgCmd = ffmpgCmd.replace("#{space}", " ");
                try {
                    cmdToken = EncoderEngine.this.commandSplit(ffmpgCmd);
                }
                catch (Exception e) {
                    throw new EncoderException("Could not parse encoding profile command line", (Throwable)e);
                }
                for (i = 0; i < cmdToken.size(); ++i) {
                    if (!cmdToken.get(i).contains("#{out.name}")) continue;
                    if (i == cmdToken.size() - 1) {
                        cmdToken = cmdToken.subList(0, i);
                        break;
                    }
                    List<String> copy = cmdToken.subList(0, i - 1);
                    copy.addAll(cmdToken.subList(i + 1, cmdToken.size() - 1));
                    cmdToken = copy;
                }
                Object maxrate = null;
                for (i = 0; i < cmdToken.size(); ++i) {
                    String opt = cmdToken.get(i);
                    if (opt.startsWith("-vf") || opt.startsWith("-filter:v")) {
                        this.vfilter.set(indx, cmdToken.get(i + 1).replace("\"", ""));
                        ++i;
                        continue;
                    }
                    if (opt.startsWith("-filter_complex") || opt.startsWith("-lavfi")) {
                        ++i;
                        logger.error("Command does not support complex filters - only simple -af or -vf filters are supported");
                        throw new EncoderException("Cannot parse complex filters in" + profile.getIdentifier() + " for this operation");
                    }
                    if (opt.startsWith("-af") || opt.startsWith("-filter:a")) {
                        this.afilter.set(indx, cmdToken.get(i + 1).replace("\"", ""));
                        ++i;
                        continue;
                    }
                    if ("-i".equals(opt)) {
                        ++i;
                        continue;
                    }
                    if (opt.startsWith("-c:") || opt.startsWith("-codec:") || opt.contains("-vcodec") || opt.contains("-acodec")) {
                        String str = cmdToken.get(i + 1);
                        if (str.contains("copy")) {
                            ++i;
                            continue;
                        }
                        if (opt.startsWith("-codec:") || opt.contains("-vcodec")) {
                            cmd = (String)cmd + " " + this.adjustABRVMaps("-c:v", indx);
                            continue;
                        }
                        if (opt.startsWith("-acodec:")) {
                            cmd = (String)cmd + " " + this.adjustABRVMaps("-c:a", indx);
                            continue;
                        }
                        cmd = (String)cmd + " " + this.adjustABRVMaps(opt, indx);
                        continue;
                    }
                    cmd = (String)cmd + " " + this.adjustABRVMaps(opt, indx);
                }
                cmd = ((String)cmd).replaceAll("#\\{.*?\\}", "");
                if (size == 1) {
                    if (this.afilter.get(indx) == null) {
                        this.apads.set(indx, this.adjustForNoComplexFilter(aInputPad));
                    } else {
                        this.apads.set(indx, "[oa" + indx + "]");
                    }
                    if (this.vfilter.get(indx) == null) {
                        this.vpads.set(indx, this.adjustForNoComplexFilter(vInputPad));
                    } else {
                        this.vpads.set(indx, "[ov" + indx + "]");
                    }
                } else {
                    this.vpads.set(indx, "[ov" + indx + "]");
                    this.apads.set(indx, "[oa" + indx + "]");
                }
                if ((cmd = StringUtils.trimToNull((String)cmd)) == null) continue;
                if (vInputPad != null) {
                    this.outputs.add("-map " + this.vpads.get(indx));
                }
                if (aInputPad != null) {
                    this.outputs.add("-map " + this.apads.get(indx));
                }
                this.outputs.add((String)cmd);
                ++indx;
            }
            this.setVideoFilters();
            this.setAudioFilters();
            this.setHLSVarStreamMap(ffmpgGCmd, vInputPad != null, aInputPad != null);
        }

        private void setHLSVarStreamMap(String ffmpgCmd, boolean hasVideo, boolean hasAudio) {
            StringBuilder varStreamMap = new StringBuilder();
            varStreamMap.append(" -var_stream_map '");
            for (int i = 0; i < this.pf.size(); ++i) {
                int j = 0;
                String[] maps = new String[2];
                if (hasVideo && this.vstream.get(i) != null) {
                    maps[j] = "v:" + i;
                    ++j;
                }
                if (hasAudio && this.astream.get(i) != null) {
                    maps[j] = "a:" + i;
                }
                varStreamMap.append(EncoderEngine.this.joinNonNullString(maps, ","));
                varStreamMap.append(" ");
            }
            varStreamMap.append("' ");
            varStreamMap.append(ffmpgCmd);
            varStreamMap.append(" ");
            this.outputs.add(varStreamMap.toString());
        }

        public String adjustABRVMaps(String option, int position) {
            if (option.endsWith(":v") || option.endsWith(":a")) {
                return option + ":" + Integer.toString(position);
            }
            if (mappableOptions.contains(option)) {
                return option + ":v:" + Integer.toString(position);
            }
            return option;
        }

        public void outputAggregateReal(List<EncodingProfile> profiles, Map<String, String> params, String vInputPad, String aInputPad) throws EncoderException {
            int size = profiles.size();
            int indx = 0;
            for (EncodingProfile profile : profiles) {
                String[] arguments;
                Object cmd = "";
                String outFileName = params.get("out.name.base") + "_" + IdImpl.fromUUID().toString();
                params.put("out.name", outFileName);
                try {
                    String outSuffix = this.processParameters(profile.getSuffix(), params);
                    params.put("out.suffix", outSuffix);
                }
                catch (Exception e) {
                    throw new EncoderException("Missing Encoding Profiles");
                }
                String ffmpgCmd = profile.getExtension(EncoderEngine.CMD_SUFFIX);
                if (ffmpgCmd == null) {
                    throw new EncoderException("Missing Encoding Profile " + profile.getIdentifier() + " ffmpeg command");
                }
                for (Map.Entry<String, String> e : params.entrySet()) {
                    ffmpgCmd = ffmpgCmd.replace("#{" + e.getKey() + "}", e.getValue());
                }
                ffmpgCmd = ffmpgCmd.replace("#{space}", " ");
                try {
                    arguments = CommandLineUtils.translateCommandline(ffmpgCmd);
                }
                catch (Exception e) {
                    throw new EncoderException("Could not parse encoding profile command line", (Throwable)e);
                }
                List<String> cmdToken = Arrays.asList(arguments);
                for (int i = 0; i < cmdToken.size(); ++i) {
                    String opt = cmdToken.get(i);
                    if (opt.startsWith("-vf") || opt.startsWith("-filter:v")) {
                        this.vfilter.set(indx, cmdToken.get(i + 1).replace("\"", ""));
                        ++i;
                        continue;
                    }
                    if (opt.startsWith("-filter_complex") || opt.startsWith("-lavfi")) {
                        ++i;
                        logger.error("Command does not support complex filters - only simple -af or -vf filters are supported");
                        throw new EncoderException("Cannot parse complex filters in" + profile.getIdentifier() + " for this operation");
                    }
                    if (opt.startsWith("-af") || opt.startsWith("-filter:a")) {
                        this.afilter.set(indx, cmdToken.get(i + 1).replace("\"", ""));
                        ++i;
                        continue;
                    }
                    if ("-i".equals(opt)) {
                        ++i;
                        continue;
                    }
                    if (opt.startsWith("-c:") || opt.startsWith("-codec:") || opt.contains("-vcodec") || opt.contains("-acodec")) {
                        String str = cmdToken.get(i + 1);
                        if (str.contains("copy")) {
                            ++i;
                            continue;
                        }
                        cmd = (String)cmd + " " + opt;
                        continue;
                    }
                    cmd = (String)cmd + " " + opt;
                }
                cmd = ((String)cmd).replaceAll("#\\{.*?\\}", "");
                if (size == 1) {
                    if (this.afilter.get(indx) == null) {
                        this.apads.set(indx, this.adjustForNoComplexFilter(aInputPad));
                    } else {
                        this.apads.set(indx, "[oa" + indx + "]");
                    }
                    if (this.vfilter.get(indx) == null) {
                        this.vpads.set(indx, this.adjustForNoComplexFilter(vInputPad));
                    } else {
                        this.vpads.set(indx, "[ov" + indx + "]");
                    }
                } else {
                    this.vpads.set(indx, "[ov" + indx + "]");
                    this.apads.set(indx, "[oa" + indx + "]");
                }
                if ((cmd = StringUtils.trimToNull((String)cmd)) == null) continue;
                this.outputFiles.add(cmdToken.get(cmdToken.size() - 1));
                if (vInputPad != null) {
                    this.outputs.add("-map " + this.vpads.get(indx));
                }
                if (aInputPad != null) {
                    this.outputs.add("-map " + this.apads.get(indx));
                }
                this.outputs.add((String)cmd);
                ++indx;
            }
            this.setVideoFilters();
            this.setAudioFilters();
        }
    }
}

