/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2006 Bull S.A.
 * Contact: jonas-team@objectweb.org
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or any later
 * version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * --------------------------------------------------------------------------
 * $Id: DiscoveryHelper.java 10761 2007-06-27 08:38:39Z danesa $
 * --------------------------------------------------------------------------
 */

package org.objectweb.jonas.discovery.internal.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;

import org.objectweb.jonas.discovery.internal.comm.message.DiscEvent;
import org.objectweb.jonas.discovery.internal.comm.message.DiscGreeting;
import org.objectweb.jonas.discovery.internal.comm.message.DiscMessage;

/**
 * This class helps creating a byte[] to be sent in a datagram when sending a DiscMessage, and helps
 * reading the sent DiscMessage on receiving an object when this object is actually a DiscMessage.
 * @author Adriana Danes
 */
public class DiscoveryHelper {
    /**
     * Constant for DiscMessage objects type.
     */
    private static int DISC_MESSAGE = 0;
    /**
     * Constant for DiscEvent objects type.
     */
    private static int DISC_EVENT = 1;
    /**
     * Constant for DiscGreeting objects type.
     */
    private static int DISC_GREETING = 2;
    /**
     *
     */
    private static int RECEIVE_BUFFER_SIZE = 1024;
    /**
     * Construct a byte[] containing type info about a DiscMessage object to be sent,
     * plus the content of this message.
     * @param obj Object to be send. Only supported DiscMessage objects.
     * @return Null if the object is not an instance of DiscMessage or one of its subclasses.
     * @throws IOException Could not create an ObjectOutputStream to write into the
     * underlying ByteArrayOutputStream.
     */
    public static byte[] objectToBytes(Object obj) throws IOException {
        byte[] resultBytes = null;
        if (!(obj instanceof DiscMessage)) {
            return null;
        }
        ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
        ObjectOutputStream stream = new ObjectOutputStream(byteStream);
        try {
            if (obj instanceof DiscEvent) {
                stream.writeInt(DISC_EVENT);
            } else if (obj instanceof DiscGreeting) {
                stream.writeInt(DISC_GREETING);
            } else {
                stream.writeInt(DISC_MESSAGE);
            }
            stream.writeObject(obj);
            resultBytes = byteStream.toByteArray();
            byteStream.close();
            stream.close();
        } catch (IOException e) {
            throw e;
        } finally {
            byteStream.close();
            stream.close();
        }
        return resultBytes;
    }

    /**
     *
     * @param bytes  byte[] containing a received message
     * @return Null if the object in the received message is not of one of the known types, or the object which
     * have been sent.
     * @throws IOException Could not create an ObjectInputStream to read in it
     * @throws ClassNotFoundException Class of a serialized object cannot be found
     */
    public static Object bytesToObject(byte[] bytes) throws IOException, ClassNotFoundException {
        Object resultObject = null;
        ByteArrayInputStream byteStream = new ByteArrayInputStream(bytes);
        ObjectInputStream stream = new ObjectInputStream(byteStream);
        try {
            int type = stream.readInt();
            if (type == DISC_MESSAGE) {
                resultObject = (DiscMessage) stream.readObject();
            } else if (type == DISC_EVENT) {
                resultObject = (DiscEvent) stream.readObject();
            } else if (type == DISC_GREETING) {
                resultObject = (DiscGreeting) stream.readObject();
            }
        } catch (IOException e) {
            throw e;
        } catch (ClassNotFoundException e) {
            throw e;
        } finally {
            byteStream.close();
            stream.close();
        }
        return resultObject;
    }

    /**
     * Send a discovery message to a destination via a datagram socket (UDP).
     * The message is instance of DiscMessage class or one of its subclasses
     * (DiscEvent, DiscGreeting).
     * The socket may be simple or multicast.
     * @param msg object to be sent
     * @param socket sending point
     * @param destAddress destination address (maybe a multicast address)
     * @param destPort destination port number
     * @throws IOException if an I/O error occurs during sending,
     * or if can't create an ObjectOutputStream to write into it the message to send
     */
    private void sendDiscoveryMessage(DiscMessage msg, DatagramSocket socket
            , InetAddress destAddress, int destPort) throws IOException {

        byte[] messageBytes = objectToBytes(msg);
        // Create datagram packet for sending
        DatagramPacket packet = new DatagramPacket(messageBytes, messageBytes.length
                , destAddress, destPort);
        sendPacket(socket, packet);
    }

    /**
     * Receive a discovery message via a datagram socket (UDP).
     * The message is instance of DiscMessage class or one of its subclasses
     * (DiscEvent, DiscGreeting).
     * The socket may be simple or multicast which were joind by the caller.
     * @param socket receiving point
     * @return received object
     * @throws IOException if an I/O error occurs during receiving,
     * or if can't create an ObjectInputStream to read the object into it.
     * @throws ClassNotFoundException Class of a serialized object cannot be found
     */
    private DiscMessage receiveDiscoveryMessage(DatagramSocket socket) throws IOException, ClassNotFoundException {
        // Create datagram for receiving
        DatagramPacket packet = new DatagramPacket(new byte[RECEIVE_BUFFER_SIZE], RECEIVE_BUFFER_SIZE);
        receivePacket(socket, packet);
        return (DiscMessage) bytesToObject(packet.getData());
    }
    /**
     *
     * @param socket May be a unicast DatagramSocket or a MulticastSocket
     * @param packet datagram packet to send
     * @throws IOException if an I/O error occurs during sending
     */
    private void sendPacket(DatagramSocket socket, DatagramPacket packet) throws IOException {
        socket.send(packet);
    }
    /**
     *
     * @param socket May be a unicast DatagramSocket or a MulticastSocket which the caller joind
     * @param packet datagram packet to send
     * @throws IOException if an I/O error occurs during receiving
     */
    private void receivePacket(DatagramSocket socket, DatagramPacket packet) throws IOException {
        socket.receive(packet);
    }
}