/*
 * Decompiled with CFR 0.152.
 */
package org.netpreserve.jwarc;

import java.io.IOException;
import java.io.InputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.URI;
import java.net.URLConnection;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.netpreserve.jwarc.Capture;
import org.netpreserve.jwarc.CaptureIndex;
import org.netpreserve.jwarc.HttpRequest;
import org.netpreserve.jwarc.HttpResponse;
import org.netpreserve.jwarc.HttpServer;
import org.netpreserve.jwarc.LengthedBody;
import org.netpreserve.jwarc.MediaType;
import org.netpreserve.jwarc.MessageBody;
import org.netpreserve.jwarc.WarcReader;
import org.netpreserve.jwarc.WarcResponse;

class WarcServer {
    private static final DateTimeFormatter ARC_DATE = DateTimeFormatter.ofPattern("yyyyMMddHHmmss").withZone(ZoneOffset.UTC);
    private static final DateTimeFormatter RFC_1123_UTC = DateTimeFormatter.RFC_1123_DATE_TIME.withZone(ZoneOffset.UTC);
    private static final MediaType LINK_FORMAT = MediaType.parse("application/link-format");
    private static final Pattern REPLAY_RE = Pattern.compile("/replay/([0-9]{14})/(.*)");
    private final HttpServer httpServer;
    private final CaptureIndex index;
    private byte[] script = "<!doctype html><script src='/__jwarc__/inject.js'></script>\n".getBytes(StandardCharsets.US_ASCII);

    WarcServer(ServerSocket serverSocket, List<Path> warcs) throws IOException {
        this.httpServer = new HttpServer(serverSocket, this::handle);
        this.index = new CaptureIndex(warcs);
    }

    void listen() throws IOException {
        this.httpServer.listen();
    }

    private void handle(Socket socket, String target, HttpRequest request) throws Exception {
        if (target.equals("/")) {
            Capture entrypoint = this.index.entrypoint();
            if (entrypoint == null) {
                this.error(socket, 404, "Empty collection");
                return;
            }
            HttpServer.send(socket, ((HttpResponse.Builder)((HttpResponse.Builder)new HttpResponse.Builder(307, "Redirect").addHeader("Content-Length", "0")).addHeader("Location", "/replay/" + ARC_DATE.format(entrypoint.date()) + "/" + entrypoint.uri())).build());
        } else if (target.equals("/__jwarc__/sw.js")) {
            this.serve(socket, "sw.js");
        } else if (target.equals("/__jwarc__/inject.js")) {
            this.serve(socket, "inject.js");
        } else if (target.startsWith("/replay/")) {
            if (!request.headers().first("x-serviceworker").isPresent()) {
                HttpServer.send(socket, ((HttpResponse.Builder)new HttpResponse.Builder(200, "OK").body(MediaType.HTML, this.script)).build());
                return;
            }
            Matcher m = REPLAY_RE.matcher(target);
            if (!m.matches()) {
                this.error(socket, 404, "Malformed replay url");
                return;
            }
            Instant date = Instant.from(ARC_DATE.parse(m.group(1)));
            this.replay(socket, m.group(2), date, false);
        } else if (target.startsWith("/timemap/")) {
            URI uri = URI.create(target.substring("/timemap/".length()));
            NavigableSet<Capture> versions = this.index.query(uri);
            if (versions.isEmpty()) {
                this.error(socket, 404, "Not found in archive");
                return;
            }
            StringBuilder sb = new StringBuilder();
            sb.append("<").append(((Capture)versions.first()).uri()).append(">;rel=\"original\"");
            for (Capture entry : versions) {
                sb.append(",\n</replay/").append(ARC_DATE.format(entry.date())).append("/").append(entry.uri()).append(">;rel=\"memento\",datetime=\"").append(RFC_1123_UTC.format(entry.date()) + "\"");
            }
            sb.append("\n");
            HttpServer.send(socket, ((HttpResponse.Builder)new HttpResponse.Builder(200, "OK").body(LINK_FORMAT, sb.toString().getBytes(StandardCharsets.UTF_8))).build());
        } else {
            Instant date = request.headers().first("Accept-Datetime").map(s -> Instant.from(RFC_1123_UTC.parse((CharSequence)s))).orElse(Instant.EPOCH);
            this.replay(socket, target, date, true);
        }
    }

    private void replay(Socket socket, String target, Instant date, boolean proxy) throws IOException {
        URI uri = URI.create(target);
        NavigableSet<Capture> versions = this.index.query(uri);
        if (versions.isEmpty()) {
            this.error(socket, 404, "Not found in archive");
            return;
        }
        Capture capture = this.closest(versions, uri, date);
        try (FileChannel channel = FileChannel.open(capture.file(), StandardOpenOption.READ);){
            channel.position(capture.position());
            WarcReader reader = new WarcReader(channel);
            WarcResponse record = (WarcResponse)reader.next().get();
            HttpResponse http = record.http();
            HttpResponse.Builder b = new HttpResponse.Builder(http.status(), http.reason());
            for (Map.Entry<String, List<String>> e : http.headers().map().entrySet()) {
                if (e.getKey().equalsIgnoreCase("Strict-Transport-Security") || e.getKey().equalsIgnoreCase("Transfer-Encoding") || e.getKey().equalsIgnoreCase("Public-Key-Pins")) continue;
                for (String value : e.getValue()) {
                    b.addHeader(e.getKey(), value);
                }
            }
            b.setHeader("Connection", "keep-alive");
            b.setHeader("Memento-Datetime", RFC_1123_UTC.format(record.date()));
            if (!proxy) {
                b.setHeader("Link", this.mementoLinks(versions, capture));
            }
            if (proxy) {
                b.setHeader("Vary", "Accept-Datetime");
            }
            MessageBody body = http.body();
            if (!proxy && MediaType.HTML.equals(http.contentType().base())) {
                body = LengthedBody.create(body, ByteBuffer.wrap(this.script), (long)this.script.length + body.size());
            }
            b.body(http.contentType(), body, body.size());
            HttpServer.send(socket, b.build());
        }
    }

    private String mementoLinks(NavigableSet<Capture> versions, Capture current) {
        StringBuilder sb = new StringBuilder();
        sb.append("<").append(current.uri()).append(">;rel=\"original\",");
        sb.append("</timemap/").append(current.uri()).append(">;rel=\"timemap\";type=\"").append(LINK_FORMAT).append('\"');
        this.mementoLink(sb, "first ", current, (Capture)versions.first());
        this.mementoLink(sb, "prev ", current, versions.lower(current));
        this.mementoLink(sb, "next ", current, versions.higher(current));
        this.mementoLink(sb, "last ", current, (Capture)versions.last());
        return sb.toString();
    }

    private void mementoLink(StringBuilder sb, String rel, Capture current, Capture capture) {
        if (capture == null || capture.date().equals(current.date())) {
            return;
        }
        if (sb.length() != 0) {
            sb.append(',');
        }
        sb.append("</replay/").append(ARC_DATE.format(capture.date())).append("/").append(capture.uri()).append(">;rel=\"").append(rel).append("memento\";datetime=\"").append(RFC_1123_UTC.format(capture.date())).append("\"");
    }

    private void error(Socket socket, int status, String reason) throws IOException {
        HttpServer.send(socket, ((HttpResponse.Builder)((HttpResponse.Builder)new HttpResponse.Builder(status, reason).body(MediaType.HTML, reason.getBytes(StandardCharsets.UTF_8))).setHeader("Connection", "keep-alive")).build());
    }

    private void serve(Socket socket, String resource) throws IOException {
        URLConnection conn = this.getClass().getResource(resource).openConnection();
        long length = conn.getContentLengthLong();
        if (length == -1L) {
            byte[] buf = new byte[8192];
            try (InputStream stream = conn.getInputStream();){
                int n;
                length = 0L;
                while ((n = stream.read(buf)) != -1) {
                    length += (long)n;
                }
            }
        }
        try (InputStream stream = conn.getInputStream();){
            HttpServer.send(socket, ((HttpResponse.Builder)((HttpResponse.Builder)new HttpResponse.Builder(200, "OK").body(MediaType.parse("application/javascript"), Channels.newChannel(stream), length)).setHeader("Service-Worker-Allowed", "/")).build());
        }
    }

    private Capture closest(NavigableSet<Capture> versions, URI uri, Instant date) {
        Duration db;
        Capture key = new Capture(uri, date);
        Capture a = versions.floor(key);
        Capture b = versions.higher(key);
        if (a == null) {
            return b;
        }
        if (b == null) {
            return a;
        }
        Duration da = Duration.between(a.date(), date);
        return da.compareTo(db = Duration.between(b.date(), date)) < 0 ? a : b;
    }
}

