/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.LongConsumer;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.neo4j.internal.helpers.Numbers;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.IOUtils;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.memory.HeapScopedBuffer;
import org.neo4j.io.memory.NativeScopedBuffer;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.files.LogFile;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.checkpoint.CheckpointFile;
import org.neo4j.kernel.impl.transaction.log.files.checkpoint.CheckpointInfo;
import org.neo4j.memory.MemoryTracker;

public class CorruptedLogsTruncator {
    public static final String CORRUPTED_TX_LOGS_BASE_NAME = "corrupted-neostore.transaction.db";
    private static final String LOG_FILE_ARCHIVE_PATTERN = "corrupted-neostore.transaction.db-%d-%d-%d.zip";
    private final Path storeDir;
    private final LogFiles logFiles;
    private final FileSystemAbstraction fs;
    private final MemoryTracker memoryTracker;

    public CorruptedLogsTruncator(Path storeDir, LogFiles logFiles, FileSystemAbstraction fs, MemoryTracker memoryTracker) {
        this.storeDir = storeDir;
        this.logFiles = logFiles;
        this.fs = fs;
        this.memoryTracker = memoryTracker;
    }

    public void truncate(LogPosition positionAfterLastRecoveredTransaction) throws IOException {
        long recoveredTransactionOffset;
        long recoveredTransactionLogVersion = positionAfterLastRecoveredTransaction.getLogVersion();
        if (this.isRecoveredLogCorrupted(recoveredTransactionLogVersion, recoveredTransactionOffset = positionAfterLastRecoveredTransaction.getByteOffset()) || this.haveMoreRecentLogFiles(recoveredTransactionLogVersion)) {
            Optional<CheckpointInfo> corruptCheckpoint = this.findFirstCorruptDetachedCheckpoint(recoveredTransactionLogVersion, recoveredTransactionOffset);
            this.backupCorruptedContent(recoveredTransactionLogVersion, recoveredTransactionOffset, corruptCheckpoint);
            this.truncateLogFiles(recoveredTransactionLogVersion, recoveredTransactionOffset, corruptCheckpoint);
        }
    }

    private void truncateLogFiles(long recoveredTransactionLogVersion, long recoveredTransactionOffset, Optional<CheckpointInfo> corruptCheckpoint) throws IOException {
        LogFile transactionLogFile = this.logFiles.getLogFile();
        this.truncateFilesFromVersion(recoveredTransactionLogVersion, recoveredTransactionOffset, transactionLogFile.getHighestLogVersion(), transactionLogFile::getLogFileForVersion);
        if (corruptCheckpoint.isPresent()) {
            LogPosition checkpointPosition = corruptCheckpoint.get().getCheckpointEntryPosition();
            CheckpointFile checkpointFile = this.logFiles.getCheckpointFile();
            this.truncateFilesFromVersion(checkpointPosition.getLogVersion(), checkpointPosition.getByteOffset(), checkpointFile.getCurrentDetachedLogVersion(), checkpointFile::getDetachedCheckpointFileForVersion);
        }
    }

    private void truncateFilesFromVersion(long recoveredLogVersion, long recoveredOffset, long highestLogVersion, Function<Long, Path> getFileForVersion) throws IOException {
        Path lastRecoveredLog = getFileForVersion.apply(recoveredLogVersion);
        this.fs.truncate(lastRecoveredLog, recoveredOffset);
        CorruptedLogsTruncator.forEachSubsequentFile(recoveredLogVersion, highestLogVersion, IOUtils.uncheckedLongConsumer(fileIndex -> this.fs.deleteFile((Path)getFileForVersion.apply(fileIndex))));
    }

    private static void forEachSubsequentFile(long recoveredLogVersion, long highestLogVersion, LongConsumer action) {
        for (long fileIndex = recoveredLogVersion + 1L; fileIndex <= highestLogVersion; ++fileIndex) {
            action.accept(fileIndex);
        }
    }

    private void backupCorruptedContent(long recoveredTransactionLogVersion, long recoveredTransactionOffset, Optional<CheckpointInfo> corruptCheckpoint) throws IOException {
        Path corruptedLogArchive = this.getArchiveFile(recoveredTransactionLogVersion, recoveredTransactionOffset);
        try (ZipOutputStream recoveryContent = new ZipOutputStream(this.fs.openAsOutputStream(corruptedLogArchive, false));
             HeapScopedBuffer bufferScope = new HeapScopedBuffer(1, ByteUnit.MebiByte, this.memoryTracker);){
            LogFile transactionLogFile = this.logFiles.getLogFile();
            this.copyLogsContent(recoveredTransactionLogVersion, recoveredTransactionOffset, transactionLogFile.getHighestLogVersion(), recoveryContent, bufferScope, transactionLogFile::getLogFileForVersion);
            if (corruptCheckpoint.isPresent()) {
                LogPosition checkpointPosition = corruptCheckpoint.get().getCheckpointEntryPosition();
                CheckpointFile checkpointFile = this.logFiles.getCheckpointFile();
                this.copyLogsContent(checkpointPosition.getLogVersion(), checkpointPosition.getByteOffset(), checkpointFile.getCurrentDetachedLogVersion(), recoveryContent, bufferScope, checkpointFile::getDetachedCheckpointFileForVersion);
            }
        }
    }

    private void copyLogsContent(long recoveredLogVersion, long recoveredOffset, long highestLogVersion, ZipOutputStream recoveryContent, HeapScopedBuffer bufferScope, Function<Long, Path> getFileForVersion) throws IOException {
        this.copyLogContent(recoveredLogVersion, recoveredOffset, recoveryContent, bufferScope.getBuffer(), getFileForVersion);
        CorruptedLogsTruncator.forEachSubsequentFile(recoveredLogVersion, highestLogVersion, fileIndex -> {
            try {
                this.copyLogContent(fileIndex, 0L, recoveryContent, bufferScope.getBuffer(), getFileForVersion);
            }
            catch (IOException io) {
                throw new UncheckedIOException(io);
            }
        });
    }

    private Path getArchiveFile(long recoveredTransactionLogVersion, long recoveredTransactionOffset) throws IOException {
        Path corruptedLogsFolder = this.storeDir.resolve(CORRUPTED_TX_LOGS_BASE_NAME);
        this.fs.mkdirs(corruptedLogsFolder);
        return corruptedLogsFolder.resolve(String.format(LOG_FILE_ARCHIVE_PATTERN, recoveredTransactionLogVersion, recoveredTransactionOffset, System.currentTimeMillis()));
    }

    private void copyLogContent(long logFileIndex, long logOffset, ZipOutputStream destination, ByteBuffer byteBuffer, Function<Long, Path> getFileForVersion) throws IOException {
        Path logFile = getFileForVersion.apply(logFileIndex);
        if (this.fs.getFileSize(logFile) == logOffset) {
            return;
        }
        this.addLogFileToZipStream(logOffset, destination, byteBuffer, logFile);
    }

    private void addLogFileToZipStream(long logOffset, ZipOutputStream destination, ByteBuffer byteBuffer, Path logFile) throws IOException {
        ZipEntry zipEntry = new ZipEntry(logFile.getFileName().toString());
        destination.putNextEntry(zipEntry);
        try (StoreChannel transactionLogChannel = this.fs.read(logFile);){
            transactionLogChannel.position(logOffset);
            while (transactionLogChannel.read(byteBuffer) >= 0) {
                byteBuffer.flip();
                destination.write(byteBuffer.array(), byteBuffer.position(), byteBuffer.remaining());
                byteBuffer.clear();
            }
        }
        destination.closeEntry();
    }

    private boolean haveMoreRecentLogFiles(long recoveredTransactionLogVersion) {
        return this.logFiles.getLogFile().getHighestLogVersion() > recoveredTransactionLogVersion;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isRecoveredLogCorrupted(long recoveredTransactionLogVersion, long recoveredTransactionOffset) throws IOException {
        try {
            LogFile logFile = this.logFiles.getLogFile();
            if (Files.size(logFile.getLogFileForVersion(recoveredTransactionLogVersion)) <= recoveredTransactionOffset) return false;
            try (PhysicalLogVersionedStoreChannel channel = logFile.openForVersion(recoveredTransactionLogVersion);
                 NativeScopedBuffer scopedBuffer = new NativeScopedBuffer(Numbers.safeCastLongToInt((long)ByteUnit.kibiBytes((long)64L)), this.memoryTracker);){
                channel.position(recoveredTransactionOffset);
                ByteBuffer byteBuffer = scopedBuffer.getBuffer();
                while (channel.read(byteBuffer) >= 0) {
                    byteBuffer.flip();
                    while (byteBuffer.hasRemaining()) {
                        if (byteBuffer.get() == 0) continue;
                        boolean bl = true;
                        return bl;
                    }
                    byteBuffer.clear();
                }
                return false;
            }
        }
        catch (NoSuchFileException ignored) {
            return false;
        }
    }

    private Optional<CheckpointInfo> findFirstCorruptDetachedCheckpoint(long recoveredTransactionLogVersion, long recoveredTransactionOffset) throws IOException {
        List<CheckpointInfo> detachedCheckpoints = this.logFiles.getCheckpointFile().getReachableDetachedCheckpoints();
        for (CheckpointInfo checkpoint : detachedCheckpoints) {
            LogPosition transactionLogPosition = checkpoint.getTransactionLogPosition();
            long logVersion = transactionLogPosition.getLogVersion();
            if (logVersion <= recoveredTransactionLogVersion && (logVersion != recoveredTransactionLogVersion || transactionLogPosition.getByteOffset() <= recoveredTransactionOffset)) continue;
            return Optional.of(checkpoint);
        }
        return Optional.empty();
    }
}

