/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.util.Arrays;
import org.neo4j.exceptions.UnderlyingStorageException;
import org.neo4j.internal.helpers.Numbers;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.io.memory.HeapScopedBuffer;
import org.neo4j.kernel.impl.transaction.log.LogEntryCursor;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.LogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.ReadAheadLogChannel;
import org.neo4j.kernel.impl.transaction.log.ReadableClosablePositionAwareChecksumChannel;
import org.neo4j.kernel.impl.transaction.log.entry.CheckPoint;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntry;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntryCommit;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntryReader;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntryStart;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeader;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.recovery.LogTailScannerMonitor;
import org.neo4j.kernel.recovery.Recovery;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.NullLogProvider;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.monitoring.Monitors;
import org.neo4j.storageengine.api.StoreId;

public class LogTailScanner {
    static final long NO_TRANSACTION_ID = -1L;
    private final LogFiles logFiles;
    private final LogEntryReader logEntryReader;
    private LogTailInformation logTailInformation;
    private final LogTailScannerMonitor monitor;
    private final boolean failOnCorruptedLogFiles;
    private final Log log;
    private final MemoryTracker memoryTracker;

    public LogTailScanner(LogFiles logFiles, LogEntryReader logEntryReader, Monitors monitors, MemoryTracker memoryTracker) {
        this(logFiles, logEntryReader, monitors, false, memoryTracker);
    }

    public LogTailScanner(LogFiles logFiles, LogEntryReader logEntryReader, Monitors monitors, boolean failOnCorruptedLogFiles, MemoryTracker memoryTracker) {
        this(logFiles, logEntryReader, monitors, failOnCorruptedLogFiles, (LogProvider)NullLogProvider.getInstance(), memoryTracker);
    }

    public LogTailScanner(LogFiles logFiles, LogEntryReader logEntryReader, Monitors monitors, boolean failOnCorruptedLogFiles, LogProvider log, MemoryTracker memoryTracker) {
        this.logFiles = logFiles;
        this.logEntryReader = logEntryReader;
        this.monitor = (LogTailScannerMonitor)monitors.newMonitor(LogTailScannerMonitor.class, new String[0]);
        this.failOnCorruptedLogFiles = failOnCorruptedLogFiles;
        this.log = log.getLog(this.getClass());
        this.memoryTracker = memoryTracker;
    }

    private LogTailInformation findLogTail() throws IOException {
        long highestLogVersion;
        long version = highestLogVersion = this.logFiles.getHighestLogVersion();
        long versionToSearchForCommits = highestLogVersion;
        LogEntryStart latestStartEntry = null;
        long oldestStartEntryTransaction = -1L;
        long oldestVersionFound = -1L;
        byte latestLogEntryVersion = 0;
        boolean startRecordAfterCheckpoint = false;
        boolean corruptedTransactionLogs = false;
        while (version >= this.logFiles.getLowestLogVersion() && version >= 0L) {
            this.log.info("Scanning transaction file with version %d for checkpoint entries", new Object[]{version});
            oldestVersionFound = version;
            CheckPoint latestCheckPoint = null;
            StoreId storeId = StoreId.UNKNOWN;
            try (PhysicalLogVersionedStoreChannel channel = this.logFiles.openForVersion(version);
                 LogEntryCursor cursor = new LogEntryCursor(this.logEntryReader, (ReadableClosablePositionAwareChecksumChannel)new ReadAheadLogChannel(channel, this.memoryTracker));){
                LogHeader logHeader = this.logFiles.extractHeader(version);
                storeId = logHeader.getStoreId();
                while (cursor.next()) {
                    LogEntry entry = cursor.get();
                    if (entry instanceof CheckPoint) {
                        latestCheckPoint = (CheckPoint)entry;
                    } else if (entry instanceof LogEntryCommit) {
                        if (oldestStartEntryTransaction == -1L) {
                            oldestStartEntryTransaction = ((LogEntryCommit)entry).getTxId();
                        }
                    } else if (entry instanceof LogEntryStart) {
                        LogEntryStart startEntry = (LogEntryStart)entry;
                        if (version == versionToSearchForCommits) {
                            latestStartEntry = startEntry;
                        }
                        startRecordAfterCheckpoint = true;
                    }
                    if (version != versionToSearchForCommits && latestLogEntryVersion != 0) continue;
                    latestLogEntryVersion = entry.getVersion();
                }
                this.verifyReaderPosition(highestLogVersion, version, channel);
            }
            catch (Error | ClosedByInterruptException e) {
                throw e;
            }
            catch (Throwable t) {
                this.monitor.corruptedLogFile(version, t);
                if (this.failOnCorruptedLogFiles) {
                    Recovery.throwUnableToCleanRecover(t);
                }
                corruptedTransactionLogs = true;
            }
            if (latestCheckPoint != null) {
                return this.checkpointTailInformation(highestLogVersion, latestStartEntry, oldestVersionFound, latestLogEntryVersion, latestCheckPoint, corruptedTransactionLogs, storeId);
            }
            --version;
            if (latestStartEntry != null) continue;
            --versionToSearchForCommits;
        }
        return new LogTailInformation(corruptedTransactionLogs || startRecordAfterCheckpoint, oldestStartEntryTransaction, oldestVersionFound, highestLogVersion, latestLogEntryVersion);
    }

    private void verifyReaderPosition(long highestLogVersion, long version, LogVersionedStoreChannel channel) throws IOException {
        LogPosition logPosition = this.logEntryReader.lastPosition();
        this.verifyLogVersion(version, logPosition);
        long logFileSize = channel.size();
        long channelLeftovers = Math.subtractExact(logFileSize, logPosition.getByteOffset());
        if (channelLeftovers != 0L) {
            this.verifyLastFile(highestLogVersion, version, logPosition, logFileSize, channelLeftovers);
            this.verifyNoMoreReadableDataAvailable(version, channel, logPosition, channelLeftovers);
        }
    }

    private void verifyLogVersion(long version, LogPosition logPosition) {
        if (logPosition.getLogVersion() != version) {
            throw new IllegalStateException(String.format("Expected to observe log positions only for log file with version %d but encountered version %d while reading %s.", version, logPosition.getLogVersion(), FileUtils.getCanonicalFile((File)this.logFiles.getLogFileForVersion(version))));
        }
    }

    private void verifyLastFile(long highestLogVersion, long version, LogPosition logPosition, long logFileSize, long channelLeftovers) {
        if (version != highestLogVersion) {
            throw new RuntimeException(String.format("Transaction log files with version %d has %d unreadable bytes. Was able to read upto %d but %d is available.", version, channelLeftovers, logPosition.getByteOffset(), logFileSize));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyNoMoreReadableDataAvailable(long version, LogVersionedStoreChannel channel, LogPosition logPosition, long channelLeftovers) throws IOException {
        long initialPosition = channel.position();
        try {
            channel.position(logPosition.getByteOffset());
            try (HeapScopedBuffer scopedBuffer = new HeapScopedBuffer(Numbers.safeCastLongToInt((long)Math.min(ByteUnit.kibiBytes((long)12L), channelLeftovers)), this.memoryTracker);){
                ByteBuffer byteBuffer = scopedBuffer.getBuffer();
                channel.readAll(byteBuffer);
                byteBuffer.flip();
                if (!LogTailScanner.isAllZerosBuffer(byteBuffer)) {
                    throw new RuntimeException(String.format("Transaction log files with version %d has some data available after last readable log entry. Last readable position %d, read ahead buffer content: %s.", version, logPosition.getByteOffset(), LogTailScanner.dumpBufferToString(byteBuffer)));
                }
            }
        }
        finally {
            channel.position(initialPosition);
        }
    }

    LogTailInformation checkpointTailInformation(long highestLogVersion, LogEntryStart latestStartEntry, long oldestVersionFound, byte latestLogEntryVersion, CheckPoint latestCheckPoint, boolean corruptedTransactionLogs, StoreId storeId) throws IOException {
        LogPosition checkPointLogPosition = latestCheckPoint.getLogPosition();
        ExtractedTransactionRecord transactionRecord = this.extractFirstTxIdAfterPosition(checkPointLogPosition, highestLogVersion);
        long firstTxIdAfterPosition = transactionRecord.getId();
        boolean startRecordAfterCheckpoint = firstTxIdAfterPosition != -1L || latestStartEntry != null && latestStartEntry.getStartPosition().compareTo(latestCheckPoint.getLogPosition()) >= 0;
        boolean corruptedLogs = transactionRecord.isFailure() || corruptedTransactionLogs;
        return new LogTailInformation(latestCheckPoint, corruptedLogs || startRecordAfterCheckpoint, firstTxIdAfterPosition, oldestVersionFound, highestLogVersion, latestLogEntryVersion, storeId);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    protected ExtractedTransactionRecord extractFirstTxIdAfterPosition(LogPosition initialPosition, long maxLogVersion) throws IOException {
        long initialVersion;
        long logVersion = initialVersion = initialPosition.getLogVersion();
        while (logVersion <= maxLogVersion) {
            if (this.logFiles.versionExists(logVersion)) {
                LogPosition currentPosition = logVersion != initialVersion ? this.logFiles.extractHeader(logVersion).getStartPosition() : initialPosition;
                try (PhysicalLogVersionedStoreChannel storeChannel = this.logFiles.openForVersion(logVersion);){
                    storeChannel.position(currentPosition.getByteOffset());
                    try (LogEntryCursor cursor = new LogEntryCursor(this.logEntryReader, (ReadableClosablePositionAwareChecksumChannel)new ReadAheadLogChannel(storeChannel, this.memoryTracker));){
                        while (true) {
                            if (cursor.next()) {
                                LogEntry entry = cursor.get();
                                if (!(entry instanceof LogEntryCommit)) continue;
                                ExtractedTransactionRecord extractedTransactionRecord = new ExtractedTransactionRecord(((LogEntryCommit)entry).getTxId());
                                return extractedTransactionRecord;
                                continue;
                            }
                            break;
                        }
                    }
                }
                catch (Throwable t) {
                    this.monitor.corruptedLogFile(currentPosition.getLogVersion(), t);
                    return new ExtractedTransactionRecord(true);
                }
                logVersion = currentPosition.getLogVersion() + 1L;
                continue;
            }
            ++logVersion;
        }
        return new ExtractedTransactionRecord();
    }

    public LogTailInformation getTailInformation() throws UnderlyingStorageException {
        if (this.logTailInformation == null) {
            try {
                this.logTailInformation = this.findLogTail();
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Error encountered while parsing transaction logs", (Throwable)e);
            }
        }
        return this.logTailInformation;
    }

    private static String dumpBufferToString(ByteBuffer byteBuffer) {
        byte[] data = new byte[byteBuffer.limit()];
        byteBuffer.get(data);
        return Arrays.toString(data);
    }

    private static boolean isAllZerosBuffer(ByteBuffer byteBuffer) {
        if (byteBuffer.hasArray()) {
            byte[] array;
            for (byte b : array = byteBuffer.array()) {
                if (b == 0) continue;
                return false;
            }
        } else {
            while (byteBuffer.hasRemaining()) {
                if (byteBuffer.get() == 0) continue;
                return false;
            }
        }
        return true;
    }

    public static class LogTailInformation {
        public final CheckPoint lastCheckPoint;
        public final long firstTxIdAfterLastCheckPoint;
        public final long oldestLogVersionFound;
        public final long currentLogVersion;
        public final byte latestLogEntryVersion;
        private final boolean recordAfterCheckpoint;
        public final StoreId lastStoreId;

        public LogTailInformation(boolean recordAfterCheckpoint, long firstTxIdAfterLastCheckPoint, long oldestLogVersionFound, long currentLogVersion, byte latestLogEntryVersion) {
            this(null, recordAfterCheckpoint, firstTxIdAfterLastCheckPoint, oldestLogVersionFound, currentLogVersion, latestLogEntryVersion, StoreId.UNKNOWN);
        }

        LogTailInformation(CheckPoint lastCheckPoint, boolean recordAfterCheckpoint, long firstTxIdAfterLastCheckPoint, long oldestLogVersionFound, long currentLogVersion, byte latestLogEntryVersion, StoreId lastStoreId) {
            this.lastCheckPoint = lastCheckPoint;
            this.firstTxIdAfterLastCheckPoint = firstTxIdAfterLastCheckPoint;
            this.oldestLogVersionFound = oldestLogVersionFound;
            this.currentLogVersion = currentLogVersion;
            this.latestLogEntryVersion = latestLogEntryVersion;
            this.recordAfterCheckpoint = recordAfterCheckpoint;
            this.lastStoreId = lastStoreId;
        }

        public boolean commitsAfterLastCheckpoint() {
            return this.recordAfterCheckpoint;
        }

        public boolean logsMissing() {
            return this.lastCheckPoint == null && this.oldestLogVersionFound == -1L;
        }

        public boolean isRecoveryRequired() {
            return this.recordAfterCheckpoint || this.logsMissing();
        }

        public String toString() {
            return "LogTailInformation{lastCheckPoint=" + this.lastCheckPoint + ", firstTxIdAfterLastCheckPoint=" + this.firstTxIdAfterLastCheckPoint + ", oldestLogVersionFound=" + this.oldestLogVersionFound + ", currentLogVersion=" + this.currentLogVersion + ", latestLogEntryVersion=" + this.latestLogEntryVersion + ", recordAfterCheckpoint=" + this.recordAfterCheckpoint + "}";
        }
    }

    static class ExtractedTransactionRecord {
        private final long id;
        private final boolean failure;

        ExtractedTransactionRecord() {
            this(-1L, false);
        }

        ExtractedTransactionRecord(long txId) {
            this(txId, false);
        }

        ExtractedTransactionRecord(boolean failure) {
            this(-1L, failure);
        }

        private ExtractedTransactionRecord(long txId, boolean failure) {
            this.id = txId;
            this.failure = failure;
        }

        public long getId() {
            return this.id;
        }

        public boolean isFailure() {
            return this.failure;
        }
    }
}

