/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.store;

import java.io.File;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.ArrayUtil;
import org.neo4j.helpers.Exceptions;
import org.neo4j.helpers.collection.Visitor;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.PageCacheOpenOptions;
import org.neo4j.io.pagecache.PageCursor;
import org.neo4j.io.pagecache.PagedFile;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.store.IntStoreHeader;
import org.neo4j.kernel.impl.store.InvalidIdGeneratorException;
import org.neo4j.kernel.impl.store.InvalidRecordException;
import org.neo4j.kernel.impl.store.RecordPageLocationCalculator;
import org.neo4j.kernel.impl.store.RecordStore;
import org.neo4j.kernel.impl.store.StoreHeader;
import org.neo4j.kernel.impl.store.StoreHeaderFormat;
import org.neo4j.kernel.impl.store.StoreNotFoundException;
import org.neo4j.kernel.impl.store.UnderlyingStorageException;
import org.neo4j.kernel.impl.store.format.RecordFormat;
import org.neo4j.kernel.impl.store.id.IdGenerator;
import org.neo4j.kernel.impl.store.id.IdGeneratorFactory;
import org.neo4j.kernel.impl.store.id.IdRange;
import org.neo4j.kernel.impl.store.id.IdSequence;
import org.neo4j.kernel.impl.store.id.IdType;
import org.neo4j.kernel.impl.store.id.validation.IdValidator;
import org.neo4j.kernel.impl.store.record.AbstractBaseRecord;
import org.neo4j.kernel.impl.store.record.Record;
import org.neo4j.kernel.impl.store.record.RecordLoad;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.Logger;

public abstract class CommonAbstractStore<RECORD extends AbstractBaseRecord, HEADER extends StoreHeader>
implements RecordStore<RECORD>,
AutoCloseable {
    static final String UNKNOWN_VERSION = "Unknown";
    protected final Config configuration;
    protected final PageCache pageCache;
    protected final IdType idType;
    protected final IdGeneratorFactory idGeneratorFactory;
    protected final Log log;
    protected final String storeVersion;
    protected final RecordFormat<RECORD> recordFormat;
    final File storageFile;
    private final File idFile;
    private final String typeDescriptor;
    protected PagedFile pagedFile;
    protected int recordSize;
    private IdGenerator idGenerator;
    private boolean storeOk = true;
    private RuntimeException causeOfStoreNotOk;
    private final StoreHeaderFormat<HEADER> storeHeaderFormat;
    private HEADER storeHeader;
    private final OpenOption[] openOptions;

    public CommonAbstractStore(File file, File idFile, Config configuration, IdType idType, IdGeneratorFactory idGeneratorFactory, PageCache pageCache, LogProvider logProvider, String typeDescriptor, RecordFormat<RECORD> recordFormat, StoreHeaderFormat<HEADER> storeHeaderFormat, String storeVersion, OpenOption ... openOptions) {
        this.storageFile = file;
        this.idFile = idFile;
        this.configuration = configuration;
        this.idGeneratorFactory = idGeneratorFactory;
        this.pageCache = pageCache;
        this.idType = idType;
        this.typeDescriptor = typeDescriptor;
        this.recordFormat = recordFormat;
        this.storeHeaderFormat = storeHeaderFormat;
        this.storeVersion = storeVersion;
        this.openOptions = openOptions;
        this.log = logProvider.getLog(this.getClass());
    }

    void initialise(boolean createIfNotExists) {
        try {
            this.checkAndLoadStorage(createIfNotExists);
        }
        catch (Exception e) {
            this.closeAndThrow(e);
        }
    }

    private void closeAndThrow(Exception e) {
        if (this.pagedFile != null) {
            try {
                this.closeStoreFile();
            }
            catch (IOException failureToClose) {
                e.addSuppressed(failureToClose);
            }
        }
        Exceptions.throwIfUnchecked((Throwable)e);
        throw new RuntimeException(e);
    }

    public String getTypeDescriptor() {
        return this.typeDescriptor;
    }

    protected void checkAndLoadStorage(boolean createIfNotExists) {
        int filePageSize;
        int pageSize = this.pageCache.pageSize();
        try (PagedFile pagedFile = this.pageCache.map(this.storageFile, pageSize, new OpenOption[]{PageCacheOpenOptions.ANY_PAGE_SIZE});){
            this.extractHeaderRecord(pagedFile);
            filePageSize = this.pageCache.pageSize() - this.pageCache.pageSize() % this.getRecordSize();
        }
        catch (NoSuchFileException | StoreNotFoundException e) {
            if (createIfNotExists) {
                try {
                    this.createStore(pageSize);
                    return;
                }
                catch (IOException e1) {
                    e.addSuppressed(e1);
                }
            }
            if (e instanceof StoreNotFoundException) {
                throw (StoreNotFoundException)e;
            }
            throw new StoreNotFoundException("Store file not found: " + this.storageFile, e);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to open store file: " + this.storageFile, e);
        }
        this.loadStorage(filePageSize);
    }

    private void createStore(int pageSize) throws IOException {
        try (PagedFile file = this.pageCache.map(this.storageFile, pageSize, new OpenOption[]{StandardOpenOption.CREATE});){
            this.initialiseNewStoreFile(file);
        }
        this.checkAndLoadStorage(false);
    }

    private void loadStorage(int filePageSize) {
        try {
            this.pagedFile = this.pageCache.map(this.storageFile, filePageSize, this.openOptions);
            this.loadIdGenerator();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to open store file: " + this.storageFile, e);
        }
    }

    protected void initialiseNewStoreFile(PagedFile file) throws IOException {
        if (this.getNumberOfReservedLowIds() > 0) {
            try (PageCursor pageCursor = file.io(0L, 2);){
                if (pageCursor.next()) {
                    pageCursor.setOffset(0);
                    this.createHeaderRecord(pageCursor);
                    if (pageCursor.checkAndClearBoundsFlag()) {
                        throw new UnderlyingStorageException("Out of page bounds when writing header; page size too small: " + this.pageCache.pageSize() + " bytes.");
                    }
                }
            }
        }
        this.recordSize = this.determineRecordSize();
        this.idGeneratorFactory.create(this.idFile, this.getNumberOfReservedLowIds(), false);
    }

    private void createHeaderRecord(PageCursor cursor) {
        int offset = cursor.getOffset();
        this.storeHeaderFormat.writeHeader(cursor);
        cursor.setOffset(offset);
        this.readHeaderAndInitializeRecordFormat(cursor);
    }

    private void extractHeaderRecord(PagedFile pagedFile) throws IOException {
        block16: {
            if (this.getNumberOfReservedLowIds() > 0) {
                try (PageCursor pageCursor = pagedFile.io(0L, 1);){
                    if (pageCursor.next()) {
                        do {
                            pageCursor.setOffset(0);
                            this.readHeaderAndInitializeRecordFormat(pageCursor);
                        } while (pageCursor.shouldRetry());
                        if (pageCursor.checkAndClearBoundsFlag()) {
                            throw new UnderlyingStorageException("Out of page bounds when reading header; page size too small: " + this.pageCache.pageSize() + " bytes.");
                        }
                        break block16;
                    }
                    throw new StoreNotFoundException("Fail to read header record of store file: " + this.storageFile);
                }
            }
            this.readHeaderAndInitializeRecordFormat(null);
        }
        this.recordSize = this.determineRecordSize();
    }

    protected long pageIdForRecord(long id) {
        return RecordPageLocationCalculator.pageIdForRecord(id, this.pagedFile.pageSize(), this.recordSize);
    }

    protected int offsetForId(long id) {
        return RecordPageLocationCalculator.offsetForId(id, this.pagedFile.pageSize(), this.recordSize);
    }

    @Override
    public int getRecordsPerPage() {
        return this.pagedFile.pageSize() / this.recordSize;
    }

    public byte[] getRawRecordData(long id) throws IOException {
        byte[] data = new byte[this.recordSize];
        long pageId = this.pageIdForRecord(id);
        int offset = this.offsetForId(id);
        try (PageCursor cursor = this.pagedFile.io(pageId, 1);){
            if (cursor.next()) {
                cursor.setOffset(offset);
                cursor.mark();
                do {
                    cursor.setOffsetToMark();
                    cursor.getBytes(data);
                } while (cursor.shouldRetry());
                this.checkForDecodingErrors(cursor, id, RecordLoad.CHECK);
            }
        }
        return data;
    }

    private void readHeaderAndInitializeRecordFormat(PageCursor cursor) {
        this.storeHeader = this.storeHeaderFormat.readHeader(cursor);
    }

    private void loadIdGenerator() {
        try {
            if (this.storeOk) {
                this.openIdGenerator();
            }
        }
        catch (InvalidIdGeneratorException e) {
            this.setStoreNotOk(e);
        }
        finally {
            if (!this.getStoreOk()) {
                this.log.debug(this.storageFile + " non clean shutdown detected");
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean isInUse(long id) {
        long pageId = this.pageIdForRecord(id);
        int offset = this.offsetForId(id);
        try (PageCursor cursor = this.pagedFile.io(pageId, 1);){
            boolean recordIsInUse = false;
            if (cursor.next()) {
                cursor.setOffset(offset);
                cursor.mark();
                do {
                    cursor.setOffsetToMark();
                    recordIsInUse = this.isInUse(cursor);
                } while (cursor.shouldRetry());
                this.checkForDecodingErrors(cursor, id, RecordLoad.NORMAL);
            }
            boolean bl = recordIsInUse;
            return bl;
        }
        catch (IOException e) {
            throw new UnderlyingStorageException(e);
        }
    }

    @Override
    public PageCursor openPageCursorForReading(long id) {
        try {
            long pageId = this.pageIdForRecord(id);
            return this.pagedFile.io(pageId, 1);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException(e);
        }
    }

    final void rebuildIdGenerator() {
        boolean fastRebuild;
        long defraggedCount;
        block16: {
            int blockSize = this.getRecordSize();
            if (blockSize <= 0) {
                throw new InvalidRecordException("Illegal blockSize: " + blockSize);
            }
            this.log.info("Rebuilding id generator for[" + this.getStorageFile() + "] ...");
            this.closeIdGenerator();
            this.createIdGenerator(this.idFile);
            this.openIdGenerator();
            defraggedCount = 0L;
            fastRebuild = this.isOnlyFastIdGeneratorRebuildEnabled(this.configuration);
            try {
                long foundHighId = this.scanForHighId();
                this.setHighId(foundHighId);
                if (fastRebuild) break block16;
                try (PageCursor cursor = this.pagedFile.io(0L, 10);){
                    defraggedCount = this.rebuildIdGeneratorSlow(cursor, this.getRecordsPerPage(), blockSize, foundHighId);
                }
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Unable to rebuild id generator " + this.getStorageFile(), e);
            }
        }
        this.log.info("[" + this.getStorageFile() + "] high id=" + this.getHighId() + " (defragged=" + defraggedCount + ")");
        this.log.info(this.getStorageFile() + " rebuild id generator, highId=" + this.getHighId() + " defragged count=" + defraggedCount);
        if (!fastRebuild) {
            this.closeIdGenerator();
            this.openIdGenerator();
        }
    }

    protected boolean isOnlyFastIdGeneratorRebuildEnabled(Config config) {
        return config.get(GraphDatabaseSettings.rebuild_idgenerators_fast);
    }

    private long rebuildIdGeneratorSlow(PageCursor cursor, int recordsPerPage, int blockSize, long foundHighId) throws IOException {
        if (!cursor.isWriteLocked()) {
            throw new IllegalArgumentException("The store scanning id generator rebuild process requires a page cursor that is write-locked");
        }
        long defragCount = 0L;
        long[] freedBatch = new long[recordsPerPage];
        int startingId = this.getNumberOfReservedLowIds();
        boolean done = false;
        while (!done && cursor.next()) {
            int i;
            long idPageOffset = cursor.getCurrentPageId() * (long)recordsPerPage;
            int defragged = 0;
            for (i = startingId; i < recordsPerPage; ++i) {
                int offset = i * blockSize;
                cursor.setOffset(offset);
                long recordId = idPageOffset + (long)i;
                if (recordId >= foundHighId) {
                    done = true;
                    break;
                }
                if (!this.isInUse(cursor)) {
                    freedBatch[defragged++] = recordId;
                    continue;
                }
                if (!this.isRecordReserved(cursor)) continue;
                cursor.setOffset(offset);
                cursor.putByte(Record.NOT_IN_USE.byteValue());
                cursor.putInt(0);
                freedBatch[defragged++] = recordId;
            }
            this.checkIdScanCursorBounds(cursor);
            for (i = 0; i < defragged; ++i) {
                this.freeId(freedBatch[i]);
            }
            defragCount += (long)defragged;
            startingId = 0;
        }
        return defragCount;
    }

    private void checkIdScanCursorBounds(PageCursor cursor) {
        if (cursor.checkAndClearBoundsFlag()) {
            throw new UnderlyingStorageException("Out of bounds access on page " + cursor.getCurrentPageId() + " detected while scanning the " + this.storageFile + " file for deleted records");
        }
    }

    void setStoreNotOk(RuntimeException cause) {
        this.storeOk = false;
        this.causeOfStoreNotOk = cause;
        this.idGenerator = null;
    }

    boolean getStoreOk() {
        return this.storeOk;
    }

    void checkStoreOk() {
        if (!this.storeOk) {
            throw this.causeOfStoreNotOk;
        }
    }

    @Override
    public long nextId() {
        this.assertIdGeneratorInitialized();
        return this.idGenerator.nextId();
    }

    private void assertIdGeneratorInitialized() {
        if (this.idGenerator == null) {
            throw new IllegalStateException("IdGenerator is not initialized");
        }
    }

    @Override
    public IdRange nextIdBatch(int size) {
        this.assertIdGeneratorInitialized();
        return this.idGenerator.nextIdBatch(size);
    }

    @Override
    public void freeId(long id) {
        IdGenerator generator = this.idGenerator;
        if (generator != null) {
            generator.freeId(id);
        }
    }

    @Override
    public long getHighId() {
        return this.idGenerator != null ? this.idGenerator.getHighId() : this.scanForHighId();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setHighId(long highId) {
        IdGenerator generator = this.idGenerator;
        if (generator != null) {
            IdGenerator idGenerator = generator;
            synchronized (idGenerator) {
                if (highId > generator.getHighId()) {
                    generator.setHighId(highId);
                }
            }
        }
    }

    void makeStoreOk() {
        if (!this.storeOk) {
            this.rebuildIdGenerator();
            this.storeOk = true;
            this.causeOfStoreNotOk = null;
        }
    }

    @Override
    public File getStorageFile() {
        return this.storageFile;
    }

    void openIdGenerator() {
        this.idGenerator = this.idGeneratorFactory.open(this.idFile, this.getIdType(), this::scanForHighId, this.recordFormat.getMaxId());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected long scanForHighId() {
        try (PageCursor cursor = this.pagedFile.io(0L, 1);){
            int recordsPerPage = this.getRecordsPerPage();
            int recordSize = this.getRecordSize();
            long highestId = this.getNumberOfReservedLowIds();
            long chunkSizeInPages = 256L;
            long chunkEndId = this.pagedFile.getLastPageId();
            block14: while (true) {
                if (chunkEndId < 0L) {
                    long l = this.getNumberOfReservedLowIds();
                    return l;
                }
                long chunkStartId = Math.max(chunkEndId - 256L, 0L);
                CommonAbstractStore.preFetchChunk(cursor, chunkStartId, chunkEndId);
                long currentId = chunkEndId;
                while (true) {
                    boolean found;
                    if (currentId < chunkStartId || !cursor.next(currentId)) {
                        chunkEndId = chunkStartId - 1L;
                        continue block14;
                    }
                    block16: do {
                        found = false;
                        for (int offset = recordsPerPage * recordSize - recordSize; offset >= 0; offset -= recordSize) {
                            cursor.setOffset(offset);
                            if (!this.isInUse(cursor)) continue;
                            highestId = cursor.getCurrentPageId() * (long)recordsPerPage + (long)(offset / recordSize) + 1L;
                            found = true;
                            continue block16;
                        }
                    } while (cursor.shouldRetry());
                    this.checkIdScanCursorBounds(cursor);
                    if (found) {
                        long l = highestId;
                        return l;
                    }
                    --currentId;
                }
                break;
            }
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to find high id by scanning backwards " + this.getStorageFile(), e);
        }
    }

    private static void preFetchChunk(PageCursor cursor, long pageIdStart, long pageIdEnd) throws IOException {
        for (long currentPageId = pageIdStart; currentPageId <= pageIdEnd; ++currentPageId) {
            cursor.next(currentPageId);
        }
    }

    protected int determineRecordSize() {
        return this.recordFormat.getRecordSize((StoreHeader)this.storeHeader);
    }

    @Override
    public final int getRecordSize() {
        return this.recordSize;
    }

    @Override
    public int getRecordDataSize() {
        return this.recordSize - this.recordFormat.getRecordHeaderSize();
    }

    private boolean isInUse(PageCursor cursor) {
        return this.recordFormat.isInUse(cursor);
    }

    protected boolean isRecordReserved(PageCursor cursor) {
        return false;
    }

    private void createIdGenerator(File fileName) {
        this.idGeneratorFactory.create(fileName, 0L, false);
    }

    void closeIdGenerator() {
        if (this.idGenerator != null) {
            this.idGenerator.close();
        }
    }

    @Override
    public void flush() {
        try {
            this.pagedFile.flushAndForce();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Failed to flush", e);
        }
    }

    void assertNotClosed() {
        if (this.pagedFile == null) {
            throw new IllegalStateException(this + " for file '" + this.storageFile + "' is closed");
        }
    }

    @Override
    public void close() {
        try {
            this.closeStoreFile();
        }
        catch (IOException | IllegalStateException e) {
            throw new UnderlyingStorageException("Failed to close store file: " + this.getStorageFile(), e);
        }
    }

    private void closeStoreFile() throws IOException {
        try {
            if (this.pagedFile != null) {
                this.pagedFile.close();
            }
            if (this.idGenerator != null) {
                if (ArrayUtil.contains(this.openOptions, StandardOpenOption.DELETE_ON_CLOSE)) {
                    this.idGenerator.delete();
                } else {
                    this.idGenerator.close();
                }
            }
        }
        finally {
            this.pagedFile = null;
        }
    }

    @Override
    public long getHighestPossibleIdInUse() {
        return this.idGenerator != null ? this.idGenerator.getHighestPossibleIdInUse() : this.scanForHighId() - 1L;
    }

    @Override
    public void setHighestPossibleIdInUse(long highId) {
        this.setHighId(highId + 1L);
    }

    public long getNumberOfIdsInUse() {
        this.assertIdGeneratorInitialized();
        return this.idGenerator.getNumberOfIdsInUse();
    }

    @Override
    public int getNumberOfReservedLowIds() {
        return this.storeHeaderFormat.numberOfReservedRecords();
    }

    public IdType getIdType() {
        return this.idType;
    }

    void logVersions(Logger logger) {
        logger.log(String.format("  %s[%s] %s", this.getTypeDescriptor(), this.getStorageFile().getName(), this.storeVersion));
    }

    void logIdUsage(Logger logger) {
        logger.log(String.format("  %s[%s]: used=%s high=%s", this.getTypeDescriptor(), this.getStorageFile().getName(), this.getNumberOfIdsInUse(), this.getHighestPossibleIdInUse()));
    }

    void visitStore(Visitor<CommonAbstractStore<RECORD, HEADER>, RuntimeException> visitor) {
        visitor.visit((Object)this);
    }

    final void deleteIdGenerator() {
        if (this.idGenerator != null) {
            this.idGenerator.delete();
            this.idGenerator = null;
            this.setStoreNotOk(new IllegalStateException("IdGenerator is not initialized"));
        }
    }

    @Override
    public long getNextRecordReference(RECORD record) {
        return this.recordFormat.getNextRecordReference(record);
    }

    @Override
    public RECORD newRecord() {
        return this.recordFormat.newRecord();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public RECORD getRecord(long id, RECORD record, RecordLoad mode) {
        try (PageCursor cursor = this.pagedFile.io((long)this.getNumberOfReservedLowIds(), 1);){
            this.readIntoRecord(id, record, mode, cursor);
            RECORD RECORD = record;
            return RECORD;
        }
        catch (IOException e) {
            throw new UnderlyingStorageException(e);
        }
    }

    @Override
    public void getRecordByCursor(long id, RECORD record, RecordLoad mode, PageCursor cursor) throws UnderlyingStorageException {
        try {
            this.readIntoRecord(id, record, mode, cursor);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException(e);
        }
    }

    private void readIntoRecord(long id, RECORD record, RecordLoad mode, PageCursor cursor) throws IOException {
        ((AbstractBaseRecord)record).setId(id);
        long pageId = this.pageIdForRecord(id);
        int offset = this.offsetForId(id);
        if (cursor.next(pageId)) {
            cursor.setOffset(offset);
            this.readRecordFromPage(id, record, mode, cursor);
        } else {
            this.verifyAfterNotRead(record, mode);
        }
    }

    @Override
    public void nextRecordByCursor(RECORD record, RecordLoad mode, PageCursor cursor) throws UnderlyingStorageException {
        if (cursor.getCurrentPageId() < -1L) {
            throw new IllegalArgumentException("Pages are assumed to be positive or -1 if not initialized");
        }
        try {
            int offset = cursor.getOffset();
            long id = ((AbstractBaseRecord)record).getId() + 1L;
            ((AbstractBaseRecord)record).setId(id);
            long pageId = cursor.getCurrentPageId();
            if (offset >= this.pagedFile.pageSize() || pageId < 0L) {
                if (!cursor.next()) {
                    this.verifyAfterNotRead(record, mode);
                    return;
                }
                cursor.setOffset(0);
            }
            this.readRecordFromPage(id, record, mode, cursor);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException(e);
        }
    }

    private void readRecordFromPage(long id, RECORD record, RecordLoad mode, PageCursor cursor) throws IOException {
        cursor.mark();
        do {
            this.prepareForReading(cursor, record);
            this.recordFormat.read(record, cursor, mode, this.recordSize);
        } while (cursor.shouldRetry());
        this.checkForDecodingErrors(cursor, id, mode);
        this.verifyAfterReading(record, mode);
    }

    @Override
    public void updateRecord(RECORD record) {
        long id = ((AbstractBaseRecord)record).getId();
        IdValidator.assertValidId(this.getIdType(), id, this.recordFormat.getMaxId());
        long pageId = this.pageIdForRecord(id);
        int offset = this.offsetForId(id);
        try (PageCursor cursor = this.pagedFile.io(pageId, 2);){
            if (cursor.next()) {
                cursor.setOffset(offset);
                this.recordFormat.write(record, cursor, this.recordSize);
                this.checkForDecodingErrors(cursor, id, RecordLoad.NORMAL);
                if (!((AbstractBaseRecord)record).inUse()) {
                    this.freeId(id);
                }
                if (!(((AbstractBaseRecord)record).inUse() && ((AbstractBaseRecord)record).requiresSecondaryUnit() || !((AbstractBaseRecord)record).hasSecondaryUnitId())) {
                    this.freeId(((AbstractBaseRecord)record).getSecondaryUnitId());
                }
            }
        }
        catch (IOException e) {
            throw new UnderlyingStorageException(e);
        }
    }

    @Override
    public void prepareForCommit(RECORD record) {
        this.prepareForCommit(record, this);
    }

    @Override
    public void prepareForCommit(RECORD record, IdSequence idSequence) {
        if (((AbstractBaseRecord)record).inUse()) {
            this.recordFormat.prepare(record, this.recordSize, idSequence);
        }
    }

    @Override
    public <EXCEPTION extends Exception> void scanAllRecords(Visitor<RECORD, EXCEPTION> visitor) throws EXCEPTION {
        try (PageCursor cursor = this.openPageCursorForReading(0L);){
            RECORD record = this.newRecord();
            long highId = this.getHighId();
            for (long id = (long)this.getNumberOfReservedLowIds(); id < highId; ++id) {
                this.getRecordByCursor(id, record, RecordLoad.CHECK, cursor);
                if (!((AbstractBaseRecord)record).inUse()) continue;
                visitor.visit(record);
            }
        }
    }

    @Override
    public List<RECORD> getRecords(long firstId, RecordLoad mode) {
        if (Record.NULL_REFERENCE.is(firstId)) {
            return Collections.emptyList();
        }
        ArrayList<RECORD> records = new ArrayList<RECORD>();
        long id = firstId;
        try (PageCursor cursor = this.openPageCursorForReading(firstId);){
            RECORD record;
            do {
                record = this.newRecord();
                this.getRecordByCursor(id, record, mode, cursor);
                records.add(record);
            } while (!Record.NULL_REFERENCE.is(id = this.getNextRecordReference(record)));
        }
        return records;
    }

    private void verifyAfterNotRead(RECORD record, RecordLoad mode) {
        ((AbstractBaseRecord)record).clear();
        mode.verify((AbstractBaseRecord)record);
    }

    final void checkForDecodingErrors(PageCursor cursor, long recordId, RecordLoad mode) {
        if (mode.checkForOutOfBounds(cursor)) {
            this.throwOutOfBoundsException(recordId);
        }
        mode.clearOrThrowCursorError(cursor);
    }

    private void throwOutOfBoundsException(long recordId) {
        RECORD record = this.newRecord();
        ((AbstractBaseRecord)record).setId(recordId);
        long pageId = this.pageIdForRecord(recordId);
        int offset = this.offsetForId(recordId);
        throw new UnderlyingStorageException(CommonAbstractStore.buildOutOfBoundsExceptionMessage(record, pageId, offset, this.recordSize, this.pagedFile.pageSize(), this.storageFile.getAbsolutePath()));
    }

    static String buildOutOfBoundsExceptionMessage(AbstractBaseRecord record, long pageId, int offset, int recordSize, int pageSize, String filename) {
        return "Access to record " + record + " went out of bounds of the page. The record size is " + recordSize + " bytes, and the access was at offset " + offset + " bytes into page " + pageId + ", and the pages have a capacity of " + pageSize + " bytes. The mapped store file in question is " + filename;
    }

    private void verifyAfterReading(RECORD record, RecordLoad mode) {
        if (!mode.verify((AbstractBaseRecord)record)) {
            ((AbstractBaseRecord)record).clear();
        }
    }

    private void prepareForReading(PageCursor cursor, RECORD record) {
        ((AbstractBaseRecord)record).setInUse(false);
        cursor.setOffsetToMark();
    }

    @Override
    public void ensureHeavy(RECORD record) {
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }

    @Override
    public int getStoreHeaderInt() {
        return ((IntStoreHeader)this.storeHeader).value();
    }
}

