/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.store.id;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.atomic.AtomicLong;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.kernel.impl.store.InvalidIdGeneratorException;
import org.neo4j.kernel.impl.store.UnderlyingStorageException;
import org.neo4j.kernel.impl.store.id.FreeIdKeeper;
import org.neo4j.kernel.impl.store.id.IdGenerator;
import org.neo4j.kernel.impl.store.id.IdRange;
import org.neo4j.kernel.impl.store.id.validation.IdValidator;

public class IdGeneratorImpl
implements IdGenerator {
    public static final int HEADER_SIZE = 9;
    private static final byte CLEAN_GENERATOR = 0;
    private static final byte STICKY_GENERATOR = 1;
    public static final long INTEGER_MINUS_ONE = 0xFFFFFFFFL;
    private int grabSize = -1;
    private final AtomicLong highId = new AtomicLong(-1L);
    private final File file;
    private final FileSystemAbstraction fs;
    private StoreChannel fileChannel;
    private final long max;
    private final boolean aggressiveReuse;
    private FreeIdKeeper keeper;

    public IdGeneratorImpl(FileSystemAbstraction fs, File file, int grabSize, long max, boolean aggressiveReuse, long highId) {
        this.fs = fs;
        this.aggressiveReuse = aggressiveReuse;
        if (grabSize < 1) {
            throw new IllegalArgumentException("Illegal grabSize: " + grabSize);
        }
        this.max = max;
        this.file = file;
        this.grabSize = grabSize;
        this.initGenerator();
        this.highId.set(Math.max(this.highId.get(), highId));
    }

    @Override
    public synchronized long nextId() {
        this.assertStillOpen();
        long nextDefragId = this.keeper.getId();
        if (nextDefragId != -1L) {
            return nextDefragId;
        }
        long id = this.highId.get();
        if (IdValidator.isReservedId(id)) {
            id = this.highId.incrementAndGet();
        }
        IdValidator.assertValidId(id, this.max);
        this.highId.incrementAndGet();
        return id;
    }

    private void assertStillOpen() {
        if (this.fileChannel == null) {
            throw new IllegalStateException("Closed id generator " + this.file);
        }
    }

    @Override
    public synchronized IdRange nextIdBatch(int size) {
        long id;
        this.assertStillOpen();
        int count = 0;
        long[] defragIds = new long[size];
        while (count < size && (id = this.keeper.getId()) != -1L) {
            defragIds[count++] = id;
        }
        long[] tmpArray = defragIds;
        defragIds = new long[count];
        System.arraycopy(tmpArray, 0, defragIds, 0, count);
        int sizeLeftForRange = size - count;
        long start = this.highId.get();
        this.setHighId(start + (long)sizeLeftForRange);
        return new IdRange(defragIds, start, sizeLeftForRange);
    }

    @Override
    public void setHighId(long id) {
        IdValidator.assertIdWithinCapacity(id, this.max);
        this.highId.set(id);
    }

    @Override
    public long getHighId() {
        return this.highId.get();
    }

    @Override
    public long getHighestPossibleIdInUse() {
        return this.getHighId() - 1L;
    }

    @Override
    public synchronized void freeId(long id) {
        if (IdValidator.isReservedId(id)) {
            return;
        }
        if (this.fileChannel == null) {
            throw new IllegalStateException("Generator closed " + this.file);
        }
        if (id < 0L || id >= this.highId.get()) {
            throw new IllegalArgumentException("Illegal id[" + id + "], highId is " + this.highId.get());
        }
        this.keeper.freeId(id);
    }

    @Override
    public synchronized void close() {
        if (this.isClosed()) {
            return;
        }
        try {
            this.keeper.close();
            ByteBuffer buffer = ByteBuffer.allocate(9);
            this.writeHeader(buffer);
            this.fileChannel.force(false);
            this.markAsCleanlyClosed(buffer);
            this.closeChannel();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to close id generator " + this.file, e);
        }
    }

    private boolean isClosed() {
        return this.highId.get() == -1L;
    }

    private void closeChannel() throws IOException {
        this.fileChannel.force(false);
        this.fileChannel.close();
        this.fileChannel = null;
        this.highId.set(-1L);
    }

    private void markAsCleanlyClosed(ByteBuffer buffer) throws IOException {
        buffer.clear();
        buffer.put((byte)0);
        buffer.limit(1);
        buffer.flip();
        this.fileChannel.position(0L);
        this.fileChannel.write(buffer);
    }

    private void writeHeader(ByteBuffer buffer) throws IOException {
        this.fileChannel.position(0L);
        buffer.put((byte)1).putLong(this.highId.get());
        buffer.flip();
        this.fileChannel.write(buffer);
    }

    public static void createGenerator(FileSystemAbstraction fs, File fileName, long highId, boolean throwIfFileExists) {
        if (fs == null) {
            throw new IllegalArgumentException("Null filesystem");
        }
        if (fileName == null) {
            throw new IllegalArgumentException("Null filename");
        }
        if (throwIfFileExists && fs.fileExists(fileName)) {
            throw new IllegalStateException("Can't create IdGeneratorFile[" + fileName + "], file already exists");
        }
        try (StoreChannel channel = fs.create(fileName);){
            channel.truncate(0L);
            ByteBuffer buffer = ByteBuffer.allocate(9);
            buffer.put((byte)0).putLong(highId).flip();
            channel.write(buffer);
            channel.force(false);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to create id generator" + fileName, e);
        }
    }

    private synchronized void initGenerator() {
        try {
            this.fileChannel = this.fs.open(this.file, "rw");
            ByteBuffer buffer = this.readHeader();
            IdGeneratorImpl.markAsSticky(this.fileChannel, buffer);
            this.fileChannel.position(9L);
            this.keeper = new FreeIdKeeper(this.fileChannel, this.grabSize, this.aggressiveReuse);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to init id generator " + this.file, e);
        }
    }

    public static void markAsSticky(StoreChannel fileChannel, ByteBuffer buffer) throws IOException {
        buffer.clear();
        buffer.put((byte)1).limit(1).flip();
        fileChannel.position(0L);
        fileChannel.write(buffer);
        fileChannel.force(false);
    }

    private ByteBuffer readHeader() throws IOException {
        try {
            ByteBuffer buffer = IdGeneratorImpl.readHighIdFromHeader(this.fileChannel, this.file);
            this.highId.set(buffer.getLong());
            return buffer;
        }
        catch (InvalidIdGeneratorException e) {
            this.fileChannel.close();
            throw e;
        }
    }

    private static ByteBuffer readHighIdFromHeader(StoreChannel channel, File fileName) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(9);
        int read = channel.read(buffer);
        if (read != 9) {
            throw new InvalidIdGeneratorException("Unable to read header, bytes read: " + read);
        }
        buffer.flip();
        byte storageStatus = buffer.get();
        if (storageStatus != 0) {
            throw new InvalidIdGeneratorException("Sticky generator[ " + fileName + "] delete this id file and build a new one");
        }
        return buffer;
    }

    public static long readHighId(FileSystemAbstraction fileSystem, File file) throws IOException {
        try (StoreChannel channel = fileSystem.open(file, "r");){
            long l = IdGeneratorImpl.readHighIdFromHeader(channel, file).getLong();
            return l;
        }
    }

    public synchronized void dumpFreeIds() throws IOException {
        this.keeper.dumpFreeIds();
        System.out.println("\nNext free id: " + this.highId);
        this.close();
    }

    @Override
    public synchronized long getNumberOfIdsInUse() {
        return this.highId.get() - this.keeper.getCount();
    }

    @Override
    public long getDefragCount() {
        return this.keeper.getCount();
    }

    @Override
    public synchronized void delete() {
        if (!this.isClosed()) {
            try {
                this.closeChannel();
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Unable to safe close id generator " + this.file, e);
            }
        }
        if (!this.fs.deleteFile(this.file)) {
            throw new UnderlyingStorageException("Unable to delete id generator " + this.file);
        }
    }

    public String toString() {
        return "IdGeneratorImpl " + this.hashCode() + " [highId=" + this.highId + ", defragged=" + this.keeper.getCount() + ", fileName=" + this.file + ", max=" + this.max + ", aggressive=" + this.aggressiveReuse + "]";
    }
}

