/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log.files;

import java.io.IOException;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.HashSet;
import java.util.concurrent.atomic.AtomicLong;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.io.fs.ChannelNativeAccessor;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.kernel.KernelVersion;
import org.neo4j.kernel.KernelVersionProvider;
import org.neo4j.kernel.impl.transaction.log.LogFileCreateEvent;
import org.neo4j.kernel.impl.transaction.log.LogFormatVersionProvider;
import org.neo4j.kernel.impl.transaction.log.LogHeaderCache;
import org.neo4j.kernel.impl.transaction.log.LogTracers;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.StoreChannelNativeAccessor;
import org.neo4j.kernel.impl.transaction.log.entry.IncompleteLogHeaderException;
import org.neo4j.kernel.impl.transaction.log.entry.LogFormat;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeader;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeaderReader;
import org.neo4j.kernel.impl.transaction.log.files.SequentialFilesHelper;
import org.neo4j.kernel.impl.transaction.log.files.TransactionLogFilesContext;
import org.neo4j.kernel.impl.transaction.tracing.DatabaseTracer;
import org.neo4j.logging.InternalLog;
import org.neo4j.memory.MemoryTracker;

public class TransactionLogChannelAllocator {
    private final TransactionLogFilesContext logFilesContext;
    private final FileSystemAbstraction fileSystem;
    private final SequentialFilesHelper fileHelper;
    private final LogHeaderCache logHeaderCache;
    private final ChannelNativeAccessor nativeChannelAccessor;
    private final DatabaseTracer databaseTracer;
    private final AtomicLong rotationThreshold;

    public TransactionLogChannelAllocator(TransactionLogFilesContext logFilesContext, SequentialFilesHelper fileHelper, LogHeaderCache logHeaderCache, AtomicLong rotationThreshold) {
        this.logFilesContext = logFilesContext;
        this.fileSystem = logFilesContext.getFileSystem();
        this.databaseTracer = logFilesContext.getDatabaseTracers().getDatabaseTracer();
        this.fileHelper = fileHelper;
        this.logHeaderCache = logHeaderCache;
        this.nativeChannelAccessor = new StoreChannelNativeAccessor(logFilesContext.fileSystem(), logFilesContext.getNativeAccess(), logFilesContext.getLogProvider(), (ChannelNativeAccessor.OutOfDiskHandler)new TransactionLogOutOfDiskHandler(logFilesContext));
        this.rotationThreshold = rotationThreshold;
    }

    public PhysicalLogVersionedStoreChannel createLogChannel(long version, long lastAppendIndex, int previousLogFileChecksum, KernelVersionProvider kernelVersionProvider, LogFormatVersionProvider logFormatProvider) throws IOException {
        AllocatedFile allocatedFile = this.allocateFile(version);
        StoreChannel storeChannel = allocatedFile.storeChannel();
        Path logFile = allocatedFile.path();
        LogHeader header = this.maybeInitLogHeader(version, lastAppendIndex, previousLogFileChecksum, kernelVersionProvider, logFormatProvider, storeChannel, logFile);
        assert (header.getLogVersion() == version);
        this.logHeaderCache.putHeader(version, header);
        storeChannel.position(header.getStartPosition().getByteOffset());
        return new PhysicalLogVersionedStoreChannel(storeChannel, version, header.getLogFormatVersion(), logFile, this.nativeChannelAccessor, (LogTracers)this.databaseTracer);
    }

    public void initializeLogFile(long version, long lastAppendIndex, int previousLogFileChecksum, KernelVersionProvider kernelVersionProvider, LogFormatVersionProvider logFormatVersionProvider) throws IOException {
        AllocatedFile allocatedFile = this.allocateFile(version);
        try (StoreChannel storeChannel = allocatedFile.storeChannel();){
            this.maybeInitLogHeader(version, lastAppendIndex, previousLogFileChecksum, kernelVersionProvider, logFormatVersionProvider, storeChannel, allocatedFile.path());
        }
    }

    private LogHeader maybeInitLogHeader(long version, long lastAppendIndex, int previousLogFileChecksum, KernelVersionProvider kernelVersionProvider, LogFormatVersionProvider logFormatProvider, StoreChannel storeChannel, Path logFile) throws IOException {
        LogHeader header = LogHeaderReader.readLogHeader((ReadableByteChannel)storeChannel, (boolean)false, (Path)logFile, (MemoryTracker)this.logFilesContext.getMemoryTracker());
        if (header == null) {
            try (LogFileCreateEvent createEvent = this.databaseTracer.createLogFile();){
                storeChannel.position(0L);
                KernelVersion kernelVersion = kernelVersionProvider.kernelVersion();
                header = logFormatProvider.getCurrentLogFormat().newHeader(version, lastAppendIndex, -1L, this.logFilesContext.getStoreId(), this.logFilesContext.getEnvelopeSegmentBlockSizeBytes(), previousLogFileChecksum, kernelVersion);
                LogFormat.writeLogHeader((StoreChannel)storeChannel, (LogHeader)header, (MemoryTracker)this.logFilesContext.getMemoryTracker());
                createEvent.fileCreated(header.getStartPosition().getByteOffset());
            }
        }
        return header;
    }

    public PhysicalLogVersionedStoreChannel createLogChannelExistingVersion(long version) throws IOException {
        Path logFile;
        AllocatedFile allocatedFile = this.allocateExistingFile(version);
        StoreChannel storeChannel = allocatedFile.storeChannel();
        LogHeader header = LogHeaderReader.readLogHeader((ReadableByteChannel)storeChannel, (boolean)true, (Path)(logFile = allocatedFile.path()), (MemoryTracker)this.logFilesContext.getMemoryTracker());
        if (header == null) {
            throw new IncompleteLogHeaderException(allocatedFile.path, (int)storeChannel.position(), -1);
        }
        assert (header.getLogVersion() == version);
        this.logHeaderCache.putHeader(version, header);
        storeChannel.position(header.getStartPosition().getByteOffset());
        return new PhysicalLogVersionedStoreChannel(storeChannel, version, header.getLogFormatVersion(), logFile, this.nativeChannelAccessor, (LogTracers)this.databaseTracer);
    }

    public PhysicalLogVersionedStoreChannel openLogChannel(long version) throws IOException {
        return this.openLogChannel(version, false);
    }

    public PhysicalLogVersionedStoreChannel openLogChannel(long version, boolean raw) throws IOException {
        Path fileToOpen = this.fileHelper.getFileForVersion(version);
        if (!this.fileSystem.fileExists(fileToOpen)) {
            throw new NoSuchFileException(fileToOpen.toAbsolutePath().toString());
        }
        this.databaseTracer.openLogFile(fileToOpen);
        StoreChannel rawChannel = null;
        try {
            rawChannel = this.fileSystem.read(fileToOpen);
            LogHeader header = LogHeaderReader.readLogHeader((ReadableByteChannel)rawChannel, (boolean)true, (Path)fileToOpen, (MemoryTracker)this.logFilesContext.getMemoryTracker());
            if (header == null) {
                throw new IncompleteLogHeaderException(fileToOpen, 0, 8);
            }
            if (header.getLogVersion() != version) {
                throw new IllegalStateException(String.format("Unexpected log file header. Expected header version: %d, actual header: %s", version, header));
            }
            PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel(rawChannel, version, header.getLogFormatVersion(), fileToOpen, this.nativeChannelAccessor, (LogTracers)this.databaseTracer, raw);
            if (!raw) {
                this.nativeChannelAccessor.adviseSequentialAccessAndKeepInCache(rawChannel, fileToOpen);
            }
            return versionedStoreChannel;
        }
        catch (NoSuchFileException cause) {
            throw (NoSuchFileException)new NoSuchFileException(fileToOpen.toAbsolutePath().toString()).initCause(cause);
        }
        catch (Throwable unexpectedError) {
            if (rawChannel != null) {
                try {
                    rawChannel.close();
                }
                catch (IOException e) {
                    unexpectedError.addSuppressed(e);
                }
            }
            throw unexpectedError;
        }
    }

    public LogHeader readLogHeaderForVersion(long version) throws IOException {
        Path fileToOpen = this.fileHelper.getFileForVersion(version);
        if (!this.fileSystem.fileExists(fileToOpen)) {
            throw new NoSuchFileException(fileToOpen.toAbsolutePath().toString());
        }
        try (StoreChannel read = this.fileSystem.read(fileToOpen);){
            LogHeader logHeader = LogHeaderReader.readLogHeader((ReadableByteChannel)read, (boolean)true, (Path)fileToOpen, (MemoryTracker)this.logFilesContext.getMemoryTracker());
            return logHeader;
        }
    }

    private AllocatedFile allocateFile(long version) throws IOException {
        Path file = this.fileHelper.getFileForVersion(version);
        boolean fileExist = this.fileSystem.fileExists(file);
        StoreChannel storeChannel = this.fileSystem.write(file);
        if (fileExist) {
            this.nativeChannelAccessor.adviseSequentialAccessAndKeepInCache(storeChannel, file);
        } else if (this.logFilesContext.getTryPreallocateTransactionLogs().get()) {
            this.nativeChannelAccessor.preallocateSpace(storeChannel, this.rotationThreshold.get(), file);
        }
        return new AllocatedFile(file, storeChannel);
    }

    private AllocatedFile allocateExistingFile(long version) throws IOException {
        Path file = this.fileHelper.getFileForVersion(version);
        boolean fileExist = this.fileSystem.fileExists(file);
        if (!fileExist) {
            throw new NoSuchFileException(file.toAbsolutePath().toString());
        }
        StoreChannel storeChannel = this.fileSystem.write(file);
        this.nativeChannelAccessor.adviseSequentialAccessAndKeepInCache(storeChannel, file);
        return new AllocatedFile(file, storeChannel);
    }

    private static class TransactionLogOutOfDiskHandler
    implements ChannelNativeAccessor.OutOfDiskHandler {
        private final InternalLog log;
        private final Config config;
        private final String databaseName;

        public TransactionLogOutOfDiskHandler(TransactionLogFilesContext logFilesContext) {
            this.log = logFilesContext.getLogProvider().getLog(this.getClass());
            this.config = logFilesContext.config();
            this.databaseName = logFilesContext.getDatabaseName();
        }

        public void handle(String error) {
            this.log.error("Warning! System is running out of disk space. Failed to preallocate file since disk does not have enough space left. Please provision more space to avoid that. Allocation failure details: " + error);
            if (((Boolean)this.config.get(GraphDatabaseInternalSettings.dynamic_read_only_failover)).booleanValue()) {
                this.log.error("Switching database to read only mode.");
                this.markDatabaseReadOnly();
            } else {
                this.log.error("Dynamic switchover to read-only mode is disabled. The database will continue execution in the current mode.");
            }
        }

        private void markDatabaseReadOnly() {
            HashSet<String> readOnlyDatabases = new HashSet<String>((Collection)this.config.get(GraphDatabaseSettings.read_only_databases));
            readOnlyDatabases.add(this.databaseName);
            this.config.setDynamic(GraphDatabaseSettings.read_only_databases, readOnlyDatabases, "Dynamic failover to read-only mode.");
        }
    }

    private record AllocatedFile(Path path, StoreChannel storeChannel) {
    }
}

