/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.api.index;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.helpers.BiConsumer;
import org.neo4j.helpers.Exceptions;
import org.neo4j.helpers.Pair;
import org.neo4j.helpers.collection.Iterables;
import org.neo4j.kernel.api.TokenNameLookup;
import org.neo4j.kernel.api.exceptions.index.IndexActivationFailedKernelException;
import org.neo4j.kernel.api.exceptions.index.IndexNotFoundKernelException;
import org.neo4j.kernel.api.exceptions.index.IndexPopulationFailedKernelException;
import org.neo4j.kernel.api.exceptions.schema.ConstraintVerificationFailedKernelException;
import org.neo4j.kernel.api.index.IndexDescriptor;
import org.neo4j.kernel.api.index.IndexEntryConflictException;
import org.neo4j.kernel.api.index.IndexUpdater;
import org.neo4j.kernel.api.index.InternalIndexState;
import org.neo4j.kernel.api.index.NodePropertyUpdate;
import org.neo4j.kernel.api.index.SchemaIndexProvider;
import org.neo4j.kernel.impl.api.UpdateableSchemaState;
import org.neo4j.kernel.impl.api.index.IndexMap;
import org.neo4j.kernel.impl.api.index.IndexMapReference;
import org.neo4j.kernel.impl.api.index.IndexPopulationFailure;
import org.neo4j.kernel.impl.api.index.IndexProxy;
import org.neo4j.kernel.impl.api.index.IndexProxySetup;
import org.neo4j.kernel.impl.api.index.IndexStoreView;
import org.neo4j.kernel.impl.api.index.IndexUpdateMode;
import org.neo4j.kernel.impl.api.index.IndexUpdaterMap;
import org.neo4j.kernel.impl.api.index.IndexUpdates;
import org.neo4j.kernel.impl.api.index.SchemaIndexProviderMap;
import org.neo4j.kernel.impl.api.index.sampling.IndexSamplingConfig;
import org.neo4j.kernel.impl.api.index.sampling.IndexSamplingController;
import org.neo4j.kernel.impl.api.index.sampling.IndexSamplingControllerFactory;
import org.neo4j.kernel.impl.api.index.sampling.IndexSamplingMode;
import org.neo4j.kernel.impl.store.UnderlyingStorageException;
import org.neo4j.kernel.impl.store.record.IndexRule;
import org.neo4j.kernel.impl.util.JobScheduler;
import org.neo4j.kernel.impl.util.StringLogger;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;
import org.neo4j.kernel.logging.Logging;
import org.neo4j.register.Register;
import org.neo4j.register.Registers;

public class IndexingService
extends LifecycleAdapter {
    private final IndexSamplingController samplingController;
    private final IndexProxySetup proxySetup;
    private final IndexStoreView storeView;
    private final SchemaIndexProviderMap providerMap;
    private final IndexMapReference indexMapRef;
    private final Iterable<IndexRule> indexRules;
    private final StringLogger logger;
    private final TokenNameLookup tokenNameLookup;
    private final Monitor monitor;
    private final Set<Long> recoveredNodeIds = new HashSet<Long>();
    public static final Monitor NO_MONITOR = new MonitorAdapter(){};
    private volatile State state = State.NOT_STARTED;

    protected IndexingService(IndexProxySetup proxySetup, SchemaIndexProviderMap providerMap, IndexMapReference indexMapRef, IndexStoreView storeView, Iterable<IndexRule> indexRules, IndexSamplingController samplingController, TokenNameLookup tokenNameLookup, Logging logging, Monitor monitor) {
        this.proxySetup = proxySetup;
        this.providerMap = providerMap;
        this.indexMapRef = indexMapRef;
        this.storeView = storeView;
        this.indexRules = indexRules;
        this.samplingController = samplingController;
        this.tokenNameLookup = tokenNameLookup;
        this.monitor = monitor;
        this.logger = logging.getMessagesLog(this.getClass());
    }

    public static IndexingService create(IndexSamplingConfig samplingConfig, JobScheduler scheduler, SchemaIndexProviderMap providerMap, IndexStoreView storeView, TokenNameLookup tokenNameLookup, UpdateableSchemaState updateableSchemaState, Iterable<IndexRule> indexRules, Logging logging, Monitor monitor) {
        if (providerMap == null || providerMap.getDefaultProvider() == null) {
            throw new IllegalStateException("You cannot run the database without an index provider, please make sure that a valid provider (subclass of " + SchemaIndexProvider.class.getName() + ") is on your classpath.");
        }
        IndexMapReference indexMapRef = new IndexMapReference();
        IndexSamplingControllerFactory factory = new IndexSamplingControllerFactory(samplingConfig, storeView, scheduler, tokenNameLookup, logging);
        IndexSamplingController indexSamplingController = factory.create(indexMapRef);
        IndexProxySetup proxySetup = new IndexProxySetup(samplingConfig, storeView, providerMap, updateableSchemaState, tokenNameLookup, scheduler, logging);
        return new IndexingService(proxySetup, providerMap, indexMapRef, storeView, indexRules, indexSamplingController, tokenNameLookup, logging, monitor);
    }

    @Override
    public void init() {
        IndexMap indexMap = this.indexMapRef.indexMapSnapshot();
        for (IndexRule indexRule : this.indexRules) {
            IndexProxy indexProxy;
            long indexId = indexRule.getId();
            IndexDescriptor descriptor = new IndexDescriptor(indexRule.getLabel(), indexRule.getPropertyKey());
            SchemaIndexProvider.Descriptor providerDescriptor = indexRule.getProviderDescriptor();
            SchemaIndexProvider provider = this.providerMap.apply(providerDescriptor);
            InternalIndexState initialState = provider.getInitialState(indexId);
            this.logger.info(this.proxySetup.indexStateInfo("init", indexId, initialState, descriptor));
            boolean constraint = indexRule.isConstraintIndex();
            switch (initialState) {
                case ONLINE: {
                    indexProxy = this.proxySetup.createOnlineIndexProxy(indexId, descriptor, providerDescriptor, constraint);
                    break;
                }
                case POPULATING: {
                    indexProxy = this.proxySetup.createRecoveringIndexProxy(descriptor, providerDescriptor, constraint);
                    break;
                }
                case FAILED: {
                    IndexPopulationFailure failure = IndexPopulationFailure.failure(provider.getPopulationFailure(indexId));
                    indexProxy = this.proxySetup.createFailedIndexProxy(indexId, descriptor, providerDescriptor, constraint, failure);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("" + (Object)((Object)initialState));
                }
            }
            indexMap.putIndexProxy(indexId, indexProxy);
        }
        this.indexMapRef.setIndexMap(indexMap);
    }

    @Override
    public void start() throws IOException {
        this.state = State.STARTING;
        this.applyRecoveredUpdates();
        IndexMap indexMap = this.indexMapRef.indexMapSnapshot();
        final HashMap<Long, Pair<IndexDescriptor, SchemaIndexProvider.Descriptor>> rebuildingDescriptors = new HashMap<Long, Pair<IndexDescriptor, SchemaIndexProvider.Descriptor>>();
        indexMap.foreachIndexProxy(new BiConsumer<Long, IndexProxy>(){

            @Override
            public void accept(Long indexId, IndexProxy proxy) {
                InternalIndexState state = proxy.getState();
                IndexDescriptor descriptor = proxy.getDescriptor();
                IndexingService.this.logger.info(IndexingService.this.proxySetup.indexStateInfo("start", indexId, state, descriptor));
                switch (state) {
                    case ONLINE: {
                        break;
                    }
                    case POPULATING: {
                        rebuildingDescriptors.put(indexId, Pair.of(descriptor, proxy.getProviderDescriptor()));
                        break;
                    }
                }
            }
        });
        this.dropRecoveringIndexes(indexMap, rebuildingDescriptors);
        for (Map.Entry entry : rebuildingDescriptors.entrySet()) {
            long indexId = (Long)entry.getKey();
            Pair descriptors = (Pair)entry.getValue();
            IndexDescriptor indexDescriptor = (IndexDescriptor)descriptors.first();
            SchemaIndexProvider.Descriptor providerDescriptor = (SchemaIndexProvider.Descriptor)descriptors.other();
            IndexProxy proxy = this.proxySetup.createPopulatingIndexProxy(indexId, indexDescriptor, providerDescriptor, false, this.monitor);
            proxy.start();
            indexMap.putIndexProxy(indexId, proxy);
        }
        this.indexMapRef.setIndexMap(indexMap);
        this.samplingController.recoverIndexSamples();
        this.samplingController.start();
        this.state = State.RUNNING;
    }

    @Override
    public void stop() {
        this.state = State.STOPPED;
        this.closeAllIndexes();
    }

    public Register.DoubleLongRegister indexUpdatesAndSize(long indexId) throws IndexNotFoundKernelException {
        IndexProxy indexProxy = this.indexMapRef.getOnlineIndexProxy(indexId);
        Register.DoubleLongRegister output = Registers.newDoubleLongRegister();
        this.storeView.indexUpdatesAndSize(indexProxy.getDescriptor(), output);
        return output;
    }

    public double indexUniqueValuesPercentage(long indexId) throws IndexNotFoundKernelException {
        IndexProxy indexProxy = this.indexMapRef.getOnlineIndexProxy(indexId);
        Register.DoubleLongRegister output = Registers.newDoubleLongRegister();
        this.storeView.indexSample(indexProxy.getDescriptor(), output);
        long unique = output.readFirst();
        long size = output.readSecond();
        if (size == 0L) {
            return 1.0;
        }
        return (double)unique / (double)size;
    }

    public void createIndex(IndexRule rule) {
        long ruleId;
        IndexMap indexMap = this.indexMapRef.indexMapSnapshot();
        IndexProxy index = indexMap.getIndexProxy(ruleId = rule.getId());
        if (index != null) {
            return;
        }
        IndexDescriptor descriptor = new IndexDescriptor(rule.getLabel(), rule.getPropertyKey());
        SchemaIndexProvider.Descriptor providerDescriptor = rule.getProviderDescriptor();
        boolean constraint = rule.isConstraintIndex();
        if (this.state == State.RUNNING) {
            try {
                index = this.proxySetup.createPopulatingIndexProxy(ruleId, descriptor, providerDescriptor, constraint, this.monitor);
                index.start();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else {
            index = this.proxySetup.createRecoveringIndexProxy(descriptor, providerDescriptor, constraint);
        }
        indexMap.putIndexProxy(rule.getId(), index);
        this.indexMapRef.setIndexMap(indexMap);
    }

    public void updateIndexes(IndexUpdates updates, long transactionId, boolean forceIdempotency) {
        if (this.state == State.RUNNING) {
            IndexUpdateMode mode = forceIdempotency ? IndexUpdateMode.RECOVERY : IndexUpdateMode.ONLINE;
            try (IndexUpdaterMap updaterMap = this.indexMapRef.createIndexUpdaterMap(mode);){
                this.applyUpdates(updates, updaterMap);
            }
        } else if (this.state == State.NOT_STARTED) {
            this.recoveredNodeIds.addAll(updates.changedNodeIds());
        } else {
            throw new IllegalStateException("Cannot queue index updates while index service is " + (Object)((Object)this.state));
        }
    }

    protected void applyRecoveredUpdates() throws IOException {
        this.logger.debug("Applying recovered updates: " + this.recoveredNodeIds);
        this.monitor.applyingRecoveredData(this.recoveredNodeIds);
        if (!this.recoveredNodeIds.isEmpty()) {
            try (IndexUpdaterMap updaterMap = this.indexMapRef.createIndexUpdaterMap(IndexUpdateMode.RECOVERY);){
                for (IndexUpdater updater : updaterMap) {
                    updater.remove(this.recoveredNodeIds);
                }
                Iterator<Object> i$ = this.recoveredNodeIds.iterator();
                while (i$.hasNext()) {
                    long nodeId = (Long)i$.next();
                    Iterable<NodePropertyUpdate> updates = this.storeView.nodeAsUpdates(nodeId);
                    this.applyUpdates(updates, updaterMap);
                    this.monitor.appliedRecoveredData(updates);
                }
            }
        }
        this.recoveredNodeIds.clear();
    }

    private void applyUpdates(Iterable<NodePropertyUpdate> updates, IndexUpdaterMap updaterMap) {
        block5: for (NodePropertyUpdate update : updates) {
            int propertyKeyId = update.getPropertyKeyId();
            switch (update.getUpdateMode()) {
                case ADDED: {
                    int i;
                    int len = update.getNumberOfLabelsAfter();
                    for (i = 0; i < len; ++i) {
                        IndexDescriptor descriptor = new IndexDescriptor(update.getLabelAfter(i), propertyKeyId);
                        this.processUpdateIfIndexExists(updaterMap, update, descriptor);
                    }
                    continue block5;
                }
                case REMOVED: {
                    int i;
                    int len = update.getNumberOfLabelsBefore();
                    for (i = 0; i < len; ++i) {
                        IndexDescriptor descriptor = new IndexDescriptor(update.getLabelBefore(i), propertyKeyId);
                        this.processUpdateIfIndexExists(updaterMap, update, descriptor);
                    }
                    continue block5;
                }
                case CHANGED: {
                    int lenBefore = update.getNumberOfLabelsBefore();
                    int lenAfter = update.getNumberOfLabelsAfter();
                    int i = 0;
                    int j = 0;
                    while (i < lenBefore && j < lenAfter) {
                        int labelAfter;
                        int labelBefore = update.getLabelBefore(i);
                        if (labelBefore == (labelAfter = update.getLabelAfter(j))) {
                            IndexDescriptor descriptor = new IndexDescriptor(labelAfter, propertyKeyId);
                            this.processUpdateIfIndexExists(updaterMap, update, descriptor);
                            ++i;
                            ++j;
                            continue;
                        }
                        if (labelBefore < labelAfter) {
                            ++i;
                            continue;
                        }
                        ++j;
                    }
                    break;
                }
            }
        }
    }

    private IndexDescriptor processUpdateIfIndexExists(IndexUpdaterMap updaterMap, NodePropertyUpdate update, IndexDescriptor descriptor) {
        try {
            IndexUpdater updater = updaterMap.getUpdater(descriptor);
            if (null != updater) {
                updater.process(update);
                return descriptor;
            }
        }
        catch (IOException | IndexEntryConflictException e) {
            throw new UnderlyingStorageException(e);
        }
        return null;
    }

    public void dropIndex(IndexRule rule) {
        long indexId = rule.getId();
        IndexProxy index = this.indexMapRef.removeIndexProxy(indexId);
        if (this.state == State.RUNNING) {
            assert (index != null) : "Index " + rule + " doesn't exists";
            try {
                Future<Void> dropFuture = index.drop();
                this.awaitIndexFuture(dropFuture);
            }
            catch (Exception e) {
                throw Exceptions.launderedException(e);
            }
        }
    }

    public void triggerIndexSampling(IndexSamplingMode mode) {
        this.logger.info("Manual trigger for sampling all indexes [" + (Object)((Object)mode) + "]");
        this.samplingController.sampleIndexes(mode);
    }

    public void triggerIndexSampling(IndexDescriptor descriptor, IndexSamplingMode mode) {
        String description = descriptor.userDescription(this.tokenNameLookup);
        this.logger.info("Manual trigger for sampling index " + description + " [" + (Object)((Object)mode) + "]");
        this.samplingController.sampleIndex(descriptor, mode);
    }

    private void awaitIndexFuture(Future<Void> future) throws Exception {
        try {
            future.get(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException e) {
            Thread.interrupted();
            throw e;
        }
    }

    private void dropRecoveringIndexes(IndexMap indexMap, Map<Long, Pair<IndexDescriptor, SchemaIndexProvider.Descriptor>> recoveringIndexes) throws IOException {
        for (long indexId : recoveringIndexes.keySet()) {
            IndexProxy indexProxy = indexMap.removeIndexProxy(indexId);
            indexProxy.drop();
        }
    }

    public void activateIndex(long indexId) throws IndexNotFoundKernelException, IndexActivationFailedKernelException, IndexPopulationFailedKernelException {
        try {
            if (this.state == State.RUNNING) {
                IndexProxy index = this.getIndexProxy(indexId);
                index.awaitStoreScanCompleted();
                index.activate();
            }
        }
        catch (InterruptedException e) {
            Thread.interrupted();
            throw new IndexActivationFailedKernelException(e, "Unable to activate index, thread was interrupted.");
        }
    }

    public IndexProxy getIndexProxy(long indexId) throws IndexNotFoundKernelException {
        return this.indexMapRef.getIndexProxy(indexId);
    }

    public void validateIndex(long indexId) throws IndexNotFoundKernelException, ConstraintVerificationFailedKernelException, IndexPopulationFailedKernelException {
        this.getIndexProxy(indexId).validate();
    }

    public void flushAll() {
        for (IndexProxy index : this.indexMapRef.getAllIndexProxies()) {
            try {
                index.force();
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Unable to force " + index, e);
            }
        }
    }

    private void closeAllIndexes() {
        Iterable<IndexProxy> indexesToStop = this.indexMapRef.clear();
        ArrayList<Future<Void>> indexStopFutures = new ArrayList<Future<Void>>();
        for (IndexProxy indexProxy : indexesToStop) {
            try {
                indexStopFutures.add(indexProxy.close());
            }
            catch (IOException e) {
                this.logger.error("Unable to close index", e);
            }
        }
        for (Future future : indexStopFutures) {
            try {
                this.awaitIndexFuture(future);
            }
            catch (Exception e) {
                this.logger.error("Error awaiting index to close", e);
            }
        }
    }

    public ResourceIterator<File> snapshotStoreFiles() throws IOException {
        ArrayList<ResourceIterator<File>> snapshots = new ArrayList<ResourceIterator<File>>();
        HashSet<SchemaIndexProvider.Descriptor> fromProviders = new HashSet<SchemaIndexProvider.Descriptor>();
        for (IndexProxy indexProxy : this.indexMapRef.getAllIndexProxies()) {
            SchemaIndexProvider.Descriptor providerDescriptor = indexProxy.getProviderDescriptor();
            if (fromProviders.add(providerDescriptor)) {
                snapshots.add(this.providerMap.apply(providerDescriptor).snapshotMetaFiles());
            }
            snapshots.add(indexProxy.snapshotFiles());
        }
        return Iterables.concatResourceIterators(snapshots.iterator());
    }

    public static abstract class MonitorAdapter
    implements Monitor {
        @Override
        public void appliedRecoveredData(Iterable<NodePropertyUpdate> updates) {
        }

        @Override
        public void applyingRecoveredData(Set<Long> recoveredNodeIds) {
        }

        @Override
        public void populationCompleteOn(IndexDescriptor descriptor) {
        }

        @Override
        public void verifyDeferredConstraints() {
        }
    }

    public static interface Monitor {
        public void applyingRecoveredData(Set<Long> var1);

        public void appliedRecoveredData(Iterable<NodePropertyUpdate> var1);

        public void populationCompleteOn(IndexDescriptor var1);

        public void verifyDeferredConstraints();
    }

    static enum State {
        NOT_STARTED,
        STARTING,
        RUNNING,
        STOPPED;

    }
}

