/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.transaction.xa.Xid;
import org.neo4j.helpers.UTF8;
import org.neo4j.kernel.impl.nioneo.store.FileSystemAbstraction;
import org.neo4j.kernel.impl.transaction.XidImpl;
import org.neo4j.kernel.impl.transaction.xaframework.DirectMappedLogBuffer;
import org.neo4j.kernel.impl.transaction.xaframework.ForceMode;
import org.neo4j.kernel.impl.transaction.xaframework.LogBuffer;
import org.neo4j.kernel.monitoring.ByteCounterMonitor;
import org.neo4j.kernel.monitoring.Monitors;

public class TxLog {
    public static final int MAX_RECORD_SIZE = 131;
    public static final int LOG_ROTATION_THRESHOLD = 1000;
    public static final int SCAN_WINDOW_SIZE = 131000;
    private final ByteCounterMonitor bufferMonitor;
    public static final byte NULL_BYTE = 0;
    public static final byte TX_START = 1;
    public static final byte BRANCH_ADD = 2;
    public static final byte MARK_COMMIT = 3;
    public static final byte TX_DONE = 4;
    private final Collection<ByteArrayKey> activeTransactions = new HashSet<ByteArrayKey>();
    private final FileSystemAbstraction fileSystem;
    private File name = null;
    private LogBuffer logBuffer;
    private int recordCount = 0;

    public TxLog(File fileName, FileSystemAbstraction fileSystem, Monitors monitors) throws IOException {
        this.bufferMonitor = monitors.newMonitor(ByteCounterMonitor.class, this.getClass(), new String[0]);
        if (fileName == null) {
            throw new IllegalArgumentException("Null filename");
        }
        this.fileSystem = fileSystem;
        FileChannel fileChannel = fileSystem.open(fileName, "rw");
        fileChannel.position(fileChannel.size());
        this.logBuffer = new DirectMappedLogBuffer(fileChannel, this.bufferMonitor);
        this.name = fileName;
        this.recreateActiveTransactionState();
    }

    private void recreateActiveTransactionState() throws IOException {
        for (List<Record> tx : this.getDanglingRecords()) {
            for (Record record : tx) {
                if (record.getType() != 1) continue;
                this.activeTransactions.add(new ByteArrayKey(record.getGlobalId()));
            }
        }
    }

    public String getName() {
        return this.name.getPath();
    }

    public int getRecordCount() {
        return this.recordCount;
    }

    public synchronized void close() throws IOException {
        this.logBuffer.force();
        this.logBuffer.getFileChannel().close();
    }

    public void force() throws IOException {
        this.logBuffer.force();
    }

    public synchronized void truncate() throws IOException {
        FileChannel fileChannel = this.logBuffer.getFileChannel();
        fileChannel.position(0L);
        fileChannel.truncate(0L);
        this.recordCount = 0;
        this.logBuffer = new DirectMappedLogBuffer(fileChannel, this.bufferMonitor);
        this.activeTransactions.clear();
    }

    public synchronized void txStart(byte[] globalId) throws IOException {
        this.assertNotNull(globalId, "global id");
        if (!this.activeTransactions.add(new ByteArrayKey(globalId))) {
            throw new IllegalStateException("Global ID " + Arrays.toString(globalId) + " already started");
        }
        byte globalIdSize = (byte)globalId.length;
        this.logBuffer.put((byte)1).put(globalIdSize).put(globalId);
        ++this.recordCount;
    }

    private void assertNotNull(Object obj, String name) {
        if (obj == null) {
            throw new IllegalArgumentException("Null " + name);
        }
    }

    public synchronized void addBranch(byte[] globalId, byte[] branchId) throws IOException {
        this.assertNotNull(globalId, "global id");
        this.assertNotNull(branchId, "branch id");
        this.assertActive(globalId);
        byte globalIdSize = (byte)globalId.length;
        byte branchIdSize = (byte)branchId.length;
        this.logBuffer.put((byte)2).put(globalIdSize).put(branchIdSize).put(globalId).put(branchId);
        ++this.recordCount;
    }

    private void assertActive(byte[] globalId) {
        if (!this.activeTransactions.contains(new ByteArrayKey(globalId))) {
            throw new IllegalStateException("Global ID " + Arrays.toString(globalId) + " not active");
        }
    }

    public synchronized void markAsCommitting(byte[] globalId, ForceMode forceMode) throws IOException {
        this.assertNotNull(globalId, "global id");
        this.assertActive(globalId);
        byte globalIdSize = (byte)globalId.length;
        this.logBuffer.put((byte)3).put(globalIdSize).put(globalId);
        forceMode.force(this.logBuffer);
        ++this.recordCount;
    }

    public synchronized void txDone(byte[] globalId) throws IOException {
        this.assertNotNull(globalId, "global id");
        if (!this.activeTransactions.remove(new ByteArrayKey(globalId))) {
            throw new IllegalStateException("Global ID " + Arrays.toString(globalId) + " not active");
        }
        byte globalIdSize = (byte)globalId.length;
        this.logBuffer.put((byte)4).put(globalIdSize).put(globalId);
        ++this.recordCount;
    }

    void writeRecord(Record record, ForceMode forceMode) throws IOException {
        switch (record.getType()) {
            case 1: {
                this.txStart(record.getGlobalId());
                break;
            }
            case 2: {
                this.addBranch(record.getGlobalId(), record.getBranchId());
                break;
            }
            case 3: {
                this.markAsCommitting(record.getGlobalId(), forceMode);
                break;
            }
            default: {
                throw new IOException("Illegal record type[" + record.getType() + "]");
            }
        }
    }

    public synchronized Iterable<List<Record>> getDanglingRecords() throws IOException {
        FileChannel fileChannel = this.logBuffer.getFileChannel();
        ByteBuffer buffer = ByteBuffer.allocateDirect(131000);
        this.readFileIntoBuffer(fileChannel, buffer, 0L);
        long nextPosition = 0L;
        int seqNr = 0;
        HashMap<Xid, List<Record>> recordMap = new HashMap<Xid, List<Record>>();
        while (buffer.hasRemaining()) {
            int recordSize;
            byte recordType = buffer.get();
            switch (recordType) {
                case 1: {
                    recordSize = TxLog.readTxStartRecordInto(recordMap, buffer, seqNr++);
                    break;
                }
                case 2: {
                    recordSize = TxLog.readBranchAddRecordInto(recordMap, buffer, seqNr++);
                    break;
                }
                case 3: {
                    recordSize = TxLog.readMarkCommitRecordInto(recordMap, buffer, seqNr++);
                    break;
                }
                case 4: {
                    recordSize = TxLog.readTxDoneAndRemoveTransactionFrom(recordMap, buffer);
                    break;
                }
                case 0: {
                    recordSize = 1;
                    break;
                }
                default: {
                    throw new IOException("Unknown type: " + recordType);
                }
            }
            if (recordSize == 0) break;
            if (buffer.remaining() >= 131 || fileChannel.size() - (nextPosition += (long)recordSize) <= (long)buffer.remaining()) continue;
            this.readFileIntoBuffer(fileChannel, buffer, nextPosition);
        }
        return recordMap.values();
    }

    private void readFileIntoBuffer(FileChannel fileChannel, ByteBuffer buffer, long nextPosition) throws IOException {
        buffer.clear();
        fileChannel.position(nextPosition);
        fileChannel.read(buffer);
        buffer.flip();
    }

    private static int readTxStartRecordInto(Map<Xid, List<Record>> recordMap, ByteBuffer buffer, int seqNr) throws IOException {
        if (!buffer.hasRemaining()) {
            return 0;
        }
        byte[] globalId = new byte[buffer.get()];
        if (buffer.remaining() < globalId.length) {
            return 0;
        }
        buffer.get(globalId);
        XidImpl xid = new XidImpl(globalId, new byte[0]);
        if (recordMap.containsKey(xid)) {
            throw new IOException("Tx start for same xid[" + xid + "] found twice");
        }
        LinkedList<Record> recordList = new LinkedList<Record>();
        recordList.add(new Record(1, globalId, null, seqNr));
        recordMap.put(xid, recordList);
        return 2 + globalId.length;
    }

    private static int readBranchAddRecordInto(Map<Xid, List<Record>> recordMap, ByteBuffer buffer, int seqNr) throws IOException {
        if (buffer.remaining() < 2) {
            return 0;
        }
        byte[] globalId = new byte[buffer.get()];
        byte[] branchId = new byte[buffer.get()];
        if (buffer.remaining() < globalId.length + branchId.length) {
            return 0;
        }
        buffer.get(globalId);
        buffer.get(branchId);
        XidImpl xid = new XidImpl(globalId, new byte[0]);
        if (!recordMap.containsKey(xid)) {
            throw new IOException(String.format("Branch[%s] found for [%s] but no record list found in map", UTF8.decode(branchId), xid));
        }
        recordMap.get(xid).add(new Record(2, globalId, branchId, seqNr));
        return 3 + globalId.length + branchId.length;
    }

    private static int readMarkCommitRecordInto(Map<Xid, List<Record>> recordMap, ByteBuffer buffer, int seqNr) throws IOException {
        if (!buffer.hasRemaining()) {
            return 0;
        }
        byte[] globalId = new byte[buffer.get()];
        if (buffer.remaining() < globalId.length) {
            return 0;
        }
        buffer.get(globalId);
        XidImpl xid = new XidImpl(globalId, new byte[0]);
        if (!recordMap.containsKey(xid)) {
            throw new IOException("Committing xid[" + xid + "] mark found but no record list found in map");
        }
        List<Record> recordList = recordMap.get(xid);
        recordList.add(new Record(3, globalId, null, seqNr));
        recordMap.put(xid, recordList);
        return 2 + globalId.length;
    }

    private static int readTxDoneAndRemoveTransactionFrom(Map<Xid, List<Record>> recordMap, ByteBuffer buffer) throws IOException {
        if (!buffer.hasRemaining()) {
            return 0;
        }
        byte[] globalId = new byte[buffer.get()];
        if (buffer.remaining() < globalId.length) {
            return 0;
        }
        buffer.get(globalId);
        XidImpl xid = new XidImpl(globalId, new byte[0]);
        if (!recordMap.containsKey(xid)) {
            throw new IOException("Committing xid[" + xid + "] mark found but no record list found in map");
        }
        recordMap.remove(xid);
        return 2 + globalId.length;
    }

    public synchronized void switchToLogFile(File newFile) throws IOException {
        if (newFile == null) {
            throw new IllegalArgumentException("Null filename");
        }
        this.force();
        Iterable<List<Record>> itr = this.getDanglingRecords();
        this.close();
        ArrayList<Record> records = new ArrayList<Record>();
        for (List<Record> tx : itr) {
            records.addAll(tx);
        }
        Collections.sort(records, new Comparator<Record>(){

            @Override
            public int compare(Record r1, Record r2) {
                return r1.getSequenceNumber() - r2.getSequenceNumber();
            }
        });
        Iterator recordItr = records.iterator();
        FileChannel fileChannel = this.fileSystem.open(newFile, "rw");
        fileChannel.position(fileChannel.size());
        this.logBuffer = new DirectMappedLogBuffer(fileChannel, this.bufferMonitor);
        this.name = newFile;
        this.truncate();
        while (recordItr.hasNext()) {
            Record record = (Record)recordItr.next();
            this.writeRecord(record, ForceMode.forced);
        }
        this.force();
    }

    public static class Record {
        private byte type = 0;
        private byte[] globalId = null;
        private byte[] branchId = null;
        private int seqNr = -1;

        Record(byte type, byte[] globalId, byte[] branchId, int seqNr) {
            if (type < 1 || type > 4) {
                throw new IllegalArgumentException("Illegal type: " + type);
            }
            this.type = type;
            this.globalId = globalId;
            this.branchId = branchId;
            this.seqNr = seqNr;
        }

        public byte getType() {
            return this.type;
        }

        public byte[] getGlobalId() {
            return this.globalId;
        }

        public byte[] getBranchId() {
            return this.branchId;
        }

        public int getSequenceNumber() {
            return this.seqNr;
        }

        public String toString() {
            XidImpl xid = new XidImpl(this.globalId, this.branchId == null ? new byte[]{} : this.branchId);
            int size = 1 + this.sizeOf(this.globalId) + this.sizeOf(this.branchId);
            return "TxLogRecord[" + this.typeName() + "," + xid + "," + this.seqNr + "," + size + "]";
        }

        private int sizeOf(byte[] id) {
            if (id == null) {
                return 0;
            }
            return 1 + id.length;
        }

        String typeName() {
            switch (this.type) {
                case 1: {
                    return "TX_START";
                }
                case 2: {
                    return "BRANCH_ADD";
                }
                case 3: {
                    return "MARK_COMMIT";
                }
                case 4: {
                    return "TX_DONE";
                }
            }
            return "<unknown type>";
        }
    }

    private static final class ByteArrayKey {
        private final byte[] bytes;

        private ByteArrayKey(byte[] bytes) {
            this.bytes = bytes;
        }

        public int hashCode() {
            return Arrays.hashCode(this.bytes);
        }

        public boolean equals(Object obj) {
            return obj instanceof ByteArrayKey && Arrays.equals(this.bytes, ((ByteArrayKey)obj).bytes);
        }
    }
}

