/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.api.index;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.neo4j.helpers.Exceptions;
import org.neo4j.helpers.Pair;
import org.neo4j.helpers.collection.IteratorUtil;
import org.neo4j.kernel.api.exceptions.index.IndexActivationFailedKernelException;
import org.neo4j.kernel.api.exceptions.index.IndexNotFoundKernelException;
import org.neo4j.kernel.api.exceptions.index.IndexPopulationFailedKernelException;
import org.neo4j.kernel.api.index.IndexAccessor;
import org.neo4j.kernel.api.index.IndexConfiguration;
import org.neo4j.kernel.api.index.IndexPopulator;
import org.neo4j.kernel.api.index.InternalIndexState;
import org.neo4j.kernel.api.index.NodePropertyUpdate;
import org.neo4j.kernel.api.index.SchemaIndexProvider;
import org.neo4j.kernel.api.operations.TokenNameLookup;
import org.neo4j.kernel.impl.api.UpdateableSchemaState;
import org.neo4j.kernel.impl.api.constraints.ConstraintVerificationFailedKernelException;
import org.neo4j.kernel.impl.api.index.ContractCheckingIndexProxy;
import org.neo4j.kernel.impl.api.index.DelegatingIndexProxy;
import org.neo4j.kernel.impl.api.index.FailedIndexProxy;
import org.neo4j.kernel.impl.api.index.FailedPopulatingIndexProxyFactory;
import org.neo4j.kernel.impl.api.index.FlippableIndexProxy;
import org.neo4j.kernel.impl.api.index.IndexDescriptor;
import org.neo4j.kernel.impl.api.index.IndexPopulationFailure;
import org.neo4j.kernel.impl.api.index.IndexProxy;
import org.neo4j.kernel.impl.api.index.IndexProxyFactory;
import org.neo4j.kernel.impl.api.index.IndexStoreView;
import org.neo4j.kernel.impl.api.index.OnlineIndexProxy;
import org.neo4j.kernel.impl.api.index.PopulatingIndexProxy;
import org.neo4j.kernel.impl.api.index.RecoveringIndexProxy;
import org.neo4j.kernel.impl.api.index.RuleUpdateFilterIndexProxy;
import org.neo4j.kernel.impl.api.index.SchemaIndexProviderMap;
import org.neo4j.kernel.impl.api.index.TentativeConstraintIndexProxy;
import org.neo4j.kernel.impl.nioneo.store.IndexRule;
import org.neo4j.kernel.impl.nioneo.store.UnderlyingStorageException;
import org.neo4j.kernel.impl.util.JobScheduler;
import org.neo4j.kernel.impl.util.StringLogger;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;
import org.neo4j.kernel.logging.Logging;

public class IndexingService
extends LifecycleAdapter {
    private final ConcurrentHashMap<Long, IndexProxy> indexes = new ConcurrentHashMap();
    private boolean serviceRunning = false;
    private final JobScheduler scheduler;
    private final SchemaIndexProviderMap providerMap;
    private final IndexStoreView storeView;
    private final TokenNameLookup tokenNameLookup;
    private final Logging logging;
    private final StringLogger logger;
    private final UpdateableSchemaState updateableSchemaState;

    public IndexingService(JobScheduler scheduler, SchemaIndexProviderMap providerMap, IndexStoreView storeView, TokenNameLookup tokenNameLookup, UpdateableSchemaState updateableSchemaState, Logging logging) {
        this.scheduler = scheduler;
        this.providerMap = providerMap;
        this.storeView = storeView;
        this.logging = logging;
        this.logger = logging.getMessagesLog(this.getClass());
        this.updateableSchemaState = updateableSchemaState;
        this.tokenNameLookup = tokenNameLookup;
        if (providerMap == null || providerMap.getDefaultProvider() == null) {
            throw new IllegalStateException("You cannot run the database without an index provider, please make sure that a valid provider (subclass of " + SchemaIndexProvider.class.getName() + ") is on your classpath.");
        }
    }

    public void initIndexes(Iterator<IndexRule> indexRules) {
        for (IndexRule indexRule : IteratorUtil.loop(indexRules)) {
            IndexProxy indexProxy;
            long ruleId = indexRule.getId();
            IndexDescriptor descriptor = this.createDescriptor(indexRule);
            SchemaIndexProvider.Descriptor providerDescriptor = indexRule.getProviderDescriptor();
            SchemaIndexProvider provider = this.providerMap.apply(providerDescriptor);
            InternalIndexState initialState = provider.getInitialState(ruleId);
            this.logger.info(String.format("IndexingService.initIndexes: index on %s is %s", descriptor.userDescription(this.tokenNameLookup), initialState.name()));
            switch (initialState) {
                case ONLINE: {
                    indexProxy = this.createAndStartOnlineIndexProxy(ruleId, descriptor, providerDescriptor, indexRule.isConstraintIndex());
                    break;
                }
                case POPULATING: {
                    indexProxy = this.createAndStartRecoveringIndexProxy(ruleId, descriptor, providerDescriptor);
                    break;
                }
                case FAILED: {
                    indexProxy = this.createAndStartFailedIndexProxy(ruleId, descriptor, providerDescriptor, indexRule.isConstraintIndex(), IndexPopulationFailure.failure(provider.getPopulationFailure(ruleId)));
                    break;
                }
                default: {
                    throw new IllegalArgumentException("" + (Object)((Object)initialState));
                }
            }
            this.indexes.put(ruleId, indexProxy);
        }
    }

    @Override
    public void start() throws Exception {
        long ruleId;
        HashSet<IndexProxy> rebuildingIndexes = new HashSet<IndexProxy>();
        HashMap<Long, Pair<IndexDescriptor, SchemaIndexProvider.Descriptor>> rebuildingIndexDescriptors = new HashMap<Long, Pair<IndexDescriptor, SchemaIndexProvider.Descriptor>>();
        for (Map.Entry<Long, IndexProxy> entry : this.indexes.entrySet()) {
            ruleId = entry.getKey();
            IndexProxy indexProxy = entry.getValue();
            InternalIndexState state = indexProxy.getState();
            this.logger.info(String.format("IndexingService.start: index on %s is %s", indexProxy.getDescriptor().userDescription(this.tokenNameLookup), state.name()));
            switch (state) {
                case ONLINE: {
                    break;
                }
                case POPULATING: {
                    rebuildingIndexes.add(indexProxy);
                    Pair<IndexDescriptor, SchemaIndexProvider.Descriptor> descriptors = Pair.of(indexProxy.getDescriptor(), indexProxy.getProviderDescriptor());
                    rebuildingIndexDescriptors.put(ruleId, descriptors);
                    break;
                }
            }
        }
        this.dropIndexes(rebuildingIndexes);
        for (Map.Entry<Long, IndexProxy> entry : rebuildingIndexDescriptors.entrySet()) {
            ruleId = entry.getKey();
            Pair descriptors = (Pair)((Object)entry.getValue());
            IndexDescriptor indexDescriptor = (IndexDescriptor)descriptors.first();
            SchemaIndexProvider.Descriptor providerDescriptor = (SchemaIndexProvider.Descriptor)descriptors.other();
            IndexProxy indexProxy = this.createAndStartPopulatingIndexProxy(ruleId, indexDescriptor, providerDescriptor, this.serviceRunning);
            this.indexes.put(ruleId, indexProxy);
        }
        this.serviceRunning = true;
    }

    @Override
    public void stop() {
        this.serviceRunning = false;
        this.closeAllIndexes();
    }

    private void closeAllIndexes() {
        ArrayList<IndexProxy> indexesToStop = new ArrayList<IndexProxy>(this.indexes.values());
        this.indexes.clear();
        ArrayList<Future<Void>> indexStopFutures = new ArrayList<Future<Void>>();
        for (IndexProxy indexProxy : indexesToStop) {
            try {
                indexStopFutures.add(indexProxy.close());
            }
            catch (IOException e) {
                this.logger.error("Unable to close index", e);
            }
        }
        for (Future future : indexStopFutures) {
            try {
                this.awaitIndexFuture(future);
            }
            catch (Exception e) {
                this.logger.error("Error awaiting index to close", e);
            }
        }
    }

    public IndexProxy getProxyForRule(long indexId) throws IndexNotFoundKernelException {
        IndexProxy indexProxy = this.indexes.get(indexId);
        if (indexProxy == null) {
            throw new IndexNotFoundKernelException("No index with id " + indexId + " exists.");
        }
        return indexProxy;
    }

    public void createIndex(IndexRule rule) {
        long ruleId = rule.getId();
        IndexProxy index = this.indexes.get(ruleId);
        IndexDescriptor descriptor = this.createDescriptor(rule);
        if (this.serviceRunning) {
            assert (index == null) : "Index " + rule + " already exists";
            try {
                index = this.createAndStartPopulatingIndexProxy(ruleId, descriptor, rule.getProviderDescriptor(), rule.isConstraintIndex());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else if (index == null) {
            index = this.createAndStartRecoveringIndexProxy(ruleId, descriptor, rule.getProviderDescriptor());
        }
        this.indexes.put(rule.getId(), index);
    }

    private String indexUserDescription(IndexDescriptor descriptor, SchemaIndexProvider.Descriptor providerDescriptor) {
        return String.format("%s [provider: %s]", descriptor.userDescription(this.tokenNameLookup), providerDescriptor.toString());
    }

    public void updateIndexes(Iterable<NodePropertyUpdate> updates) {
        if (this.serviceRunning) {
            for (IndexProxy index : this.indexes.values()) {
                try {
                    index.update(updates);
                }
                catch (IOException e) {
                    throw new UnderlyingStorageException("Unable to update " + index, e);
                }
            }
        } else {
            for (IndexProxy index : this.indexes.values()) {
                try {
                    index.recover(updates);
                }
                catch (IOException e) {
                    throw new UnderlyingStorageException("Unable to update " + index, e);
                }
            }
        }
    }

    public void dropIndex(IndexRule rule) {
        IndexProxy index = this.indexes.remove(rule.getId());
        if (this.serviceRunning) {
            assert (index != null) : "Index " + rule + " doesn't exists";
            try {
                Future<Void> dropFuture = index.drop();
                this.awaitIndexFuture(dropFuture);
            }
            catch (Exception e) {
                throw Exceptions.launderedException(e);
            }
        }
    }

    private IndexProxy createAndStartPopulatingIndexProxy(final long ruleId, final IndexDescriptor descriptor, final SchemaIndexProvider.Descriptor providerDescriptor, final boolean unique) throws IOException {
        final FlippableIndexProxy flipper = new FlippableIndexProxy();
        String indexUserDescription = this.indexUserDescription(descriptor, providerDescriptor);
        IndexPopulator populator = this.getPopulatorFromProvider(providerDescriptor, ruleId, new IndexConfiguration(unique));
        FailedPopulatingIndexProxyFactory failureDelegateFactory = new FailedPopulatingIndexProxyFactory(descriptor, providerDescriptor, populator, indexUserDescription);
        PopulatingIndexProxy populatingIndex = new PopulatingIndexProxy(this.scheduler, descriptor, providerDescriptor, failureDelegateFactory, populator, flipper, this.storeView, indexUserDescription, this.updateableSchemaState, this.logging);
        flipper.flipTo(populatingIndex);
        flipper.setFlipTarget(new IndexProxyFactory(){

            @Override
            public IndexProxy create() {
                try {
                    OnlineIndexProxy onlineProxy = new OnlineIndexProxy(descriptor, providerDescriptor, IndexingService.this.getOnlineAccessorFromProvider(providerDescriptor, ruleId, new IndexConfiguration(unique)));
                    if (unique) {
                        return new TentativeConstraintIndexProxy(flipper, onlineProxy);
                    }
                    return onlineProxy;
                }
                catch (IOException e) {
                    return IndexingService.this.createAndStartFailedIndexProxy(ruleId, descriptor, providerDescriptor, unique, IndexPopulationFailure.failure(e));
                }
            }
        });
        IndexProxy result = this.contractCheckedProxy(flipper, false);
        result = this.serviceDecoratedProxy(ruleId, result);
        result.start();
        return result;
    }

    private IndexProxy createAndStartOnlineIndexProxy(long ruleId, IndexDescriptor descriptor, SchemaIndexProvider.Descriptor providerDescriptor, boolean unique) {
        try {
            IndexAccessor onlineAccessor = this.getOnlineAccessorFromProvider(providerDescriptor, ruleId, new IndexConfiguration(unique));
            IndexProxy result = new OnlineIndexProxy(descriptor, providerDescriptor, onlineAccessor);
            result = this.contractCheckedProxy(result, true);
            return this.serviceDecoratedProxy(ruleId, result);
        }
        catch (IOException e) {
            return this.createAndStartFailedIndexProxy(ruleId, descriptor, providerDescriptor, unique, IndexPopulationFailure.failure(e));
        }
    }

    private IndexProxy createAndStartFailedIndexProxy(long ruleId, IndexDescriptor descriptor, SchemaIndexProvider.Descriptor providerDescriptor, boolean unique, IndexPopulationFailure populationFailure) {
        IndexPopulator indexPopulator = this.getPopulatorFromProvider(providerDescriptor, ruleId, new IndexConfiguration(unique));
        String indexUserDescription = this.indexUserDescription(descriptor, providerDescriptor);
        IndexProxy result = new FailedIndexProxy(descriptor, providerDescriptor, indexUserDescription, indexPopulator, populationFailure);
        result = this.contractCheckedProxy(result, true);
        return this.serviceDecoratedProxy(ruleId, result);
    }

    private IndexProxy createAndStartRecoveringIndexProxy(long ruleId, IndexDescriptor descriptor, SchemaIndexProvider.Descriptor providerDescriptor) {
        IndexProxy result = new RecoveringIndexProxy(descriptor, providerDescriptor);
        result = this.contractCheckedProxy(result, true);
        return this.serviceDecoratedProxy(ruleId, result);
    }

    private IndexPopulator getPopulatorFromProvider(SchemaIndexProvider.Descriptor providerDescriptor, long ruleId, IndexConfiguration config) {
        SchemaIndexProvider indexProvider = this.providerMap.apply(providerDescriptor);
        return indexProvider.getPopulator(ruleId, config);
    }

    private IndexAccessor getOnlineAccessorFromProvider(SchemaIndexProvider.Descriptor providerDescriptor, long ruleId, IndexConfiguration config) throws IOException {
        SchemaIndexProvider indexProvider = this.providerMap.apply(providerDescriptor);
        return indexProvider.getOnlineAccessor(ruleId, config);
    }

    private IndexProxy contractCheckedProxy(IndexProxy result, boolean started) {
        result = new ContractCheckingIndexProxy(result, started);
        return result;
    }

    private IndexProxy serviceDecoratedProxy(long ruleId, IndexProxy result) {
        result = new RuleUpdateFilterIndexProxy(result);
        result = new ServiceStateUpdatingIndexProxy(ruleId, result);
        return result;
    }

    private IndexDescriptor createDescriptor(IndexRule rule) {
        return new IndexDescriptor(rule.getLabel(), rule.getPropertyKey());
    }

    private void awaitIndexFuture(Future<Void> future) throws Exception {
        try {
            future.get(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException e) {
            Thread.interrupted();
            throw e;
        }
    }

    private void dropIndexes(Set<IndexProxy> recoveringIndexes) throws Exception {
        for (IndexProxy indexProxy : recoveringIndexes) {
            indexProxy.drop().get();
        }
    }

    public void activateIndex(long indexId) throws IndexNotFoundKernelException, IndexActivationFailedKernelException, IndexPopulationFailedKernelException {
        IndexProxy index = this.getProxyForRule(indexId);
        try {
            index.awaitStoreScanCompleted();
            index.activate();
        }
        catch (InterruptedException e) {
            Thread.interrupted();
            throw new IndexActivationFailedKernelException(e, "Unable to activate index, thread was interrupted.");
        }
    }

    public void validateIndex(long indexId) throws IndexNotFoundKernelException, ConstraintVerificationFailedKernelException, IndexPopulationFailedKernelException {
        this.getProxyForRule(indexId).validate();
    }

    public void flushAll() {
        for (IndexProxy index : this.indexes.values()) {
            try {
                index.force();
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Unable to force " + index, e);
            }
        }
    }

    class ServiceStateUpdatingIndexProxy
    extends DelegatingIndexProxy {
        private final long ruleId;

        ServiceStateUpdatingIndexProxy(long ruleId, IndexProxy delegate) {
            super(delegate);
            this.ruleId = ruleId;
        }

        @Override
        public Future<Void> drop() throws IOException {
            IndexingService.this.indexes.remove(this.ruleId, this);
            return super.drop();
        }
    }
}

