package org.mule.weave.v2.api.tooling.ast

import java.util.Optional
import java.util.function.Predicate

/**
  * Helper to operate with [[DWAstNode]].
  */
object DWAstNodeHelper {

  /**
    * Get the list of [[DWAstNode]] node children corresponding to an specific kind.
    * @param parent The [[DWAstNode]] node for searching.
    * @param kind The kind to search.
    * @return Get the list of [[DWAstNode]] that matches with the given kind.
    */
  def getChildrenByKind(parent: DWAstNode, kind: String): Array[DWAstNode] = {
    val children = getChildrenOf(parent, (t: DWAstNode) => {
      t.getKind == kind
    })
    children
  }

  /**
    * Get the list of [[DWAstNode]] node children corresponding to an specific condition.
    * @param parent The [[DWAstNode]] node for searching.
    * @param condition The condition to evaluate.
    * @return @return Get the list of [[DWAstNode]] that matches with the given condition.
    */
  def getChildrenOf(parent: DWAstNode, condition: Predicate[DWAstNode]): Array[DWAstNode] = {
    val children = parent.getChildren.filter(child => condition.test(child))
    children
  }

  /**
    * Get the `NAME_IDENTIFIER` node children for the given node.
    * @param node The [[DWAstNode]] node for searching.
    * @return @return the `NAME_IDENTIFIER` node children for the given node..
    */
  def getNameIdentifier(node: DWAstNode): Optional[DWAstNode] = {
    val maybeNameIdentifier = getChildrenByKind(node, DWAstNodeKind.NAME_IDENTIFIER).headOption
    if (maybeNameIdentifier.isDefined) {
      Optional.of(maybeNameIdentifier.get)
    } else {
      Optional.empty()
    }
  }

  /**
    * Get the descendant [[DWAstNode]] node children corresponding to an specific kind.
    * @param parent The [[DWAstNode]] node for searching.
    * @param kind The kind to search.
    * @return Get the descendant [[DWAstNode]] that matches with the given kind.
    */
  def getDescendantsOf(parent: DWAstNode, kind: String): Array[DWAstNode] = {
    if (parent.getKind == kind) {
      Array(parent)
    } else {
      val result: Array[DWAstNode] = parent.getChildren.flatMap(child => {
        getDescendantsOf(child, kind)
      })
      result
    }
  }
}
