package org.mule.weave.v2.api.tooling.annotation

import org.mule.weave.v2.api.tooling.ast.DWAstNode
import org.mule.weave.v2.api.tooling.message.MessageCollector

/**
  * Annotation processor executes on each node being annotated by the given annotation on the specified compilation phase
  */
trait DWAnnotationProcessor {

  /**
    * Run the annotation processing logic on the registered phase
    *
    * @param annotatedNode The [[DWAstNode]] that was annotated.
    * @param annotation The [[DWAstNode]] that was used.
    * @param context The context information available on this phase.
    */
  def run(annotatedNode: DWAstNode, annotation: DWAstNode, context: DWAnnotationContext): Unit

  /**
    * @return The [[DWProcessingPhase]] where this [[DWAnnotationProcessor]] is going to be executed.
    */
  def getProcessingPhase: DWProcessingPhase
}

/**
  * Represent the context information available during the current [[DWAnnotationProcessor]] execution.
  */
trait DWAnnotationContext {
  /**
    * @return The [[MessageCollector]] instance.
    */
  def getMessageCollector: MessageCollector
}

/**
  * Represents the [[DWAnnotationProcessor]] processing phase.
  */
sealed trait DWProcessingPhase

/**
  * [[DWProcessingPhase]] for the parsing validation phase.
  */
object DWParsingProcessingPhase extends DWProcessingPhase

/**
  * [[DWProcessingPhase]] for the canonical validation phase.
  */
object DWCanonicalProcessingPhase extends DWProcessingPhase

/**
  * [[DWProcessingPhase]] for the scope validation phase.
  */
object DWScopeProcessingPhase extends DWProcessingPhase

/**
  * [[DWProcessingPhase]] for the type-check validation phase.
  */
object DWTypeProcessingPhase extends DWProcessingPhase

/**
  * Base class for [[DWAnnotationProcessor]] executed at [[DWParsingProcessingPhase]] phase.
  */
abstract class DWAbstractParsingAnnotationProcessor extends DWAnnotationProcessor {
  final override def getProcessingPhase: DWProcessingPhase = DWParsingProcessingPhase
}

/**
  * Base class for [[DWAnnotationProcessor]] executed at [[DWCanonicalProcessingPhase]] phase.
  */
abstract class DWAbstractCanonicalAnnotationProcessor extends DWAnnotationProcessor {
  final override def getProcessingPhase: DWProcessingPhase = DWCanonicalProcessingPhase
}

/**
  * Base class for [[DWAnnotationProcessor]] executed at [[DWScopeProcessingPhase]] phase.
  */
abstract class DWAbstractScopeAnnotationProcessor extends DWAnnotationProcessor {
  final override def getProcessingPhase: DWProcessingPhase = DWScopeProcessingPhase
}

/**
  * Base class for [[DWAnnotationProcessor]] executed at [[DWTypeProcessingPhase]] phase.
  */
abstract class DWAbstractTypeAnnotationProcessor extends DWAnnotationProcessor {
  final override def getProcessingPhase: DWProcessingPhase = DWTypeProcessingPhase
}