package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.InvalidTypeParameterCall
import org.mule.weave.v2.parser.InvalidTypeRef
import org.mule.weave.v2.parser.annotation.InjectedNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.types.TypeParameterNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.{ NameIdentifier, VariableReferenceNode }
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.Reference
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.scope.VariableScope

/**
  * Validates that a type reference reference to a Type value and not something else
  *
  * @tparam T
  */
class TypeParameterCheckerPhase[T <: AstNode] extends CompilationPhase[ScopeGraphResult[T], ScopeGraphResult[T]] {

  override def doCall(source: ScopeGraphResult[T], context: ParsingContext): PhaseResult[_ <: ScopeGraphResult[T]] = {
    val typeReferenceNodes: Seq[TypeReferenceNode] = source.scope.astNavigator().allWithType(classOf[TypeReferenceNode])
    var astModified = false
    typeReferenceNodes.foreach((typeRefNode) => {
      val containerName: NameIdentifier = context.nameIdentifier
      val scope: ScopesNavigator = source.scope
      val wasModified: Boolean = injectTypeParametersOnTypeReferenceNode(containerName, scope.scopeOf(typeRefNode.variable).get, typeRefNode, context)
      if (wasModified) {
        astModified = wasModified
      }
    })
    if (astModified) {
      source.scope.invalidate()
    }
    SuccessResult(source, context)
  }

  /**
    * Returns true if ast was modified
    */
  private def injectTypeParametersOnTypeReferenceNode(containerName: NameIdentifier, scope: VariableScope, typeRefNode: TypeReferenceNode, context: ParsingContext): Boolean = {

    var astModified = false
    val nameIdentifier: NameIdentifier = typeRefNode.variable
    val maybeReference: Option[Reference] = scope.resolveVariable(nameIdentifier)

    maybeReference match {
      case Some(typeReference) => {
        val parentName: NameIdentifier = typeReference.moduleSource.getOrElse(containerName)
        val referenceAstNavigator: AstNavigator = typeReference.scope.astNavigator()
        val parentNode = referenceAstNavigator.parentOf(typeReference.referencedNode).get
        parentNode match {
          case td: TypeDirective => {
            if (td.typeParametersListNode.isEmpty && typeRefNode.typeArguments.nonEmpty) {
              context.messageCollector.error(InvalidTypeParameterCall(nameIdentifier.name, typeRefNode.typeArguments.get.size, td.typeParametersListNode.size), typeRefNode.location())
            }
            td.typeParametersListNode.foreach((typeParams) => {
              if (typeParams.children().nonEmpty) {
                val typeArguments = typeRefNode.typeArguments
                if (typeArguments.isDefined) {
                  //If parameters are defined and arguments are defined then they should have the same arity
                  if (typeArguments.get.size != typeParams.children().size) {
                    context.messageCollector.error(InvalidTypeParameterCall(nameIdentifier.name, typeArguments.get.size, typeParams.children().size), typeRefNode.location())
                  }
                } else {
                  //We inject default type parameters
                  val typeNodes = typeParams.typeParameters.map((tp) => {
                    tp.base
                      .map((astNode) => {
                        val node = astNode.cloneAst()
                        node match {
                          case typeReferenceNode: TypeReferenceNode => {
                            val mayBeReferencedTypeParam = typeReference.scope.resolveVariable(typeReferenceNode.variable)
                            mayBeReferencedTypeParam match {
                              case Some(referencedTypeParam) => {
                                val localName = referencedTypeParam.referencedNode.localName()
                                val typeParamFQN = referencedTypeParam.moduleSource.getOrElse(parentName).::(localName.name)
                                astModified = true
                                typeReferenceNode.typeArguments.map((typeArgs) => {
                                  typeArgs.map({
                                    case typeReferenceNode: TypeReferenceNode => {
                                      injectTypeParametersOnTypeReferenceNode(parentName, typeReference.scope, typeReferenceNode, context)
                                      typeReferenceNode
                                    }
                                    case tn => tn
                                  })
                                })
                                typeReferenceNode.copy(typeParamFQN)
                              }
                              case None => typeReferenceNode
                            }
                          }
                          case _ => node
                        }

                      })
                      .getOrElse(TypeReferenceNode(NameIdentifier("Any")).annotate(InjectedNodeAnnotation()))
                  })
                  typeRefNode.typeArguments = Some(typeNodes)
                }
              }
            })
          }
          case _: TypeParameterNode => {
            if (typeRefNode.typeArguments.nonEmpty) {
              context.messageCollector.error(InvalidTypeParameterCall(nameIdentifier.name, typeRefNode.typeArguments.get.size, 0), typeRefNode.location())
            }
          }
          case n => {
            if (typeRefNode.variable.name != NameIdentifier.INSERTED_FAKE_VARIABLE_NAME) {
              //Inserted Fake Variable can be ignore
              context.messageCollector.error(InvalidTypeRef(typeRefNode, n), typeRefNode.location())
            }
          }
        }
      }
      case None =>
    }
    astModified
  }
}
