package org.mule.weave.v2.parser.phase

import java.lang

import org.mule.weave.v2.parser.InvalidSyntaxMessage
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.ParserPosition
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.sdk.WeaveResource
import org.parboiled2.ErrorFormatter
import org.parboiled2.ParseError
import org.parboiled2.ParserInput

class ParsingPhase[T <: AstNode](rootRule: (ParsingContentInput, ParsingContext) => Either[ParseError, T]) extends CompilationPhase[ParsingContentInput, ParsingResult[T]] {

  private def doParse(input: ParsingContentInput, context: ParsingContext): Option[T] = {
    val result: Either[ParseError, T] = rootRule(input, context)
    result match {
      case Right(x) => Some(x)
      case Left(y) =>
        val start = ParserPosition(y.position, input.input)
        val end = ParserPosition(y.principalPosition, input.input)
        val location: WeaveLocation = WeaveLocation(start, end, input.nameIdentifier)
        val format: String = y.format(input.input, new ErrorFormatter() {
          override def formatErrorLine(sb: lang.StringBuilder, error: ParseError, input: ParserInput): lang.StringBuilder = {
            sb
          }
        })
        context.messageCollector.error(InvalidSyntaxMessage(format), location)
        None
    }
  }

  override def doCall(input: ParsingContentInput, context: ParsingContext): PhaseResult[ParsingResult[T]] = {
    val parse: Option[T] = doParse(input, context)
    if (parse.isDefined) {
      val result: T = parse.get
      SuccessResult(ParsingResult(input, result), context)
    } else {
      FailureResult(context)
    }
  }
}

/**
  * This class is a wrapper for results that are [[org.mule.weave.v2.parser.ast.AstNode]]s
  *
  * @tparam T Specific AstNode type
  */
trait AstNodeResultAware[T <: AstNode] {
  def astNode: T
}

/**
  * This class is a wrapper for results that are [[org.mule.weave.v2.parser.ast.AstNode]]s
  *
  *
  */
trait DummyScopeNavigatorAware {
  def dummyScopesNavigator: ScopesNavigator
}

trait ParsingContentAware {
  def input: ParsingContentInput
}

case class ParsingContentInput(resource: WeaveResource, nameIdentifier: NameIdentifier, input: SafeStringBasedParserInput)

case class ParsingResult[T <: AstNode](input: ParsingContentInput, astNode: T) extends AstNodeResultAware[T] with ParsingContentAware

class ParsingResultWithDummyScopeNavigator[T <: AstNode](_input: ParsingContentInput, _astNode: T, val dummyScopesNavigator: ScopesNavigator) extends ParsingResult[T](_input, _astNode) with DummyScopeNavigatorAware

object ParsingPhase {
  def apply[T <: AstNode](rootRule: (ParsingContentInput, ParsingContext) => Either[ParseError, T]) = {
    new ParsingPhase[T](rootRule)
  }
}

