package org.mule.weave.v2.visibility

import org.mule.weave.v2.parser.AccessViolation
import org.mule.weave.v2.parser.DeprecatedFeature
import org.mule.weave.v2.parser.ExperimentalFeature
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.annotation.AnnotationCapableNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNodeHelper
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware
import org.mule.weave.v2.parser.phase.VerificationPhase

class AccessCheckPhase[R <: AstNode, T <: AstNodeResultAware[R] with ScopeNavigatorResultAware] extends VerificationPhase[R, T] {

  override def verify(source: T, context: ParsingContext): Unit = {
    AstNodeHelper.traverseChildren(
      source.astNode, {
        case vrn: VariableReferenceNode => {
          val variable = vrn.variable
          checkAccess(source, variable, context, vrn)
          true
        }
        case vrn: AnnotationNode => {
          val variable = vrn.name
          checkAccess(source, variable, context, vrn)
          true
        }
        case trn: TypeReferenceNode => {
          val variable = trn.variable
          checkAccess(source, variable, context, trn)
          true
        }
        case npn: NamespaceNode => {
          val variable = npn.prefix
          checkAccess(source, variable, context, npn)
          true
        }
        case _ => {
          true
        }
      })
  }

  private def checkAccess(source: T, variable: NameIdentifier, context: ParsingContext, astNode: AstNode): Unit = {
    val maybeReference = source.scope.resolveVariable(variable)
    maybeReference.foreach(ref => {
      ref.moduleSource match {
        case Some(moduleName) =>
          val maybeNode = ref.scope.astNavigator().parentWithType(ref.referencedNode, classOf[AnnotationCapableNode])
          maybeNode.foreach(annotationCapableNode => {
            annotationCapableNode.codeAnnotations.foreach(annotation => {
              ref.scope.resolveVariable(annotation.name) match {
                case Some(annotationRef) =>
                  val annotationModule = annotationRef.moduleSource.getOrElse(moduleName)
                  annotationModule.::(annotationRef.referencedNode.name) match {
                    case NameIdentifier.INTERNAL_ANNOTATION =>
                      if (!moduleName.name.equals(context.nameIdentifier.name)) {
                        validateInternalFunctionCall(variable, annotation, context, astNode, moduleName)
                      }
                    case NameIdentifier.EXPERIMENTAL_ANNOTATION =>
                      // Ignoring @Experimental warning on 'dw::' files
                      if (!NameIdentifier.isDataWeaveSdkFile(context.nameIdentifier)) {
                        if (context.nameIdentifier.loader.isDefined) {
                          val isInternalAnnotation = ref.fqnReferenceName == NameIdentifier.INTERNAL_ANNOTATION
                          // Ignoring @Experimental warning cause by @Internal on Custom Loaders
                          if (!isInternalAnnotation) {
                            context.messageCollector.warning(ExperimentalFeature(variable), astNode.location())
                          }
                        } else {
                          context.messageCollector.warning(ExperimentalFeature(variable), astNode.location())
                        }
                      }
                    case NameIdentifier.DEPRECATED_ANNOTATION =>
                      val deprecatedAnnotation: Map[String, String] = AnnotationNodeHelper.argsAsMap(annotation)
                      context.messageCollector.warning(DeprecatedFeature(ref.fqnReferenceName, deprecatedAnnotation.getOrElse("since", ""), deprecatedAnnotation.get("replacement")), astNode.location())
                    case _ =>
                  }
                case None =>
              }
            })
          })
        case _ =>
      }
    })
  }

  private def validateInternalFunctionCall(variable: NameIdentifier, annotation: AnnotationNode, context: ParsingContext, astNode: AstNode, moduleName: NameIdentifier): Unit = {
    val callerName: NameIdentifier = context.nameIdentifier
    val mayBePermits: Option[Seq[String]] = AnnotationNodeHelper.argStringSeq("permits", annotation)
    mayBePermits match {
      case Some(allowedCallers) => {
        val valid = allowedCallers.exists((allowedCaller) => callerName.name.startsWith(allowedCaller))
        if (!valid) {
          val allowedPatterns = if (allowedCallers.isEmpty) Seq(moduleName.name) else allowedCallers
          context.messageCollector.error(AccessViolation(variable, callerName.name, allowedPatterns), astNode.location())
        }
      }
      case None =>
    }
  }
}
