package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.parser.TypeCoercedMessage
import org.mule.weave.v2.parser.TypeMismatch
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.updates.ArrayIndexUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.AttributeNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.FieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.MultiFieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionNode
import org.mule.weave.v2.parser.ast.updates.UpdateNode
import org.mule.weave.v2.parser.ast.updates.UpdateSelectorNode
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.NamespaceType
import org.mule.weave.v2.ts.NothingType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.Edge
import org.mule.weave.v2.ts.EdgeLabels
import org.mule.weave.v2.ts.TypeCoercer
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.UriType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.WeaveTypeResolver
import org.mule.weave.v2.ts.updaters.AttributeFieldUpdater
import org.mule.weave.v2.ts.updaters.IndexFieldUpdater
import org.mule.weave.v2.ts.updaters.ObjectFieldUpdater
import org.mule.weave.v2.ts.updaters.RootFieldUpdater
import org.mule.weave.v2.ts.updaters.TerminalNodeUpdater
import org.mule.weave.v2.ts.updaters.TypeUpdater

import scala.collection.Seq

class UpdateNodeResolver(un: UpdateNode) extends WeaveTypeResolver {

  override def supportsPartialResolution(): Boolean = false

  def buildTree(sn: UpdateSelectorNode, index: Int, updater: TypeUpdater, node: TypeNode, conditional: Boolean, forceCreate: Boolean): Unit = {
    val maybeUpdater: Option[TypeUpdater] = sn match {
      case FieldNameUpdateSelectorNode(NameNode(nameNode, ns, _), _) => {
        val maybeUri: Option[String] = resolveNamespaceUri(node, ns)
        val selectQName = nameNode match {
          case StringNode(theSelector, _) => Some(QName(theSelector, maybeUri))
          case _                          => None
        }
        updater.children()
          .find({
            case updater: ObjectFieldUpdater => updater.maybeQName.equals(selectQName)
            case _                           => false
          }).orElse(Some(ObjectFieldUpdater(selectQName, updater, multiSelection = false)))
      }
      case FieldNameUpdateSelectorNode(_, _) => {
        None
      }
      case MultiFieldNameUpdateSelectorNode(NameNode(nameNode, ns, _), _) => {
        val maybeUri: Option[String] = resolveNamespaceUri(node, ns)
        val selectQName = nameNode match {
          case StringNode(theSelector, _) => Some(QName(theSelector, maybeUri))
          case _                          => None
        }
        updater.children()
          .find({
            case updater: ObjectFieldUpdater => updater.maybeQName.equals(selectQName)
            case _                           => false
          }).orElse(Some(ObjectFieldUpdater(selectQName, updater, multiSelection = true)))
      }
      case MultiFieldNameUpdateSelectorNode(_, _) => {
        None
      }
      case AttributeNameUpdateSelectorNode(NameNode(StringNode(name, _), ns, _), _) => {
        val maybeUri: Option[String] = resolveNamespaceUri(node, ns)
        val selectQName = QName(name, maybeUri)
        updater.children()
          .find({
            case updater: AttributeFieldUpdater => updater.attributeName.equals(selectQName)
            case _                              => false
          }).orElse(Some(AttributeFieldUpdater(selectQName, updater)))
      }
      case AttributeNameUpdateSelectorNode(_, _) => {
        None
      }
      case ArrayIndexUpdateSelectorNode(_, _) => {
        updater.children()
          .find({
            case _: IndexFieldUpdater => true
            case _                    => false
          }).orElse(Some(IndexFieldUpdater(updater)))
      }
    }

    maybeUpdater match {
      case Some(element) => {
        //Mark it force if required
        if (forceCreate) {
          element.forceCreate = true
        }

        sn.child match {
          case Some(child: UpdateSelectorNode) => {
            buildTree(child, index, element, node, conditional, forceCreate)
          }
          case _ => {
            TerminalNodeUpdater(index, conditional, forceCreate, element)
          }
        }
      }
      case None =>
    }
  }

  private def resolveNamespaceUri(node: TypeNode, ns: Option[AstNode]): Option[String] = {
    ns match {
      case Some(NamespaceNode(prefix)) => {
        val incomingNamespace = node.incomingType(EdgeLabels.NAMESPACE_PREFIX(prefix.name))
        incomingNamespace match {
          case Some(NamespaceType(_, Some(UriType(Some(uri))))) => Some(uri)
          case _ => None
        }
      }
      case _ => None
    }
  }

  override def resolveExpectedType(node: TypeNode, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): Seq[(Edge, WeaveType)] = Seq()

  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val expressions: Seq[UpdateExpressionNode] = un.matchers.expressions
    val updater = RootFieldUpdater()
    expressions.zipWithIndex.foreach((sn) => {
      sn._1.selector match {
        case usn: UpdateSelectorNode => buildTree(usn, sn._2, updater, node, sn._1.condition.isDefined, sn._1.forceCreate)
        case _                       =>
      }

    })
    val maybeExpressionType = node.incomingEdge(EdgeLabels.UPDATE_TO).flatMap((edge) => edge.mayBeIncomingType())
    maybeExpressionType.map((wt) => {
      updater.update(wt, (label, wt) => {
        resolveUpdateReturnType(label, wt, node, ctx)
      })
    })
  }

  private def resolveUpdateReturnType(label: String, valueType: WeaveType, node: TypeNode, ctx: WeaveTypeResolutionContext) = {
    val mayBeCondition = node.incomingType(EdgeLabels.CONDITION(label))
    if (mayBeCondition.isDefined) {
      val conditionType = mayBeCondition.get
      val condition = FunctionCallNodeResolver.resolveReturnType(conditionType.asInstanceOf[FunctionType], Seq(valueType, NumberType()), Seq(), node, ctx)
      condition match {
        case Some(actualType) => {
          actualType match {
            case BooleanType(value, _) => {
              //If it false then it is never going to be executed
              if (value.isDefined && !value.get) {
                Some(NothingType())
              } else {
                node.incomingType(EdgeLabels.FUNCTION(label)).flatMap((ft) => {
                  FunctionCallNodeResolver.resolveReturnType(ft.asInstanceOf[FunctionType], Seq(valueType, NumberType()), Seq(), node, ctx)
                })
              }
            }
            case _ => {
              if (!TypeHelper.canBeAssignedTo(actualType, BooleanType(), ctx)) {
                if (TypeCoercer.coerce(actualType, BooleanType(), ctx).isEmpty) {
                  ctx.error(TypeMismatch(BooleanType(), actualType), node.incomingEdge(EdgeLabels.CONDITION(label)).get.source, actualType.location())
                } else
                  ctx.warning(TypeCoercedMessage(BooleanType(), actualType), node.incomingEdge(EdgeLabels.CONDITION(label)).get.source)
              }
              node.incomingType(EdgeLabels.FUNCTION(label)).flatMap((ft) => {
                FunctionCallNodeResolver.resolveReturnType(ft.asInstanceOf[FunctionType], Seq(valueType, NumberType()), Seq(), node, ctx)
              })
            }
          }
        }
        case _ => {
          node.incomingType(EdgeLabels.FUNCTION(label)).flatMap((ft) => {
            FunctionCallNodeResolver.resolveReturnType(ft.asInstanceOf[FunctionType], Seq(valueType, NumberType()), Seq(), node, ctx)
          })
        }
      }
    } else {
      node.incomingType(EdgeLabels.FUNCTION(label)).flatMap((ft) => {
        FunctionCallNodeResolver.resolveReturnType(ft.asInstanceOf[FunctionType], Seq(valueType, NumberType()), Seq(), node, ctx)
      })
    }
  }
}
