package org.mule.weave.v2.ts.resolvers

import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.MessageKind
import org.mule.weave.v2.parser.TypePhaseCategory
import org.mule.weave.v2.parser.ast.functions.OverloadedFunctionNode
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.FunctionTypeParameter
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext
import org.mule.weave.v2.ts.WeaveTypeResolver

import scala.collection.Seq
import scala.collection.mutable.ArrayBuffer
import scala.collection.mutable.ListBuffer

object OverloadedFunctionTypeResolver extends WeaveTypeResolver {

  override def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType] = {
    val incomingTypes: Seq[WeaveType] = node.incomingTypes()
    val incomingFunctionTypes = new ListBuffer[FunctionType]()
    for (incomingType <- incomingTypes) {
      incomingType match {
        case x: FunctionType =>
          incomingFunctionTypes += x
        case _ =>
          val msg = Message(MessageKind.INVALID_OVERLOADED_FUNCTION_TYPE_MESSAGE_KIND, s"All members of an overloaded function should be functions, but found $incomingType", TypePhaseCategory)
          ctx.error(msg, node)
      }
    }

    val fnTypeParameters: Seq[FunctionTypeParameter] = getFunctionParams(incomingFunctionTypes)
    val name: Option[String] = getFunctionName(node)
    val returnTypes: Seq[WeaveType] = incomingFunctionTypes.map(_.returnType)
    val resolvedReturnType: WeaveType = TypeHelper.unify(returnTypes)
    Some(FunctionType(Seq(), fnTypeParameters, resolvedReturnType, incomingFunctionTypes, name = name))
  }

  def getFunctionName(node: TypeNode): Option[String] = {
    val overloadedFunctionNode: OverloadedFunctionNode = node.astNode.asInstanceOf[OverloadedFunctionNode]
    overloadedFunctionNode.functionDirectives.headOption.map(_.variable.name)
  }

  def getFunctionParams(functionNodes: Seq[FunctionType]): Seq[FunctionTypeParameter] = {
    val result = ArrayBuffer[FunctionTypeParameter]()
    val functionNodesByCardinality: Seq[FunctionType] = functionNodes.sortBy(_.params.length)
    val biggestCardinal: Seq[FunctionTypeParameter] = functionNodesByCardinality.last.params
    val maxAmountParams: Int = biggestCardinal.length
    var i = 0
    while (i < maxAmountParams) {
      val optional = functionNodes.exists((fun) => {
        val paramList = fun.params
        //Either one of the overloads has less params or one is optional
        paramList.size <= i || paramList.apply(i).optional
      })

      val paramTypes: Seq[WeaveType] = functionNodes.flatMap((nodeFunPair) => {
        val paramList = nodeFunPair.params
        if (paramList.length > i) {
          Some(paramList(i).wtype)
        } else {
          None
        }
      })
      //We use the name of the function with more parameters don't know if this is ok but it works ;)
      result.+=(FunctionTypeParameter(biggestCardinal(i).name, TypeHelper.unify(paramTypes), optional))
      i = i + 1
    }
    result
  }

}
