package org.mule.weave.v2.ts

import org.mule.weave.v2.grammar._
import org.mule.weave.v2.parser._
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.UndefinedExpressionNode
import org.mule.weave.v2.parser.ast.conditional.DefaultNode
import org.mule.weave.v2.parser.ast.conditional.IfNode
import org.mule.weave.v2.parser.ast.conditional.UnlessNode
import org.mule.weave.v2.parser.ast.functions._
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.header.directives.OutputDirective
import org.mule.weave.v2.parser.ast.logical.AndNode
import org.mule.weave.v2.parser.ast.logical.OrNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.operators.OpNode
import org.mule.weave.v2.parser.ast.patterns._
import org.mule.weave.v2.parser.ast.selectors.ExistsSelectorNode
import org.mule.weave.v2.parser.ast.selectors.NullSafeNode
import org.mule.weave.v2.parser.ast.selectors.NullUnSafeNode
import org.mule.weave.v2.parser.ast.structure._
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaPropertyNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.updates.ArrayIndexUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.AttributeNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.FieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.MultiFieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.UpdateNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.sdk.selectors.AttributeSelectorCustomTypeResolver
import org.mule.weave.v2.sdk.selectors.ValueSelectorCustomTypeResolver
import org.mule.weave.v2.ts.resolvers._

import scala.collection.Seq

/**
  * This are the resolvers of each language syntax element.
  */
trait WeaveTypeResolver {

  /**
    * Resolves the return type of the given node.
    *
    * @param node The node of the type graph
    * @param ctx  The context
    * @return The return type if it was able to resolve it
    */
  def resolveReturnType(node: TypeNode, ctx: WeaveTypeResolutionContext): Option[WeaveType]

  /**
    * This will try to infer the expected types of each incoming edge of the graph based on the expected output of this node
    *
    * @param node                 The node
    * @param incomingExpectedType The expected output of this node
    * @param ctx                  The context
    * @return The list of edge expected type pair
    */
  def resolveExpectedType(node: TypeNode, incomingExpectedType: Option[WeaveType], ctx: WeaveTypeResolutionContext): Seq[(Edge, WeaveType)] = {
    Seq()
  }

  /**
    * If it can resolve the output type with out requiring all the incoming edges to be resolved
    *
    * @return
    */
  def supportsPartialResolution(): Boolean = false
}

object WeaveTypeResolver {

  val UNSPECIFIED_PARAMETER_TYPE_PREFIX = "_Undefined"

  def apply(astNode: AstNode, typeReferenceResolver: WeaveTypeReferenceResolver, scope: ScopesNavigator, referenceResolver: ReferenceResolver): WeaveTypeResolver = {
    astNode match {
      case _: FieldNameUpdateSelectorNode          => new FieldBasedUpdateSelectorResolver(ValueSelectorCustomTypeResolver)
      case _: AttributeNameUpdateSelectorNode      => new FieldBasedUpdateSelectorResolver(AttributeSelectorCustomTypeResolver)
      case _: MultiFieldNameUpdateSelectorNode     => new FieldBasedUpdateSelectorResolver(ValueSelectorCustomTypeResolver)
      case _: ArrayIndexUpdateSelectorNode         => ArrayIndexUpdateSelectorResolver
      case un: UpdateNode                          => new UpdateNodeResolver(un)
      case _: DocumentNode                         => PassThroughTypeResolver
      case _: OutputDirective                      => PassThroughTypeResolver
      case _: NamespaceNode                        => PassThroughTypeResolver
      case _: VariableReferenceNode                => PassThroughTypeResolver
      case _: NameIdentifier                       => PassThroughTypeResolver
      case _: NullUnSafeNode                       => NullUnSafeTypeResolver
      case _: NullSafeNode                         => NullSafeTypeResolver
      case _: UsingNode                            => PassThroughTypeResolver
      case _: DoBlockNode                          => PassThroughTypeResolver
      case _: IfNode                               => IfElseResolver
      case _: UnlessNode                           => UnlessElseResolver
      case _: AndNode                              => AndTypeResolver
      case _: OrNode                               => OrTypeResolver
      case _: ErrorAstNode                         => ErrorAstNodeResolver
      case _: UndefinedExpressionNode              => LiteralTypeResolver(NothingType())
      case _: FunctionNode                         => new FunctionTypeResolver(typeReferenceResolver, referenceResolver)
      case _: OverloadedFunctionNode               => OverloadedFunctionTypeResolver
      case _: FunctionCallNode                     => FunctionCallNodeResolver
      case _: StringNode                           => StringTypeResolver
      case _: LocalDateNode                        => resolvers.LiteralTypeResolver(LocalDateType())
      case _: DateTimeNode                         => resolvers.LiteralTypeResolver(DateTimeType())
      case _: LocalDateTimeNode                    => resolvers.LiteralTypeResolver(LocalDateTimeType())
      case _: TimeNode                             => resolvers.LiteralTypeResolver(TimeType())
      case _: LocalTimeNode                        => resolvers.LiteralTypeResolver(LocalTimeType())
      case _: PeriodNode                           => resolvers.LiteralTypeResolver(PeriodType())
      case _: NumberNode                           => NumberTypeResolver
      case _: BooleanNode                          => BooleanTypeResolver
      case _: WeaveTypeNode                        => new TypeNodeTypeResolver(typeReferenceResolver)
      case _: UriNode                              => UriTypeResolver
      case _: TimeZoneNode                         => resolvers.LiteralTypeResolver(TimeZoneType())
      case _: RegexNode                            => resolvers.LiteralTypeResolver(RegexType())
      case _: NullNode                             => resolvers.LiteralTypeResolver(NullType())
      case _: ExistsSelectorNode                   => resolvers.LiteralTypeResolver(BooleanType())
      case _: DefaultNode                          => DefaultTypeResolver
      case _: ConditionalNode                      => ConditionalTypeResolver
      case node: OpNode if node.opId equals AsOpId => AsTypeResolver
      case node: OpNode if node.opId equals IsOpId =>
        val sourceRef = astNode match {
          case BinaryOpNode(_, sourceNode, _, _) => SourceReference(sourceNode, scope)
          case _                                 => None
        }

        new IsTypeResolver(sourceRef)
      case node: OpNode if (node.opId equals EqOpId) || (node.opId equals NotEqOpId) =>
        val lSource = astNode match {
          case BinaryOpNode(_, sourceNode, _, _) => SourceReference(sourceNode, scope)
          case _                                 => None
        }

        val rSource = astNode match {
          case BinaryOpNode(_, _, sourceNode, _) => SourceReference(sourceNode, scope)
          case _                                 => None
        }
        node.opId match {
          case EqOpId    => new EqTypeResolver(lSource, rSource)
          case NotEqOpId => new NotEqTypeResolver(lSource, rSource)
        }
      case node: OpNode                    => new OpNodeTypeResolver(node.opId)
      case _: ObjectNode                   => ObjectTypeResolver
      case _: NameValuePairNode            => NameValuePairTypeResolver
      case _: KeyValuePairNode             => KeyValuePairTypeResolver
      case _: NameNode                     => NameTypeResolver
      case _: DynamicNameNode              => NameTypeResolver
      case _: KeyNode                      => KeyTypeResolver
      case _: DynamicKeyNode               => KeyTypeResolver
      case _: ArrayNode                    => ArrayTypeResolver
      case _: HeadTailArrayNode            => HeadTailArrayTypeResolver
      case _: HeadTailObjectNode           => HeadTailObjectTypeResolver
      case _: PatternMatcherNode           => PatternMatcherTypeResolver
      case _: TypePatternNode              => TypePatternTypeResolver
      case _: RegexPatternNode             => RegexPatternTypeResolver
      case _: LiteralPatternNode           => LiteralPatternTypeResolver
      case _: ExpressionPatternNode        => ExpressionPatternTypeResolver
      case _: DefaultPatternNode           => DefaultPatternTypeResolver
      case _: EmptyArrayPatternNode        => EmptyArrayPatternTypeResolver
      case _: EmptyObjectPatternNode       => EmptyObjectPatternTypeResolver
      case _: DeconstructArrayPatternNode  => DeconstructArrayPatternTypeResolver
      case _: DeconstructObjectPatternNode => DeconstructObjectPatternTypeResolver
      case _: StringInterpolationNode      => StringInterpolationTypeResolver
      case node: NamespaceDirective        => new WeaveNamespaceTypeDirectiveResolver(node.prefix.name)
      case _: SchemaNode                   => ObjectTypeResolver
      case _: SchemaPropertyNode           => SchemePropertyResolver
    }
  }
}