package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.NamespaceDirective
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.types.TypeParameterNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.scope.Reference
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.ts.ConcreteTypeParamInstanceCounter.nextId
import org.mule.weave.v2.ts.WeaveTypeCloneHelper.copyLocation
import org.mule.weave.v2.utils.{ SynchronizedIdentityHashMap, ThreadSafe }

import java.util.concurrent.atomic.AtomicInteger

/**
  * Resolve a reference of a given type
  */
trait WeaveTypeReferenceResolver {

  def resolveType(refName: NameIdentifier, typeParams: Option[Seq[WeaveType]]): Option[WeaveType]

  def referenceTypeDef(refName: NameIdentifier): Option[(NameIdentifier, WeaveType, Option[Seq[TypeParameter]])]

  def resolveNamespace(prefix: NamespaceNode): Option[String]

  def scopesNavigator(): Option[ScopesNavigator]

}

object WeaveTypeReferenceResolver {

  def apply(scopesNavigator: ScopesNavigator): ScopeGraphTypeReferenceResolver = {
    new ScopeGraphTypeReferenceResolver(scopesNavigator)
  }
}

object EmptyWeaveTypeReferenceResolver extends WeaveTypeReferenceResolver {
  override def resolveType(refName: NameIdentifier, typeParams: Option[Seq[WeaveType]]): Option[WeaveType] = None

  override def referenceTypeDef(refName: NameIdentifier): Option[(NameIdentifier, WeaveType, Option[Seq[TypeParameter]])] = None

  override def resolveNamespace(prefix: NamespaceNode): Option[String] = None

  def scopesNavigator(): Option[ScopesNavigator] = None
}

/**
  * This class resolve type references. This class is being used for all Type from a given module.
  * As type resolution is lazy it needs to be ThreadSafe as different modules that depends on this may required to resolve this
  */
@ThreadSafe
class ScopeGraphTypeReferenceResolver(scopesNavigator: ScopesNavigator) extends WeaveTypeReferenceResolver {

  /**
    * This cache contains the WeaveType by the definition.
    * The WeaveTypeNode is the one on the TypeDirective, and then we apply if any the type parameters
    */
  private val resolvedTypes = SynchronizedIdentityHashMap[WeaveTypeNode, WeaveType]()

  /**
    * Abstract type parameters represent a variable (a wildcard). A concrete type parameter represents
    * an instance of this type parameter. For example a reference to it inside a body of a function
    *
    * fun test<T>(a: T): Array<T> =
    * [] << a
    *
    * In this case the type of `a` is an instance of `T` a concrete instance of T
    *
    * test("test")
    *
    * In this case the type of test (T) -> Array<T> `T` here is abstract as is a variable to be resolved on the apply of the function.
    */
  private val abstractTypeParamToConcreteTypeParam = SynchronizedIdentityHashMap[TypeParameter, TypeParameter]()

  /**
    * Given the refName it will resolve the reference from the scope and return a tuple with the fqnReferenceName, and the
    * declaration of the weave type and the type parameters.
    *
    * For instance:
    * import B from dw::module::B
    * type A = B<{value: String}>
    * When resolving B which is defined in module as:
    * type B<T> = { id: T }
    * This method will return the FQN reference name, and the WeaveType { id: T } with the T as type parameter not the one
    * declared when referenced (in A).
    */
  override def referenceTypeDef(refName: NameIdentifier): Option[(NameIdentifier, WeaveType, Option[Seq[TypeParameter]])] = {
    val typeReference: Option[Reference] = scopesNavigator.resolveVariable(refName)
    typeReference match {
      case Some(reference) =>
        val typeDeclarationNode: AstNode = resolveTypeDeclarationNode(reference)
        typeDeclarationNode match {
          case td: TypeDirective =>
            resolveReference(reference, None)
              .map(
                r =>
                  (
                    reference.fqnReferenceName,
                    r,
                    td.typeParametersListNode.map(
                      _.typeParameters.map(tp => WeaveType(tp, reference.scope.referenceResolver()).asInstanceOf[TypeParameter]))))
          case _ => None
        }
      case _ => None
    }
  }

  override def resolveType(refName: NameIdentifier, typeParams: Option[Seq[WeaveType]]): Option[WeaveType] = {
    val typeReference: Option[Reference] = scopesNavigator.resolveVariable(refName)
    typeReference match {
      case Some(reference) =>
        resolveReference(reference, typeParams)
      case _ => None
    }
  }

  override def scopesNavigator(): Option[ScopesNavigator] = Some(scopesNavigator)

  private def resolveReference(reference: Reference, typeArguments: Option[Seq[WeaveType]]) = {
    val typeDeclarationNode: AstNode = resolveTypeDeclarationNode(reference)

    typeDeclarationNode match {
      case td: TypeDirective =>
        val expression: WeaveTypeNode = td.typeExpression
        val name: String = td.variable.name
        // Resolve the reference from the proper reference resolver
        // Local reference points to this reference resolver
        val referencedType: WeaveType =
          if (reference.isCrossModule) {
            reference.scope.rootScope().referenceResolver().resolveType(expression, Some(name), td.weaveDoc.map(_.literalValue))
          } else {
            resolveType(expression, Some(name), td.weaveDoc.map(_.literalValue))
          }

        typeArguments match {
          case Some(typeArguments) =>
            td.typeParametersListNode match {
              case Some(typeParameters) =>
                val parameters: Seq[TypeParameterNode] = typeParameters.typeParameters
                // Replace the type parameters with the arguments
                // type User<T> = {name: T}
                // User<String> in here String is the argument and it needs to be replaced in name
                if (parameters.size == typeArguments.size) {
                  val argumentsByTypeName: Map[String, WeaveType] = parameters
                    .zip(typeArguments)
                    .map(tuple => (tuple._1.name.name, tuple._2))
                    .toMap
                  val typeParametersToReplace: Seq[TypeParameter] = TypeHelper.collectTypeParameters(referencedType)
                  val substitutions: Seq[(TypeParameter, WeaveType)] = typeParametersToReplace.flatMap(tp => {
                    argumentsByTypeName
                      .get(tp.name)
                      .map(arg => (tp, arg))
                  })

                  //Substitute the Type parameters with the type arguments
                  val context = new WeaveTypeResolutionContext(null)
                  val weaveType = Substitution(substitutions).apply(context, referencedType)
                  val params = substitutions.map(_._2.toString(prettyPrint = true, namesOnly = true)).mkString(", ")
                  Some(weaveType.label(s"$name<$params>"))
                } else {
                  None
                }
              case None => None
            }
          case None =>
            Some(referencedType)
        }
      case tp: TypeParameterNode =>
        Some(resolveTypeParameter(tp))
      case wtn: WeaveTypeNode =>
        Some(resolveType(wtn, None, None))
      case _ => None
    }
  }

  private def resolveTypeDeclarationNode(reference: Reference) = {
    val rootScope = reference.scope.rootScope()
    val referencedNode: NameIdentifier = reference.referencedNode
    val typeDeclarationNode: AstNode = rootScope.astNavigator().parentOf(referencedNode).get
    typeDeclarationNode
  }

  def resolveTypeParameter(tp: TypeParameterNode): WeaveType = {
    val typeParamType: WeaveType = resolveAbstractTypeParameter(tp)
    typeParamType match {
      case tp: TypeParameter => getConcreteTypeOf(tp)
      case _                 => typeParamType
    }
  }

  def resolveAbstractTypeParameter(tp: TypeParameterNode): WeaveType = {
    resolveType(tp, Some(tp.name.name), tp.weaveDoc.map(_.literalValue))
  }

  def getConcreteTypeOf(tp: TypeParameter): TypeParameter = {
    abstractTypeParamToConcreteTypeParam.getOrElseUpdate(
      tp, {
      val maybeTopType: Option[WeaveType] = tp.top.map(toConcreteTypeParam)
      val maybeBottomType: Option[WeaveType] = tp.bottom.map(toConcreteTypeParam)
      val copied = tp.copy(instanceId = Some(nextId()), top = maybeTopType, bottom = maybeBottomType)
      copyLocation(copied, tp.location())
      copied
    })
  }

  private def toConcreteTypeParam(weaveType: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(weaveType, {
      case typeParameter: TypeParameter => getConcreteTypeOf(typeParameter)
    })
  }

  private def toAbstractType(weaveType: WeaveType): WeaveType = {
    WeaveTypeTraverse.treeMap(weaveType, {
      case typeParameter: TypeParameter =>
        abstractTypeParamToConcreteTypeParam.find(_._2 eq typeParameter).map(_._1).get
    })
  }

  private def resolveType(expression: WeaveTypeNode, name: Option[String], wdoc: Option[String]): WeaveType = {
    resolvedTypes.getOrElseUpdate(
      expression, {
      val weaveType = WeaveType(expression, this).label(name).withDocumentation(wdoc, expression.location())
      weaveType match {
        case tp: TypeParameter =>
          //We need to ensure top and bottom types are abstract
          tp.copy(top = tp.top.map(toAbstractType), bottom = tp.bottom.map(toAbstractType))
        case wt => wt
      }
    })
  }

  override def resolveNamespace(namespaceRef: NamespaceNode): Option[String] = {
    val typeReference: Option[Reference] = scopesNavigator.resolveVariable(namespaceRef.prefix)
    typeReference match {
      case Some(reference) =>
        val maybeAstNode: Option[AstNode] = AstNodeHelper.find(
          reference.scope.astNode, {
            case namespaceDirective: NamespaceDirective if namespaceDirective.prefix.name.equals(reference.referencedNode.localName().name) => true
            case _ => false
          })
        maybeAstNode.map(namespaceDirective => namespaceDirective.asInstanceOf[NamespaceDirective].uri.literalValue)
      case _ => None
    }
  }
}

object ConcreteTypeParamInstanceCounter {
  var counter: AtomicInteger = new AtomicInteger(0)

  def nextId(): Number = {
    counter.incrementAndGet()
  }
}
