package org.mule.weave.v2.scope

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.header.directives.ImportedElement
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.ts.WeaveTypeReferenceResolver
import org.mule.weave.v2.utils.{ LazyValRef, SynchronizedIdentityHashMap, ThreadSafe }

import java.util.concurrent.ConcurrentHashMap

/**
  * This class represent a Variable Scope. For example a Do Block, and If section. An scope defines visibility of the variables.
  * This class needs to be thread safe
  */
@ThreadSafe
class VariableScope(
  val parsingContext: ParsingContext, //We try to capture variable scope as a Data Object
  val parentScope: Option[VariableScope],
  val name: Option[String],
  val astNode: AstNode,
  val index: Int,
  val references: Seq[NameIdentifier],
  val declarations: Seq[NameIdentifier],
  var importedModules: Seq[(ImportDirective, LazyValRef[Option[VariableScope]])]) {

  private var childScopes: Seq[VariableScope] = _
  private lazy val navigator = AstNavigator(astNode)
  private lazy val _referenceResolver: ScopeGraphTypeReferenceResolver = WeaveTypeReferenceResolver(scopesNavigator())
  private lazy val _scopesNavigator: ScopesNavigator = new ScopesNavigator(this)

  private val localReferenceCache = new ConcurrentHashMap[String, Option[Reference]]()
  private val referencesToCache = SynchronizedIdentityHashMap[NameIdentifier, Seq[Reference]]()

  /**
    * It sets the children this variable scope. As there is a double linking between parent and child we need this to be modifiable.
    * This should be called from a thread safe environment and only once
    * @param children The children to be set
    * @return This instance
    */
  def withChildren(children: Seq[VariableScope]): VariableScope = {
    if (childScopes == null) {
      this.synchronized({
        if (childScopes == null) {
          this.childScopes = children
        } else {
          throw new RuntimeException("Children already set, it can not be modified")
        }
      })
    }
    this
  }

  def astNavigator(): AstNavigator = {
    rootScope().navigator
  }

  def moduleName(): NameIdentifier = parsingContext.nameIdentifier

  /**
    * Returns the scope Navigator of this variable scope
    *
    * @return
    */
  def scopesNavigator(): ScopesNavigator = _scopesNavigator

  def referenceResolver(): ScopeGraphTypeReferenceResolver = _referenceResolver

  def collectVisibleVariables(collector: Reference => Boolean): Seq[Reference] = {
    val localAnnotations = collectLocalVariables(collector)
    val identifiers = importedModules.flatMap(im => {
      if (im._1.isImportStart()) {
        val maybeVariableScope = im._2.get()
        maybeVariableScope.map(vs => vs.collectLocalVariables(collector).map(_.copy(moduleSource = Some(im._1.importedModule.elementName))))
          .getOrElse(Seq.empty)
      } else {
        Seq.empty
      }
    })
    localAnnotations ++ identifiers
  }

  /**
    * Returns true if is the root scope meaning it has no parent
    * @return
    */
  def isRootScope(): Boolean = parentScope.isEmpty

  def collectLocalVariables(collector: Reference => Boolean): Seq[Reference] = {
    val references: Seq[Reference] = declarations.map(ni => Reference(ni, this)).filter(collector)
    val parentReferences = parentScope.map(ps => ps.collectLocalVariables(collector)).getOrElse(Seq())
    references ++ parentReferences
  }

  /**
    * Returns the child scopes
    * @return
    */
  def children(): Seq[VariableScope] = childScopes

  /**
    * Returns the Variables scope that doesn't have a parent. It will check if this is the root
    * if not it will go through its parent
    * @return The root scope
    */
  def rootScope(): VariableScope = parentScope.map(_.rootScope()).getOrElse(this)

  def resolveLocalReferenceTo(nameIdentifier: NameIdentifier): Seq[Reference] = {
    def resolveRef(vr: NameIdentifier): Option[Reference] = {
      resolveContextLocalVariable(vr) match {
        case Some(ref) if ref.referencedNode eq nameIdentifier => Some(Reference(vr, this))
        case _ => None
      }
    }

    val childRefs = childScopes.flatMap(_.resolveReferenceTo(nameIdentifier))
    val result = references.flatMap(ref => if (ref.name == nameIdentifier.name) resolveRef(ref) else None) ++ childRefs
    result
  }

  def shadowedVariables(): Seq[NameIdentifier] = {
    val identifiers = declarations
    identifiers.flatMap(identifier => parentScope.flatMap(_.searchDeclaration(identifier)))
  }

  private def searchDeclaration(identifier: NameIdentifier): Option[NameIdentifier] = {
    declarations
      .find(_.name.equals(identifier.name))
      .orElse({
        parentScope.flatMap(_.searchDeclaration(identifier))
      })
  }

  def resolveReferenceTo(nameIdentifier: NameIdentifier): Seq[Reference] = {
    def resolveRef(vr: NameIdentifier): Option[Reference] = {
      resolveVariable(vr) match {
        case Some(ref) if ref.referencedNode eq nameIdentifier =>
          Some(Reference(vr, this))
        case _ => None
      }
    }

    referencesToCache.getOrElseUpdate(
      nameIdentifier, {
      val localReferences: Seq[Reference] = references.flatMap(resolveRef)
      val childReferences: Seq[Reference] = childScopes.flatMap(_.resolveReferenceTo(nameIdentifier))
      localReferences ++ childReferences
    })
  }

  def resolveVariable(name: NameIdentifier): Option[Reference] = {
    //We search first in local scope
    //Then on parent scope
    //Last on import
    resolveContextLocalVariable(name)
      .orElse(resolveImportedVariable(name))
      .orElse(rootScope().resolveImportedVariable(name))
      .orElse(resolveFQNVariable(name))
  }

  /**
    * Returns true if that variable is defined in the parent scope
    *
    * @param name The name of the variable
    * @return True if it is defined
    */
  def isDefinedOnParentScope(name: NameIdentifier): Boolean = {
    parentScope.flatMap(_.resolveContextLocalVariable(name)).isDefined
  }

  private def resolveContextLocalVariable(name: NameIdentifier): Option[Reference] = {
    resolveLocalVariable(name)
      .orElse(parentScope.flatMap(_.resolveContextLocalVariable(name)))
  }

  def resolveLocalVariable(name: NameIdentifier): Option[Reference] = {
    localReferenceCache.computeIfAbsent(
      name.name,
      _ => {
        val resolvedLocalVariable: Option[NameIdentifier] = declarations.find(x => x.name == name.name)
        resolvedLocalVariable.map(Reference(_, this))
      })
  }

  private def resolveFQNVariable(name: NameIdentifier): Option[Reference] = {
    if (!name.isLocalReference()) {
      val moduleIdentifier: NameIdentifier = name.parent().get
      if (moduleIdentifier.equals(parsingContext.nameIdentifier)) {
        //Referencing  to this module
        resolveContextLocalVariable(name.localName())
      } else {
        val module = parsingContext.getScopeGraphForModule(moduleIdentifier)
        if (module.hasResult()) {
          val scope = module.getResult().scope.rootScope
          val variable = scope.resolveVariable(name.localName())
          toAbsoluteReference(variable, moduleIdentifier)
        } else {
          None
        }
      }
    } else {
      None
    }
  }

  /**
    * This method returns a copy of the reference with the moduleSource added
    */
  private def toAbsoluteReference(maybeReference: Option[Reference], moduleIdentifier: NameIdentifier): Option[Reference] = {
    maybeReference.map((ref: Reference) => ref.copy(moduleSource = Some(moduleIdentifier)))
  }

  private def resolveImportedVariable(name: NameIdentifier): Option[Reference] = {
    importedModules.toStream
      .flatMap(importModule => {
        val importDirective: ImportDirective = importModule._1
        val importScope = importModule._2
        val importedElements: Seq[AstNode] = importDirective.subElements.children()

        val maybeVariableScope = importScope.get()
        val maybeAbsoluteReference = if (maybeVariableScope.isDefined && name.isLocalReference() && importedElements.nonEmpty) {
          resolveLocalRefInImports(name, importDirective, maybeVariableScope.get)
        } else if (maybeVariableScope.isDefined && importedElements.isEmpty) {
          resolveRefWithModuleName(name, importDirective, maybeVariableScope.get)
        } else {
          None
        }
        maybeAbsoluteReference
      })
      .headOption
  }

  /**
    * Goes through all the elements in the imported module checking if one matches the name to resolve (star always matches) and returns an absolute reference
    *
    */
  private def resolveLocalRefInImports(nameToResolve: NameIdentifier, importDirective: ImportDirective, importScope: VariableScope): Option[Reference] = {
    val moduleIdentifier: NameIdentifier = importDirective.importedModule.elementName
    val importedElements: Seq[AstNode] = importDirective.subElements.children()

    importedElements.toStream
      .flatMap({
        case ImportedElement(NameIdentifier("*", None), None) =>
          val variable: Option[Reference] = importScope.resolveLocalVariable(nameToResolve)
          toAbsoluteReference(variable, moduleIdentifier)
        case ImportedElement(elementName, alias) =>
          val matchesImportedElemName = alias.getOrElse(elementName).name.equals(nameToResolve.name)
          if (matchesImportedElemName) {
            toAbsoluteReference(importScope.resolveLocalVariable(elementName), moduleIdentifier)
          } else {
            None
          }
      })
      .headOption
  }

  /**
    * This resolves references with the form moduleName::name
    */
  private def resolveRefWithModuleName(nameToResolve: NameIdentifier, importDirective: ImportDirective, importScope: VariableScope) = {
    val moduleIdentifier: NameIdentifier = importDirective.importedModule.elementName

    nameToResolve.parent() match {
      case Some(parentName) if parentName.parent().isEmpty =>
        val matchesImportedModuleName = getLocalModuleName(importDirective).equals(parentName.localName())
        if (matchesImportedModuleName) {
          val nameIdentifier = importScope.resolveLocalVariable(nameToResolve.localName())
          toAbsoluteReference(nameIdentifier, moduleIdentifier)
        } else {
          None
        }
      case _ => None
    }
  }

  private def getLocalModuleName(importDirective: ImportDirective): NameIdentifier = {
    importDirective.importedModule.aliasOrLocalName
  }

  def localVariables: Seq[Reference] = declarations.map(Reference(_, this))

  private def importedVariables: Seq[Reference] = {
    importedModules.filter(im => im._2.get().isDefined).flatMap {
      case (importDirective, importScope) =>
        val moduleName = importDirective.importedModule.elementName
        importDirective.subElements.children().flatMap {
          case ImportedElement(NameIdentifier("*", None), None) =>
            importScope.get().get.visibleVariables.map(_.copy(moduleSource = Some(moduleName)))
          case importedElement: ImportedElement =>
            Seq(Reference(importedElement.aliasOrLocalName, importScope.get().get, Some(moduleName)))
        }
    }
  }

  def visibleVariables: Seq[Reference] = {
    val parentDeclarations = parentScope match {
      case Some(parent) => parent.visibleVariables
      case _            => Nil
    }
    val importedVars: Map[String, Reference] = importedVariables.map(x => x.referencedNode.name -> x).toMap
    val parentVarMap: Map[String, Reference] = parentDeclarations.map(x => x.referencedNode.name -> x).toMap
    val localVarMap: Map[String, Reference] = localVariables.map(x => x.referencedNode.name -> x).toMap
    val visibleVars: Seq[Reference] = (localVarMap ++ importedVars ++ parentVarMap).values.toSeq
    visibleVars
  }

}
