package org.mule.weave.v2.scaffolding

import org.mule.weave.v2.scaffolding.RandomPicker.pickRandom
import org.mule.weave.v2.scaffolding.RandomPicker.randomInt

import java.util.Base64
import java.util.UUID

/**
  * Generates Random data for a given kind of data
  */
trait SampleDataGenerator {

  def generateString(): String = {
    pickRandom(DataConstants.words)
  }

  def generateNumber(): Number = {
    randomInt(100)
  }

  def generateBinary(): String = {
    s""""${generateString()}" as Binary"""
  }

  def handles(): Seq[String]

  def handles(text: String): Boolean = {
    handles().contains(text)
  }
}

object SampleDataGenerator {

  val generators = Seq(
    SSNGenerator,
    GenderGenerator,
    UuidGenerator,
    FullNameGenerator,
    UserNameGenerator,
    DescriptionGenerator,
    EmailGenerator,
    LastNameGenerator,
    TitleDataGenerator,
    CreditCardGenerator,
    AddressGenerator,
    ZipcodeGenerator,
    RateGenerator,
    AgeGenerator,
    CountyGenerator,
    CityGenerator,
    PriceGenerator,
    UrlGenerator,
    NameGenerator,
    PhoneNumberGenerator,
    PDFGenerator,
    ImageGenerator,
    ZipFileGenerator,
    IdGenerator //This should be the last one
  )

  def pick(name: String): SampleDataGenerator = {
    val sanitized = name.toLowerCase.replaceAll("_|-|\\s", "")
    val maybeGenerator = generators.find(_.handles(sanitized))
    maybeGenerator.getOrElse({
      val maybeGenerator = generators
        .map((g) => {
          val distance = g
            .handles()
            .map((w) =>
              if (sanitized.contains(w)) {
                sanitized.length - w.length
              } else {
                Integer.MAX_VALUE
              })
            .min
          (g, distance)
        })
        .minBy(_._2)
      if (maybeGenerator._2 < 6) {
        maybeGenerator._1
      } else {
        DefaultDataGenerator
      }
    })
  }
}

object AgeGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    s"${randomInt(10, 100)}"
  }

  override def generateNumber(): Number = {
    randomInt(10, 100)
  }

  override def handles(): Seq[String] = Seq("age")
}

object RateGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    s"${randomInt(0, 100)}%"
  }

  override def generateNumber(): Number = {
    RandomPicker.randomDouble()
  }

  override def handles(): Seq[String] = Seq("rate", "percentage")
}

object UrlGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    s"http://acme.com/${randomInt(1000000, 9999999)}"
  }

  override def handles(): Seq[String] = {
    Seq("url")
  }

  override def handles(text: String): Boolean = {
    super.handles(text) || text.endsWith("url")
  }
}

object PhoneNumberGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    s"(${randomInt(1, 8)}${randomInt(8)}${randomInt(8)})-${randomInt(100, 743)}-${randomInt(1000, 9999)}"
  }

  override def handles(): Seq[String] = Seq("phone", "fax", "cellphone", "faxnumber", "cell", "phonenumber", "telephone", "mobile", "mobilephone", "otherphone")

  override def handles(text: String): Boolean = {
    super.handles(text) || text.endsWith("phone")
  }
}

object ZipcodeGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    randomInt(10000, 20000).toString
  }

  override def handles(): Seq[String] = Seq("zipcode")
}

object AddressGenerator extends SampleDataGenerator {

  val adresses = Seq("First Ave", "Second Ave", "Third St", "Waller St", "Loyd St", "Castro St", "Golden Gate Ave", "Geary St", "Scaramento St")

  override def generateString(): String = {
    pickRandom(adresses) + " " + randomInt(10, 100)
  }

  override def handles(): Seq[String] = Seq("address", "street")
}

object CreditCardGenerator extends SampleDataGenerator {

  val numbers = Seq("4929569929839458", "4716544672184151", "5445010187495727", "5449062304684754", "5508471298673913", "378785899909535", "377691167980243", "4485567993395817")

  override def generateString(): String = {
    pickRandom(numbers)
  }

  override def handles(): Seq[String] = Seq("creditcard")
}

object SSNGenerator extends SampleDataGenerator {

  private val SSNPrefixes = (900 to 999).toArray :+ 666

  override def generateString(): String = {
    val first = pickRandom(SSNPrefixes)
    val second = RandomPicker.randomInt(10, 99)
    val third = RandomPicker.randomInt(1000, 9999)
    s"$first-$second-$third"
  }

  override def handles(): Seq[String] = {
    Seq("ssn")
  }
}

object GenderGenerator extends SampleDataGenerator {

  private val GENDERS = Seq("male", "female")

  override def generateString(): String = {
    pickRandom(GENDERS)
  }

  override def handles(): Seq[String] = {
    Seq("gender", "sex")
  }

}

object CityGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    pickRandom(DataConstants.CITIES)
  }

  override def handles(): Seq[String] = {
    Seq("city", "cityname", "province")
  }
}

import java.math.BigDecimal

object PriceGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    generateNumber().toString
  }

  override def generateNumber(): Number = {
    val price = RandomPicker.randomFloat(0.5f, 100)
    var bd: BigDecimal = new BigDecimal(price)
    bd = bd.setScale(2, BigDecimal.ROUND_HALF_UP)
    bd.floatValue()
  }

  override def handles(): Seq[String] = {
    Seq("price", "value")
  }
}

object CountyGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    pickRandom(DataConstants.COUNTRY)
  }

  override def handles(): Seq[String] = {
    Seq("country")
  }
}

object UuidGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    UUID.randomUUID().toString
  }

  override def handles(): Seq[String] = {
    Seq("uuid")
  }
}

object IdGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    UUID.randomUUID().toString.replace("-", "").substring(0, 19)
  }

  override def handles(): Seq[String] = {
    Seq("id")
  }

  override def handles(text: String): Boolean = {
    super.handles(text) || text.endsWith("id")
  }
}

object NameGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    pickRandom(DataConstants.NAMES)
  }

  override def handles(): Seq[String] = {
    Seq("name", "firstname")
  }
}

object FullNameGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    s"${pickRandom(DataConstants.NAMES)} ${pickRandom(DataConstants.LASTNAMES)}"
  }

  override def handles(): Seq[String] = Seq("fullname", "author", "assistantname")
}

object UserNameGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    s"${pickRandom(DataConstants.NAMES)}.${pickRandom(DataConstants.LASTNAMES)}"
  }

  override def handles(): Seq[String] = Seq("username", "userid")
}

object DescriptionGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    val length = randomInt(20)
    var i = 0
    var result = ""
    while (i < length) {
      result += pickRandom(DataConstants.words) + " "
      i = i + 1
    }
    result
  }

  override def handles(): Seq[String] = Seq("description", "note")
}

object EmailGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    s"${pickRandom(DataConstants.NAMES)}.${pickRandom(DataConstants.LASTNAMES)}@gmail.com"
  }

  override def handles(): Seq[String] = Seq("email")
}

object LastNameGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    pickRandom(DataConstants.LASTNAMES)
  }

  override def handles(): Seq[String] = Seq("lastname", "surname", "familyname")
}

object DefaultDataGenerator extends SampleDataGenerator {

  override def handles(): Seq[String] = Seq()
}

object TitleDataGenerator extends SampleDataGenerator {

  override def generateString(): String = {
    pickRandom(DataConstants.TITLES)
  }

  override def handles(): Seq[String] = Seq("title")
}

trait BinarySampleDataGenerator extends SampleDataGenerator {
  val resource: String

  override def generateBinary(): String = {
    val stream = getClass.getClassLoader.getResourceAsStream(resource)
    val bytes = Stream.continually(stream.read).takeWhile(_ != -1).map(_.toByte).toArray
    val value = Base64.getEncoder.encodeToString(bytes)

    s""""$value" as Binary {base: "64"}"""
  }

}

object PDFGenerator extends BinarySampleDataGenerator {
  override val resource: String = "org/mule/weave/v2/scaffolding/dataweave_pdf.pdf"

  override def handles(): Seq[String] = Seq("pdf", "document")
}

object ImageGenerator extends BinarySampleDataGenerator {
  override val resource: String = "org/mule/weave/v2/scaffolding/dataweave_icon.png"

  override def handles(): Seq[String] = Seq("icon", "favicon", "image", "png")
}

object ZipFileGenerator extends BinarySampleDataGenerator {
  override val resource: String = "org/mule/weave/v2/scaffolding/dataweave_icon.zip"

  override def handles(): Seq[String] = Seq("zip", "compressed")
}
