package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.IncorrectTypeForTypeSelection
import org.mule.weave.v2.parser.InterpolationInTypeSelector
import org.mule.weave.v2.parser.TypeSelectorKeyNotFound
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.StringInterpolationNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.types.TypeSelectorNode
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeSelectorType
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeReferenceResolver

/**
  * Validates that a type selector reference can be executed
  *
  * @tparam T
  */
class TypeSelectorValidation[R <: AstNode, T <: AstNodeResultAware[R] with ScopeNavigatorResultAware]() extends VerificationPhase[R, T] {

  def checkSelector(actualType: WeaveType, ot: ObjectType, nameNode: NameNode, selectorNode: String, referenceResolver: WeaveTypeReferenceResolver, context: ParsingContext): Unit = {
    nameNode match {
      case NameNode(StringInterpolationNode(_), _, _) => context.messageCollector.error(InterpolationInTypeSelector(), nameNode.location())
      case NameNode(StringNode(_, _), nsPrefix: Option[NamespaceNode], _) => {
        val propertyPairs = TypeHelper.selectPropertyPairs(ot, QName(selectorNode, nsPrefix.flatMap(referenceResolver.resolveNamespace)))
        if (propertyPairs.isEmpty) {
          context.messageCollector.error(TypeSelectorKeyNotFound(selectorNode, actualType), nameNode.location())
        }
      }
    }

  }

  def checkReferencedTypeAndSelector(actualType: WeaveType, resolvedType: WeaveType, nameNode: NameNode, selectorNode: String, referenceResolver: WeaveTypeReferenceResolver, context: ParsingContext): Unit = {
    resolvedType match {
      case ot: ObjectType    => checkSelector(actualType, ot, nameNode, selectorNode, referenceResolver, context)
      case rt: ReferenceType => checkReferencedTypeAndSelector(rt, rt.resolveType(), nameNode, selectorNode, referenceResolver, context)
      case UnionType(of) => of.foreach {
        checkReferencedTypeAndSelector(actualType, _, nameNode, selectorNode, referenceResolver, context)
      }
      case it: IntersectionType => checkReferencedTypeAndSelector(actualType, TypeHelper.simplifyIntersections(it), nameNode, selectorNode, referenceResolver, context)
      case _                    => context.messageCollector.error(IncorrectTypeForTypeSelection(actualType), actualType.location())
    }
  }

  override def verify(source: T, context: ParsingContext): Unit = {
    val typeSelectorNodes: Seq[TypeSelectorNode] = source.scope.astNavigator().allWithType(classOf[TypeSelectorNode])
    typeSelectorNodes.foreach(typeSelector => {
      val weaveType: WeaveType =
        WeaveType(typeSelector, source.scope.referenceResolver)
      weaveType match {
        case tst: TypeSelectorType => checkReferencedTypeAndSelector(tst.referencedType, tst.referencedType.resolveType(), typeSelector.selector, tst.refName, source.scope.referenceResolver, context)
        case _: WeaveType          => {}
      }
    })
  }
}
