package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.{ CyclicWeaveImport, MessageCollector, SelfImportingModule, UnableToResolveModule }
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.DoBlockNode
import org.mule.weave.v2.parser.ast.header.directives.ImportDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

class ImportsValidation[R <: AstNode, T <: AstNodeResultAware[R] with ScopeNavigatorResultAware] extends VerificationPhase[R, T] {

  private def checkImports(imports: Seq[ImportDirective], context: ParsingContext): Unit = {
    imports.map(_.importedModule.elementName).distinct.foreach((importedModuleName) => {
      // We ignore if we are importing the same file
      if (!context.nameIdentifier.equals(importedModuleName)) {
        val mayBeModule = context.tryToParseModule(importedModuleName)
        if (mayBeModule.isEmpty) {
          context.messageCollector.error(UnableToResolveModule(importedModuleName), importedModuleName.location())
        } else {
          context.messageCollector.mergeWith(mayBeModule.get.messages())
        }
      }
    })
  }

  private def checkCircularImports(importsVisited: Seq[NameIdentifier], astNode: AstNode, currentNameIdentifier: NameIdentifier, messageCollector: MessageCollector, context: ParsingContext): Unit = {
    val visited = importsVisited :+ currentNameIdentifier
    val directives = astNode match {
      case DocumentNode(header, _) => {
        header.directives.collect({ case d: ImportDirective => d })
      }
      case ModuleNode(_, nodes) => {
        nodes.collect({ case d: ImportDirective => d })
      }
      case DoBlockNode(header, _, _) =>
        header.directives.collect({ case d: ImportDirective => d })
      case _ => Seq.empty
    }
    directives.foreach(d => {
      val importIdentifier = d.importedModule.elementName
      if (importIdentifier == currentNameIdentifier) {
        context.messageCollector.error(SelfImportingModule(importIdentifier), importIdentifier.location())
      } else if (visited.contains(importIdentifier)) {
        context.messageCollector.error(CyclicWeaveImport(importIdentifier, visited.dropWhile(ni => ni != importIdentifier) :+ importIdentifier), importIdentifier.location())
      } else {
        context.tryToParseModule(importIdentifier)
          .map(module => module.mayBeResult
            .map(result => {
              val astNode = result.astNode
              checkCircularImports(visited, astNode, importIdentifier, messageCollector, context)
              context.messageCollector.mergeWith(module.messages())
            }))
      }
    })
  }

  override def verify(source: T, context: ParsingContext): Unit = {
    val imports = Seq[NameIdentifier]()

    source.astNode match {
      case DocumentNode(header, _) => {
        checkCircularImports(imports, source.astNode, context.nameIdentifier, context.messageCollector, context)
        checkImports(
          header.directives.collect({ case d: ImportDirective => d }),
          context)
      }
      case ModuleNode(_, nodes) => {
        checkCircularImports(imports, source.astNode, context.nameIdentifier, context.messageCollector, context)
        checkImports(
          nodes.collect({ case d: ImportDirective => d }),
          context)
      }
      case DoBlockNode(header, _, _) =>
        checkCircularImports(imports, source.astNode, context.nameIdentifier, context.messageCollector, context)
        checkImports(
          header.directives.collect({ case d: ImportDirective => d }),
          context)
      case _ =>
    }
  }
}