package org.mule.weave.v2.parser.location

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

@WeaveApi(Seq("data-weave-agent"))
trait Location {
  def locationString: String
}

object Location {
  def apply(location: String) = new SimpleLocation(location)
}

@WeaveApi(Seq("data-weave-agent"))
class SimpleLocation(location: String) extends Location {
  override def locationString: String = location

}

object SimpleLocation {
  def apply(location: String): SimpleLocation = new SimpleLocation(location)
}

@WeaveApi(Seq("data-weave-agent"))
object UnknownLocation extends WeaveLocation(UnknownPosition, UnknownPosition, NameIdentifier.anonymous) {
  override def locationString: String = "Unknown location"
}

trait IndexedLocation extends Location {
  def index: Long
}

class NoSourceLocation(override val startPosition: NoSourcePosition, override val endPosition: NoSourcePosition, override val resourceName: NameIdentifier)
    extends WeaveLocation(startPosition, endPosition, resourceName) {
  override def noErrorMarkerLocationString: String = ""
  override def locationString = ""
  override def source: () => String = () => ""
  override def sourceLines(): Seq[String] = Seq()
}

object NoSourceLocation {
  def fromLocation(location: WeaveLocation): NoSourceLocation = {
    location match {
      case UnknownLocation =>
        val pos = NoSourcePosition(0, 0, 0)
        new NoSourceLocation(pos, pos, NameIdentifier.ANONYMOUS_NAME)
      case nsl: NoSourceLocation => nsl
      case location =>
        val sPos = NoSourcePosition(location.startPosition.index, location.startPosition.line, location.startPosition.column)
        val ePos = NoSourcePosition(location.endPosition.index, location.endPosition.line, location.endPosition.column)
        new NoSourceLocation(sPos, ePos, location.resourceName)
    }
  }
}