package org.mule.weave.v2.grammar.structure

import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.grammar.Selectors
import org.mule.weave.v2.grammar.Tokens
import org.mule.weave.v2.grammar.Variables
import org.mule.weave.v2.grammar.literals.Literals
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.annotation.SingleKeyValuePairNodeAnnotation
import org.mule.weave.v2.parser.annotation.TrailingCommaAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.structure._
import org.parboiled2._

trait Object extends PositionTracking with Tokens with Variables with Selectors with Literals with Namespaces with Attributes {
  this: Object with Grammar =>

  def expr: Rule1[AstNode]

  def key: Rule1[KeyNode] = namedRule("Key") {
    pushPosition ~ (annotations ~ ws ~ namespace ~ (nameString | string | invalidFieldNameDeclaration) ~ attributes ~> createKeyNode) ~ injectPosition
  }

  val createKeyNode = (codeAnnotations: Seq[AnnotationNode], ns: Option[NamespaceNode], name: AstNode, attributes: Option[AttributesNode]) => {
    KeyNode(name, ns, attributes, codeAnnotations)
  }

  val createDynamicKeyNode = (name: AstNode, attributes: Option[AttributesNode]) => {
    val node = DynamicKeyNode(name, attributes)
    doInjectBinaryPosition(node)
    node
  }

  val createEmptyObjectNode = () => {
    ObjectNode(Seq())
  }

  val createObjectNode = (head: AstNode, members: Option[Seq[AstNode]], hasTrailingComma: Boolean) => {
    val node = ObjectNode(head +: members.getOrElse(Seq()))
    if (hasTrailingComma) {
      TrailingCommaAnnotation.annotate(node)
    }
    node
  }

  val createHeadTailObjectNode = (head: AstNode, tail: AstNode) => {
    val kvp = head.asInstanceOf[KeyValuePairNode]
    HeadTailObjectNode(kvp.key, kvp.value, tail)
  }

  val createSingleKeyValueObjectNode = (member: KeyValuePairNode) => {
    val result = ObjectNode(Vector(member))
    result.annotate(SingleKeyValuePairNodeAnnotation())
    result
  }

  val createConditionalKeyValuePairNode = (key: AstNode, value: AstNode, expr: AstNode) => {
    KeyValuePairNode(key, value, Some(expr))
  }

  protected val createConditionalKeyValuePair = (expr: AstNode, condition: AstNode) => {
    val objectNode = expr.asInstanceOf[ObjectNode]
    val keyValuePairNode = objectNode.elements.head.asInstanceOf[KeyValuePairNode]
    createConditionalKeyValuePairNode(keyValuePairNode.key, keyValuePairNode.value, condition)
  }

  def objectKVExpression = rule {
    pushPosition ~ objectExpansion ~
      optional(
        dynamicKeyValuePair |
          conditionalKeyValuePair) ~ injectPosition
  }

  def dynamicKeyValuePair = rule {
    (dynamicKeyExpr ~!~ (expr | missingExpression("Missing Object Field Expression. e.g {(myVar): 123}")) ~> createKeyValuePairNode) ~ injectBinaryPosition
  }

  def dynamicKeyExpr = rule {
    attributes ~ ws ~ objFieldSep ~> createDynamicKeyNode
  }

  def conditionalKeyValuePair = rule {
    (test(isSingleKeyValuePair(valueStack.peek)) ~ ws ~ ifKeyword ~!~ ((ws ~ ifCondition) | (fws ~ expr)) ~> createConditionalKeyValuePair) ~ injectBinaryPosition
  }

  def multipleKeyValuePairObj: Rule1[AstNode] = namedRule("Object Expression e.g {name: 'DW Rules'}") {
    pushPosition ~ (curlyBracketsStart ~!~ (emptyObject | objectExpressions)) ~ injectPosition
  }

  private def emptyObject = namedRule("}") {
    curlyBracketsEnd ~> createEmptyObjectNode
  }

  private def objectExpressions = namedRule("Object Expression. e.g {a: 123}") {
    objectElement ~!~ (headTailObject | objectElements)
  }

  def headTailObject = namedRule("Object Composition e.g {key: 123 ~ tailKVExpression()}") {
    (tilde ~ test(isSimpleKvp) ~!~ expr ~!~ (curlyBracketsEnd | fail("'}' for the Head Tail expression. e.g {key: 123 ~ tailKVExpression()}"))) ~> createHeadTailObjectNode
  }

  private def isSimpleKvp = {
    valueStack.peek.isInstanceOf[KeyValuePairNode] &&
      valueStack.peek.asInstanceOf[KeyValuePairNode].cond.isEmpty
  }

  def objectElements = namedRule(",") {
    (optional(commaSep ~ zeroOrMore(objectElement).separatedBy(commaSep)) ~ quiet(commaSep ~ push(true) | push(false)) ~!~ (curlyBracketsEnd | fail("`}` or ',' for the object expression."))) ~> createObjectNode
  }

  def objectElement = namedRule("Key Value Pair e.g {a: 123}") {
    staticKeyValuePair | objectKVExpression
  }

  def objectExpansion = namedRule("Dynamic Object Expression e.g { (vars.foo), name: 'test'}") {
    pushPosition ~ parenStart ~ expr ~ (parenEnd | fail("')' for object expression.")) ~> markEnclosedNode
  }

  private def staticKeyValuePair: Rule1[KeyValuePairNode] = namedRule("Key Value Pair e.g a: 123") {
    pushPosition ~ (key ~ ws ~ atomic(objFieldSep ~ !objFieldSep) ~!~ (expr | missingExpression("Missing Object Field Expression. e.g  {a: 123}")) ~> createKeyValuePairNode) ~ injectPosition
  }

  val createKeyValuePairNode: (AstNode, AstNode) => KeyValuePairNode = (key: AstNode, value: AstNode) => {
    KeyValuePairNode(key, value)
  }

  def singleKeyValuePairObj: Rule1[ObjectNode] = namedRule("Single Key Value Object. e.g a: 123") {
    pushPosition ~ !curlyBracketsStart ~ staticKeyValuePair ~> createSingleKeyValueObjectNode ~ injectPosition
  }

}
