package org.mule.weave.v2.grammar.literals

import org.mule.weave.v2.grammar.Grammar
import org.mule.weave.v2.grammar.Tokens
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.structure.{ DateTimeNode, LocalDateNode, LocalDateTimeNode, PeriodNode, _ }
import org.parboiled2.CharPredicate.Digit
import org.parboiled2._

trait DateLiteral extends PositionTracking with Tokens with StringBuilding with IntegerLiteral {
  this: DateLiteral with Parser =>

  import org.parboiled2.CharPredicate.Alpha

  val alphaUnderscoreSlash = Alpha ++ CharPredicate("/_-")

  val digit01 = CharPredicate('0' to '1')
  val digit02 = CharPredicate('0' to '2')
  val digit04 = CharPredicate('0' to '4')
  val digit12 = CharPredicate('1' to '2')
  val digit13 = CharPredicate('1' to '3')
  val digit15 = CharPredicate('1' to '5')
  val digit09 = CharPredicate('0' to '9')
  val digit19 = CharPredicate('1' to '9')
  val digit14 = CharPredicate('1' to '4')
  val digit03 = CharPredicate('0' to '3')
  val digit17 = CharPredicate('1' to '7')
  val createDateTimeNode = (dateTime: String) => {
    DateTimeNode(dateTime)
  }
  val createDateTimeNodeWithOptionalTimeZone = (date: String, time: Option[String], timeZone: Option[String]) => {
    time match {
      case Some(t) => {
        timeZone match {
          case Some(tz) => createDateTimeNode(date + t + tz)
          case _        => createLocalDateTimeNode(date + t)
        }
      }
      case _ => {
        createLocalDateNode(date)
      }
    }
  }
  val createLocalDateTimeNode = (dateTime: String) => {
    LocalDateTimeNode(dateTime)
  }
  val createTimeNodeWithOptionalTimeZone = (time: String, timeZone: Option[String]) => {
    timeZone match {
      case Some(tz) => createTimeNode(time + tz)
      case _        => createLocalTimeNode(time)
    }
  }

  val createLocalTimeNode = (time: String) => {
    LocalTimeNode(time)
  }
  val createTimeNode = (time: String) => {
    TimeNode(time)
  }
  val createTimeZoneNode = (timeZone: String) => {
    TimeZoneNode(timeZone)
  }
  val createLocalDateNode = (dateTime: String) => {
    LocalDateNode(dateTime)
  }

  val createPeriodNode = (period: String) => {
    PeriodNode(period)
  }

  def anyDateLiteral: Rule1[AstNode] = namedRule("Date") {
    pushPosition ~ (dateSeparator ~!~ (anyDateExpression) ~!~ dateSeparator) ~ injectPosition
  }

  def anyDateExpression: Rule1[AstNode] = rule {
    (periodLiteral | dateTimeLiteral | timeLiteral | timeZoneLiteral | MISMATCH)
  }

  def dateTimeLiteral: Rule1[AstNode] = namedRule("DateTime<YYYY-MM-DD('T'HH:mm:ss)?('Z')?>") {
    capture(date) ~!~ optional(timeExpr) ~ optional(capture(timeZone)) ~> createDateTimeNodeWithOptionalTimeZone
  }

  def timeExpr = namedRule("Time<'T'HH:mm:ss>") {
    capture(ch('T') ~!~ localTime)
  }

  def timeLiteral: Rule1[AstNode] = namedRule("Time<HH:mm:ss>") {
    capture(localTime) ~ optional(capture(timeZone)) ~> createTimeNodeWithOptionalTimeZone
  }

  def timeZoneLiteral: Rule1[TimeZoneNode] = namedRule("TimeZone<'Z'|+-HH:mm>") {
    atomic(capture(timeZone) ~> createTimeZoneNode)
  }

  def calendarDateLiteral: Rule1[LocalDateNode] = namedRule("Date") {
    capture(year ~ ch('-') ~ mm ~ optional(ch('-') ~ day)) ~> createLocalDateNode
  }

  def periodLiteral: Rule1[PeriodNode] = namedRule("Period<'P'('T'HH:mm:ss | YYYY'Y'MM'M'DD'D')>") {
    atomic(capture(duration) ~> createPeriodNode)
  }

  private def years = rule {
    optional(ch('-')) ~ oneOrMore(digits) ~ ch('Y')
  }

  private def months = rule {
    optional(ch('-')) ~ oneOrMore(digits) ~ ch('M')
  }

  private def weeks = rule {
    oneOrMore(digits) ~ ch('W')
  }

  private def days = rule {
    optional(ch('-')) ~ oneOrMore(digits) ~ ch('D')
  }

  private def hours = rule {
    optional(ch('-')) ~ oneOrMore(digits) ~ ch('H')
  }

  private def minutes = rule {
    optional(ch('-')) ~ oneOrMore(digits) ~ ch('M')
  }

  private def seconds = rule {
    optional(ch('-')) ~ oneOrMore(digits) ~ optional(ch('.') ~ oneOrMore(digits)) ~ ch('S')
  }

  private def year = rule {
    (anyOf("-+") ~!~ Digit ~ Digit ~ Digit ~ Digit ~ Digit) | (Digit ~ Digit ~ Digit ~ Digit)
  }

  private def month = rule {
    ot9 | (ch('1') ~ digit02) | (digit09 ~ digit09 ~ fail("'Month needs to be between [00 - 12]'"))
  }

  private def ot9 = rule {
    ch('0') ~ digit19
  }

  private def day = rule {
    ot9 | (digit12 ~ digit09) | (ch('3') ~ digit01) | (digit09 ~ digit09 ~ fail("'Day of month needs to be between [00 - 31]'"))
  }

  private def zt9 = rule {
    str("00") | ot9
  }

  private def www = rule {
    ch('W') ~!~ ((ot9 | (digit14 ~ digit09) | (ch('5') ~ digit03)) | (digit09 ~ digit09 ~ fail("'Year week needs to be between [00 - 53]'")))
  }

  private def dayOfWeek = rule {
    digit17 | (digit09 ~ fail("'Days of week needs to be between [1 - 7]'"))
  }

  private def ddd = rule {
    (str("00") ~ digit19) | (ch('0') ~ digit19 ~ digit09) | (digit13 ~ digit09 ~ digit09) | (digit09 ~ digit09 ~ digit09 ~ fail("'Day of year needs to be between [000 - 366]'"))
  }

  private def hh = rule {
    zt9 | (ch('1') ~ digit09) | (ch('2') ~ digit04) | digit09 ~ digit09 ~ fail("'Hour needs to be between [00 - 24]'")
  }

  private def mm = rule {
    zt9 | (digit15 ~ digit09) | (digit09 ~ digit09 ~ fail("'Minutes needs to be between [00 - 59]'"))
  }

  private def ss = rule {
    (zt9 | (digit15 ~ digit09) | str("60")) ~ optional(ch('.') ~ oneOrMore(digit09))
  }

  private def calendarDate = rule {
    mm ~ optional(ch('-') ~!~ day)
  }

  private def weekDate = rule {
    www ~ optional(ch('-') ~!~ dayOfWeek)
  }

  private def ordinalDate = rule {
    ddd
  }

  private def date = rule {
    year ~ optional(ch('-') ~!~ (ordinalDate | weekDate | calendarDate))
  }

  private def timeZone = namedRule("TimeZone<'Z' | '+-'HH:mm>") {
    atomic(ch('Z') | (anyOf("+-") ~!~ timeZoneOffset))
  }

  private def timeZoneOffset = rule {
    hh ~ ch(':') ~!~ mm
  }

  private def localTime = rule {
    hh ~ ch(':') ~!~ mm ~ optional(ch(':') ~!~ ss)
  }

  private def duration = rule {
    ch('P') ~!~ (timeDuration | dateDuration)
  }

  private def dateDuration = rule {
    optional(years) ~ optional(months) ~ optional(days)
  }

  private def timeDuration = rule {
    optional(days) ~ ch('T') ~ optional(hours) ~ optional(minutes) ~ optional(seconds)
  }

  private def dateTime = rule {
    date ~ ch('T') ~ localTime ~ timeZone
  }
}