package org.mule.weave.v2.grammar

import org.mule.weave.v2.grammar.literals.BaseExpression
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.parser.InvalidNameIdentifierError
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.functions.FunctionParameters
import org.mule.weave.v2.parser.ast.functions.UsingVariableAssignment
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.parboiled2.CharPredicate
import org.parboiled2.Rule1

trait Variables extends PositionTracking with Tokens with Identifiers with BaseExpression {
  this: org.mule.weave.v2.grammar.Variables with Grammar =>

  val createFullQualifiedNameIdentifier = (loader: Option[String], path: Seq[String], name: String) => {
    NameIdentifier.fromElements(path :+ name, loader)
  }
  val createVariableReferenceNode = (variable: NameIdentifier) => {
    VariableReferenceNode(variable)
  }
  val createOperatorReferenceNode = (name: String) => {
    val identifier = NameIdentifier(name)
    VariableReferenceNode(identifier)
  }
  val createVariableAssignmentNode = (variable: NameIdentifier, expr: AstNode) => {
    UsingVariableAssignment(variable, expr)
  }
  val createFunctionParameter = (annotations: Seq[AnnotationNode], variable: NameIdentifier, wtype: Option[WeaveTypeNode], expr: Option[AstNode]) => {
    FunctionParameter(variable, expr, wtype, annotations)
  }

  val createFunctionParameters = (annotations: Seq[FunctionParameter], missingParameter: Option[FunctionParameter]) => {
    FunctionParameters(annotations ++ missingParameter)
  }

  def variable: Rule1[VariableReferenceNode] = namedRule("variable") {
    atomic(variableReference | dollarVariable)
  }

  private def variableReference: Rule1[VariableReferenceNode] = rule {
    pushPosition ~ (namedRef ~> createVariableReferenceNode) ~ injectPosition
  }

  def dollarNamedVariable: Rule1[VariableReferenceNode] = rule {
    pushPosition ~ (dollar ~ namedVariable ~> createVariableReferenceNode) ~ injectPosition
  }

  def dollarVariable: Rule1[VariableReferenceNode] = rule {
    pushPosition ~ dollarNameIdentifier ~> createVariableReferenceNode ~ injectPosition
  }

  def dollarNameIdentifier: Rule1[NameIdentifier] = rule {
    pushPosition ~ (atomic(clearSB() ~ oneOrMore(dollar ~ appendSB('$')) ~ push(sb.toString.trim)) ~> createNameIdentifier) ~ injectPosition
  }

  def variableAssignment: Rule1[UsingVariableAssignment] = rule {
    pushPosition ~ (namedVariable ~!~ variableInitialValue ~> createVariableAssignmentNode) ~ injectPosition
  }

  def variableDeclaration: Rule1[NameIdentifier] = rule {
    namedVariable
  }

  def invalidNameDeclaration: Rule1[NameIdentifier] = rule {
    pushPosition ~ (clearSB() ~ oneOrMore((CharPredicate.AlphaNum | ch('_')) ~ appendSB()) ~ push(sb.toString) ~ notAKeyword ~> createInvalidVariableNameIdentifier) ~ injectPosition
  }

  def invalidFieldNameDeclaration: Rule1[NameIdentifier] = rule {
    pushPosition ~ (clearSB() ~ oneOrMore((CharPredicate.AlphaNum | ch('_')) ~ appendSB()) ~ push(sb.toString) ~> createInvalidFieldNameIdentifier) ~ injectPosition
  }

  def variableInitialValue: Rule1[AstNode] = rule {
    ws ~ assignment ~!~ ws ~ expr
  }

  def namedVariable: Rule1[NameIdentifier] = namedRule("Variable name") {
    pushPosition ~ (nameIdentifier ~> createNameIdentifier) ~ injectPosition
  }

  def namedRef: Rule1[NameIdentifier] = namedRule("Ref Name") {
    pushPosition ~ (validFQN | incompleteNameIdentifier) ~ injectPosition
  }

  private def validFQN = rule {
    (fullQualifiedName ~> createFullQualifiedNameIdentifier)
  }

  private def incompleteNameIdentifier = rule {
    oneOrMore(atomic(nameIdentifier ~ fqnSeparator())) ~> ((id) => new InvalidNameIdentifierError("Missing Name Segment", id))
  }
}
