package org.mule.weave.v2.module.commons.java.value

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.LocalDateTimeValue

import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneOffset
import java.util.Calendar
import java.util.Date

trait JavaLocalDateTimeValue extends LocalDateTimeValue with JavaValue[LocalDateTime]

class JavaTimeLocalDateTimeValue(date: LocalDateTime, val locationString: () => String) extends JavaLocalDateTimeValue {
  override def underlying()(implicit ctx: EvaluationContext): Any = date
  override def evaluate(implicit ctx: EvaluationContext): T = date
}

class JavaInstantTimeLocalDateTimeValue(instant: Instant, val locationString: () => String) extends JavaLocalDateTimeValue {

  lazy val localDateTime: LocalDateTime = {
    if (instant == Instant.MIN) {
      LocalDateTime.MIN
    } else if (instant == Instant.MAX) {
      LocalDateTime.MAX
    } else {
      LocalDateTime.ofInstant(instant, ZoneOffset.UTC)
    }
  }

  override def underlying()(implicit ctx: EvaluationContext): Any = instant
  override def evaluate(implicit ctx: EvaluationContext): T = localDateTime
}

class JavaUtilDateTimeValue(date: Date, val locationString: () => String) extends JavaLocalDateTimeValue {

  lazy val value: T = {
    val instance: Calendar = Calendar.getInstance()
    instance.setTime(date)
    LocalDateTime.of(
      instance.get(Calendar.YEAR),
      instance.get(Calendar.MONTH) + 1,
      instance.get(Calendar.DAY_OF_MONTH),
      instance.get(Calendar.HOUR_OF_DAY),
      instance.get(Calendar.MINUTE),
      instance.get(Calendar.SECOND),
      instance.get(Calendar.MILLISECOND) * 1000 * 1000)
  }

  override def evaluate(implicit ctx: EvaluationContext): T = value
  override def underlying()(implicit ctx: EvaluationContext): Any = date
}

class JavaSqlDateTimeValue(sqlDate: java.sql.Date, val locationString: () => String) extends JavaLocalDateTimeValue {

  lazy val value: T = {
    val date: Date = new Date(sqlDate.getTime)
    val instance: Calendar = Calendar.getInstance()
    instance.setTime(date)
    LocalDateTime.of(
      instance.get(Calendar.YEAR),
      instance.get(Calendar.MONTH) + 1,
      instance.get(Calendar.DAY_OF_MONTH),
      instance.get(Calendar.HOUR_OF_DAY),
      instance.get(Calendar.MINUTE),
      instance.get(Calendar.SECOND))
  }

  override def evaluate(implicit ctx: EvaluationContext): T = value
  override def underlying()(implicit ctx: EvaluationContext): Any = sqlDate
}

class JavaSqlTimestampValue(sqlTimestamp: java.sql.Timestamp, val locationString: () => String) extends JavaLocalDateTimeValue {

  lazy val value: T = sqlTimestamp.toLocalDateTime

  override def evaluate(implicit ctx: EvaluationContext): T = value
  override def underlying()(implicit ctx: EvaluationContext): Any = sqlTimestamp
}

object JavaLocalDateTimeValue {
  def apply(localDateTime: LocalDateTime, loc: () => String): JavaLocalDateTimeValue = {
    new JavaTimeLocalDateTimeValue(localDateTime, loc)
  }

  def apply(instant: Instant, loc: () => String): JavaLocalDateTimeValue = {
    new JavaInstantTimeLocalDateTimeValue(instant, loc)
  }

  def apply(date: Date, loc: () => String): JavaLocalDateTimeValue = {
    new JavaUtilDateTimeValue(date, loc)
  }

  def apply(timestamp: java.sql.Timestamp, loc: () => String): JavaLocalDateTimeValue = {
    new JavaSqlTimestampValue(timestamp, loc)
  }

  def apply(sqlDate: java.sql.Date, loc: () => String): JavaLocalDateTimeValue = {
    new JavaSqlDateTimeValue(sqlDate, loc)
  }
}
