package org.mule.weave.v2.module.commons.java.value

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.DateTimeValue

import java.time.ZoneId
import java.time.ZonedDateTime
import java.util.Calendar
import java.util.GregorianCalendar
import javax.xml.datatype.XMLGregorianCalendar

trait JavaDateTimeValue extends DateTimeValue with JavaValue[ZonedDateTime]

class JavaTimeDateTimeValue(dateTime: ZonedDateTime, val locationString: () => String) extends JavaDateTimeValue {
  override def underlying()(implicit ctx: EvaluationContext): Any = dateTime
  override def evaluate(implicit ctx: EvaluationContext): T = dateTime
}

class JavaCalendarDateTimeValue(cal: Calendar, val locationString: () => String) extends JavaDateTimeValue {

  lazy val value: T = {
    val zoneId: ZoneId = ZoneId.of(cal.getTimeZone.getID)
    val nanos: Int = cal.get(Calendar.MILLISECOND) * 1000000
    ZonedDateTime.of(
      cal.get(Calendar.YEAR),
      cal.get(Calendar.MONTH) + 1,
      cal.get(Calendar.DAY_OF_MONTH),
      cal.get(Calendar.HOUR_OF_DAY),
      cal.get(Calendar.MINUTE),
      cal.get(Calendar.SECOND),
      nanos,
      zoneId)
  }

  override def evaluate(implicit ctx: EvaluationContext): T = value
  override def underlying()(implicit ctx: EvaluationContext): Any = cal

}

class JavaXmlCalendarDateTimeValue(xmlCal: XMLGregorianCalendar, val locationString: () => String) extends JavaDateTimeValue {

  lazy val value: T = {
    val cal: GregorianCalendar = xmlCal.toGregorianCalendar
    val zoneId: ZoneId = ZoneId.of(cal.getTimeZone.getID)
    val nanos: Int = cal.get(Calendar.MILLISECOND) * 1000000
    ZonedDateTime.of(
      cal.get(Calendar.YEAR),
      cal.get(Calendar.MONTH) + 1,
      cal.get(Calendar.DAY_OF_MONTH),
      cal.get(Calendar.HOUR_OF_DAY),
      cal.get(Calendar.MINUTE),
      cal.get(Calendar.SECOND),
      nanos,
      zoneId)
  }

  override def evaluate(implicit ctx: EvaluationContext): T = value

  override def underlying()(implicit ctx: EvaluationContext): Any = xmlCal
}

object JavaDateTimeValue {
  def apply(dateTime: ZonedDateTime, loc: () => String): JavaValue[_] = {
    new JavaTimeDateTimeValue(dateTime, loc)
  }

  def apply(cal: Calendar, loc: () => String): JavaCalendarDateTimeValue = {
    new JavaCalendarDateTimeValue(cal, loc)
  }

  def apply(xmlCal: XMLGregorianCalendar, loc: () => String): JavaXmlCalendarDateTimeValue = {
    new JavaXmlCalendarDateTimeValue(xmlCal, loc)

  }
}
