package org.mule.weave.v2.module.commons.java.writer.converter

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.schema.Schema

import java.time.LocalDate
import java.time.LocalDateTime
import java.time.ZonedDateTime
import java.time.format.DateTimeFormatter
import java.util.Calendar
import java.util.Date
import scala.util.Try
import org.mule.weave.v2.module.commons.java.writer.converter.BaseJavaDataConverter.CalendarDataConverter$

class DateDataConverter extends DataConverter[Date] {
  override def convert(source: Any, schema: Option[Schema])(implicit ctx: EvaluationContext): Option[Date] = {
    source match {
      case date: Date => Some(date)
      case _ => {
        val maybeCalendar: Option[Calendar] = CalendarDataConverter$.convert(source, schema)
        if (maybeCalendar.isDefined) {
          Option(maybeCalendar.get.getTime)
        } else {
          source match {
            case str: String => formatDate(schema, str)
            case _           => None
          }
        }
      }
    }
  }

  def formatDate(schema: Option[Schema], str: String)(implicit ctx: EvaluationContext): Option[Date] = {
    val localDate: LocalDate = fallbackStringCoercion(str)
    if (localDate != null) {
      Option(DataConverter.to[Calendar](localDate, schema).get.getTime)
    } else {
      None
    }
  }

  def fallbackStringCoercion(str: String): LocalDate = {
    Try(LocalDate.parse(str))
      .orElse(Try(LocalDate.parse(str, DateTimeFormatter.ISO_WEEK_DATE)))
      .orElse(Try(LocalDate.parse(str, DateTimeFormatter.ISO_ORDINAL_DATE)))
      .orElse(Try(ZonedDateTime.parse(str).toLocalDate))
      .orElse(Try(LocalDateTime.parse(str).toLocalDate))
      .orElse(null)
      .get
  }

}
