package org.mule.weave.v2.module.commons.java.value

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.Value

import java.io.File
import java.io.InputStream
import java.lang.{ Byte => JByte }
import java.math.BigInteger
import java.nio.ByteBuffer
import java.time
import java.util
import java.util.Calendar
import java.util.Optional
import java.util.OptionalDouble
import java.util.OptionalInt
import java.util.OptionalLong
import java.util.TimeZone
import java.util.UUID
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.atomic.AtomicLong
import javax.xml.datatype.XMLGregorianCalendar
import scala.util.Try

trait JavaValueConverter {
  final def convert(value: Any, loc: () => String)(implicit ctx: EvaluationContext): Value[_] = {
    val maybeValue = Try(baseConvert(value, loc)).toOption.flatten

    if (maybeValue.isDefined) {
      maybeValue.get
    } else {
      doConvertValue(value, loc)
    }
  }

  def doConvertValue(value: Any, str: () => String)(implicit ctx: EvaluationContext): Value[_]

  private def baseConvert(value: Any, loc: () => String)(implicit ctx: EvaluationContext): Option[Value[_]] = {
    if (value == null) {
      return Some(new JavaNullValue(loc))
    }

    val mapSqlDateToDate = ctx.serviceManager.settingsService.execution().mapJavaSqlDateToDate

    value match {
      case s: String                                => Some(JavaStringValue(s, loc))
      case i: Int                                   => Some(JavaNumberValue(i, loc))
      case l: Long                                  => Some(JavaNumberValue(l, loc))
      case d: Double                                => Some(JavaNumberValue(d, loc))
      case b: Boolean                               => Some(JavaBooleanValue(b, loc))
      case list: util.Collection[_]                 => Some(new JavaListArrayValue(list, loc, this))
      case map: java.util.HashMap[Any, Any]         => Some(new JavaMapObjectValue(map, loc, this))

      //Optional
      case op: Optional[Any]                        => Some(this.convert(op.orElse(null), loc))
      case op: OptionalInt                          => Some(convert(if (op.isPresent) op.getAsInt else null, loc))
      case op: OptionalDouble                       => Some(convert(if (op.isPresent) op.getAsDouble else null, loc))
      case op: OptionalLong                         => Some(convert(if (op.isPresent) op.getAsLong else null, loc))
      //Binary
      case ba: Array[Byte]                          => Some(JavaBinaryValue(ba, loc))
      case ba: Array[JByte]                         => Some(JavaBinaryValue(ba, loc))
      case is: InputStream                          => Some(JavaBinaryValue(is, loc))
      case bb: ByteBuffer                           => Some(JavaBinaryValue(bb, loc))
      case file: File                               => Some(JavaBinaryValue(file, loc))
      case blob: java.sql.Blob                      => Some(JavaBinaryValue(blob.getBinaryStream, loc))
      case bite: Byte                               => Some(JavaBinaryValue(Array(bite), loc))
      case bite: JByte                              => Some(JavaBinaryValue(Array(bite), loc))
      //Boolean
      case b: AtomicBoolean                         => Some(JavaBooleanValue(b.get(), b, loc))
      //Strings
      case uuid: UUID                               => Some(JavaStringValue(uuid.toString, loc))
      case s: CharSequence                          => Some(JavaStringValue(s, loc))
      case ch: Char                                 => Some(JavaStringValue(ch, loc))
      case clob: java.sql.Clob                      => Some(JavaStringValue(clob.getCharacterStream, loc))
      case clob: java.io.Reader                     => Some(JavaStringValue(clob, loc))
      //Numbers
      case s: Short                                 => Some(JavaNumberValue(s, loc))
      case bi: BigInteger                           => Some(JavaNumberValue(bi, loc))
      case f: Float                                 => Some(JavaNumberValue(f, loc))
      case bd: java.math.BigDecimal                 => Some(JavaNumberValue(bd, loc))
      case at: AtomicLong                           => Some(JavaNumberValue(at.get(), at, loc))
      case at: AtomicInteger                        => Some(JavaNumberValue(at.get(), at, loc))
      case at: Number                               => Some(JavaNumberValue(at.doubleValue(), at, loc))
      //Dates
      case instant: time.Instant                    => Some(JavaLocalDateTimeValue(instant, loc))
      case localDateTime: time.LocalDateTime        => Some(JavaLocalDateTimeValue(localDateTime, loc))
      case dateTime: time.ZonedDateTime             => Some(JavaDateTimeValue(dateTime, loc))
      case localTime: time.LocalTime                => Some(JavaLocalTimeValue(localTime, loc))
      case time: time.OffsetTime                    => Some(JavaTimeValue(time, loc))
      case date: time.LocalDate                     => Some(JavaLocalDateValue(date, loc))
      case timeZone: time.ZoneOffset                => Some(JavaTimeZoneValue(timeZone, loc))
      case calendar: Calendar                       => Some(JavaDateTimeValue(calendar, loc))
      case calendar: XMLGregorianCalendar           => Some(JavaDateTimeValue(calendar, loc))
      case tz: TimeZone                             => Some(JavaTimeZoneValue(tz, loc))
      case date: java.sql.Timestamp                 => Some(JavaLocalDateTimeValue(date, loc))
      //Must go before util.Date
      case date: java.sql.Date if !mapSqlDateToDate => Some(JavaLocalDateTimeValue(date, loc))
      case date: java.sql.Date if mapSqlDateToDate  => Some(JavaLocalDateValue(date, loc))
      //Must go after sql.Date
      case date: util.Date                          => Some(JavaLocalDateTimeValue(date, loc))
      case clazz: Class[_]                          => Some(JavaStringValue(clazz.getCanonicalName, loc))
      case _                                        => None
    }
  }

}
