package org.mule.weave.v2.module.commons.java

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.service.SecurityPrivilegeViolation
import org.mule.weave.v2.model.service.WeaveRuntimePrivilege
import org.mule.weave.v2.module.commons.java.writer.ClassSchemaNode
import org.mule.weave.v2.module.commons.java.writer.exception.ParentClassNotDefined
import org.mule.weave.v2.parser.location.LocationCapable

trait JavaClassLoaderHelper {

  def loadClass(className: String, loader: Option[ClassLoader], location: LocationCapable)(implicit ctx: EvaluationContext): Class[_] = className match {
    case JavaTypesHelper.short_name   => java.lang.Short.TYPE
    case JavaTypesHelper.int_name     => Integer.TYPE
    case JavaTypesHelper.long_name    => java.lang.Long.TYPE
    case JavaTypesHelper.float_name   => java.lang.Float.TYPE
    case JavaTypesHelper.double_name  => java.lang.Double.TYPE
    case JavaTypesHelper.byte_name    => java.lang.Byte.TYPE
    case JavaTypesHelper.char_name    => Character.TYPE
    case JavaTypesHelper.boolean_name => java.lang.Boolean.TYPE
    case _                            => doLoadClass(className, loader, location)
  }

  def doLoadClass(className: String, loader: Option[ClassLoader], location: LocationCapable)(implicit ctx: EvaluationContext): Class[_]

  protected def validateRequiredLoadClassPrivilege(location: LocationCapable)(implicit ctx: EvaluationContext): Unit = {
    if (!ctx.serviceManager.securityManager.supports(WeaveRuntimePrivilege.JAVA_LOAD_CLASS, Array.empty)) {
      throw new SecurityPrivilegeViolation(WeaveRuntimePrivilege.JAVA_LOAD_CLASS, location.location())
    }
  }

  def buildClassSchemaTree(className: String, location: LocationCapable, parentClass: Option[ClassSchemaNode] = None): ClassSchemaNode = {
    //Important! For performance improvements, this method should be called with the className parameter sanitized with reg.matcher(className.get).replaceAll("") that cleans white spaces.
    def findCloseSymbol(completeClassName: String, startSymbol: Int): Int = {
      var balance: Int = 1
      completeClassName.indexWhere(
        (c: Char) =>
          c match {
            case '<' => {
              balance += 1
              false
            }
            case '>' => {
              balance -= 1
              balance == 0
            }
            case _ => false
          },
        startSymbol)
    }
    val toCharArray = className.toCharArray
    var endClassIndex = -1
    var i = 0
    while (i < toCharArray.length && endClassIndex == -1) {
      if (toCharArray(i) == '<' || toCharArray(i) == ',') {
        endClassIndex = i
      }
      i = i + 1
    }
    if (endClassIndex != -1) {
      val currentClass = new ClassSchemaNode(className.slice(0, endClassIndex))
      if (className.charAt(endClassIndex) == '<') {
        val indexEndChild = findCloseSymbol(className, endClassIndex + 1)
        val childClassName = className.slice(endClassIndex + 1, indexEndChild)
        buildClassSchemaTree(childClassName, location, Some(currentClass))
        if (parentClass.isDefined) parentClass.get.addChild(currentClass)
        if (indexEndChild != className.length - 1 && className.charAt(indexEndChild + 1) == ',') buildClassSchemaTree(className.slice(indexEndChild + 2, className.length), location, parentClass)
      } else {
        if (parentClass.isDefined) parentClass.get.addChild(currentClass) else throw new ParentClassNotDefined(location.location(), currentClass.className)
        buildClassSchemaTree(className.slice(endClassIndex + 1, className.length), location, parentClass)
      }
      currentClass
    } else {
      val currentClass = new ClassSchemaNode(className)
      if (parentClass.isDefined) parentClass.get.addChild(currentClass)
      currentClass
    }
  }
}

