package org.mule.weave.v2.module.dwb.reader

import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.core.xml.reader.ConfigurableIndexedReader
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.dwb.DefaultWeaveBinaryDataFormat
import org.mule.weave.v2.module.dwb.WeaveBinaryDataFormat
import org.mule.weave.v2.module.dwb.reader.indexed.IndexedWeaveBinaryParser
import org.mule.weave.v2.module.dwb.reader.memory.InMemoryWeaveBinaryParser
import org.mule.weave.v2.module.dwb.writer.WeaveBinaryWriter
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.reader.SourceProviderAwareReader

import java.io.DataInputStream

class WeaveBinaryReader(val sourceProvider: SourceProvider, val settings: WeaveBinaryReaderSettings = DefaultWeaveBinaryDataFormat.createReaderSettings())(implicit ctx: EvaluationContext) extends Reader with SourceProviderAwareReader {

  override def dataFormat: Option[DataFormat[_, _]] = Some(new WeaveBinaryDataFormat)

  override protected def doRead(name: String): Value[_] = {
    val ss = ctx.registerCloseable(SeekableStream(sourceProvider.asInputStream))
    val parser = if (!settings.indexedReader) {
      new InMemoryWeaveBinaryParser(name, ss)
    } else {
      new IndexedWeaveBinaryParser(name, ss)
    }
    parser.parse()
  }

}

class WeaveBinaryReaderSettings extends ConfigurableIndexedReader

trait WeaveBinaryParser {
  private val readBufferSize: Int = 4 * 1024
  private val readBuffer = new Array[Byte](readBufferSize) //4 KB

  def parse(): Value[_]

  def verifyMagicWord(dataInputStream: DataInputStream): Unit = {
    val magicWord: Int = dataInputStream.readInt()
    if (magicWord != WeaveBinaryWriter.MAGIC_WORD) {
      throw new IllegalArgumentException("Input is not a dwb file")
    }
  }

  def verifyVersion(dataInputStream: DataInputStream): Unit = {
    val version = dataInputStream.read()
    if (version != WeaveBinaryWriter.VERSION) {
      throw new IllegalArgumentException("Input was written using a different version")
    }
  }

  def verifyIndexPresence(dataInputStream: DataInputStream): Boolean = {
    val flag = dataInputStream.read()
    flag == 1
  }

  protected def getReadBuffer(): Array[Byte] = {
    readBuffer
  }

  /**
    * Use pre allocated buffer if possible, otherwise allocate new Byte Array
    */
  protected def getReadBuffer(length: Int): Array[Byte] = {
    if (length <= readBufferSize) {
      readBuffer
    } else {
      new Array[Byte](length)
    }
  }
}
