/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.metadata.cache;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.mule.common.Result;
import org.mule.devkit.api.metadata.ConnectorMetaDataCache;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;

/**
 * @author MuleSoft, Inc
 */
public class MetaDataCacheManager {

    private static final String ROOT_PATH = Paths.get(System.getProperty("java.io.tmpdir"), "connector_internal_cache").toString();
    private static final Logger log = Logger.getLogger(MetaDataCacheManager.class);

    public static ConnectorMetaDataCache getCache(String projectName, String namespace, String id, String version, String hashedConfigurables){

        PersistentMetaDataCache cache = new DefaultMetaDataCache();
        try {
            File persistentCache = new File(Paths.get(ROOT_PATH, projectName, namespace, id, version, hashedConfigurables).toUri());

            if (!persistentCache.exists()) {
                log.debug("Initializing Cache at " + persistentCache.getPath());
                initializeCacheFile(persistentCache);
            }

            if (cache.load(persistentCache) != Result.Status.SUCCESS){
                log.error("An error occurred while loading the cache " + persistentCache.getPath());
            }
        } catch (Exception e){
            log.error("An error occurred while initializing the cache " + e.getMessage());
        }
        return cache;
    }


    public static void destroyCache(String projectName, String namespace, String id, String version, String hashedConfigurables) {
        destroyCache(new File(Paths.get(ROOT_PATH, projectName, namespace, id, version, hashedConfigurables).toUri()));
    }

    public static void save(ConnectorMetaDataCache cache){

        // Workaround until STUDIO-6659 is completed.
        // Once this is done, location for a PersistentMetaDataCache should never be null
        if (cache != null && ((PersistentMetaDataCache)cache).getLocation() != null){
            try {
                File persistedCache = new File(((PersistentMetaDataCache)cache).getLocation().toUri());
                boolean forceWrite = false;
                if (!persistedCache.exists()){
                    initializeCacheFile(persistedCache);
                    forceWrite = true;
                }

                log.debug("Attempting to save Cache to " + persistedCache.getPath());
                log.debug("Overwrite cache " + forceWrite);
                if (((DefaultMetaDataCache) cache).save(persistedCache, forceWrite) != Result.Status.SUCCESS){
                    destroyCache(persistedCache);
                }
            } catch (Exception e) {
                log.error("Failed to save cache." + e.getMessage());
            }
        }
    }

    private static void initializeCacheFile(File persistentCache) throws IOException {

        persistentCache.getParentFile().mkdirs();
        if (!persistentCache.createNewFile()){
            log.error("Failed to initialize cache in path " + persistentCache.getPath());
        }
    }

    private static void destroyCache(File invalidCache) {
        if (invalidCache.exists()){
            try {
                FileUtils.forceDelete(invalidCache);
            } catch (Exception e) {
                log.error("Failed to destroy cache " + invalidCache.getPath() + ". " + e.getMessage());
            }
        }
    }

}
