/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.ws.common;

import org.apache.commons.lang.StringUtils;
import org.mule.devkit.api.metadata.exception.InvalidKeyException;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * @author Mulesoft, Inc
 */
public class WsdlSplitKey {

    private static final String ID = "id";
    private static final String OPERATION = "operation";

    private final String id;
    private final String operation;

    public WsdlSplitKey(String type, WsdlAdapter wsdlAdapter) throws Exception {
        if (StringUtils.isBlank(type)) {
            throw new InvalidKeyException("Key is empty");
        }

        if (wsdlAdapter.singleServiceDefinitionId().isPresent()) {
            id = wsdlAdapter.singleServiceDefinitionId().get();
            operation = type;

        } else {
            Map<String, String> idOp = splitIdOperationWithSeparator(type, wsdlAdapter.wsdlSeparator());
            id = idOp.get("id");
            operation = idOp.get("operation");
        }
    }

    public String id() {
        return id;
    }

    public String operation() {
        return operation;
    }

    private Map<String, String> splitIdOperationWithSeparator(String type, String separator) {

        if (StringUtils.isBlank(separator)) {
            throw new InvalidKeyException("Separator is empty");
        }

        if (!StringUtils.contains(type, separator)) {
            throw new InvalidKeyException(String.format("Key %s does not contains the expected service-operation separator %s", type, separator));
        } else if (StringUtils.countMatches(type, separator) != 1) {
            throw new InvalidKeyException(
                    String.format("Key %s contains too many service-operation separators %s, only one keySeparator occurrence is expected.", type, separator));
        }

        String escapedSeparator = Pattern.quote(separator);
        String[] splitKey = type.split(escapedSeparator);

        if (splitKey.length != 2) {
            throw new InvalidKeyException(String.format("Key %s is not valid, two non-empty parts must be separated with %s", type, separator));
        }

        if (StringUtils.isBlank(splitKey[0]) || StringUtils.isBlank(splitKey[1])) {
            throw new InvalidKeyException(String.format("Key %s is not valid, two non-empty parts must be present", type));
        }

        Map<String, String> result = new HashMap<String, String>();
        result.put(ID, splitKey[0]);
        result.put(OPERATION, splitKey[1]);

        return result;
    }

}
