/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.domain.entity;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.sdk.api.annotation.MinMuleVersion;

import java.io.InputStream;
import java.nio.charset.Charset;

/**
 * Utility factory to create {@link HttpEntity}s.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface HttpEntityFactory {

  /**
   * Builds a {@link HttpEntity} from a byte array.
   *
   * @param content the byte array.
   * @return the {@link HttpEntity}
   */
  HttpEntity from(byte[] content);

  /**
   * Builds a {@link HttpEntity} from a string.
   *
   * @param content the string.
   * @return the {@link HttpEntity}
   */
  HttpEntity from(String content, Charset charset);

  /**
   * Builds a {@link HttpEntity} from an input stream, without knowledge of the length.
   *
   * @param content the input stream.
   * @return the {@link HttpEntity}
   */
  HttpEntity from(InputStream content);

  /**
   * Builds a {@link HttpEntity} from an input stream, with knowledge of the length.
   *
   * @param content       the byte array.
   * @param contentLength the (known) content length.
   * @return the {@link HttpEntity}
   */
  HttpEntity from(InputStream content, Long contentLength);

  /**
   * Builds an empty {@link HttpEntity}.
   *
   * @return the {@link HttpEntity}
   */
  HttpEntity emptyEntity();

  /**
   * Builds a {@link FeedableHttpEntity}.
   *
   * @return a new {@link FeedableHttpEntity}.
   * @since 0.13
   */
  default FeedableHttpEntity feedable() {
    throw new UnsupportedOperationException("This version of Mule Runtime does not support creating feedable entities.");
  }
}
