/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.domain.entity;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;

import org.mule.runtime.api.util.MultiMap;
import org.mule.sdk.api.annotation.MinMuleVersion;

import java.io.IOException;
import java.nio.ByteBuffer;

/**
 * An object that can be fed after it was created. Implementers can leverage it for non-blocking streaming and for trailers
 * sending.
 *
 * <pre>
 * {@code
 * Feedable feedable = new FeedableImpl();
 * feedable.feed(firstFragment);
 * feedable.feed(secondFragment);
 * feedable.feed(thirdFragment);
 * feedable.complete();
 * }
 * </pre>
 *
 * @since 0.13
 */
@MinMuleVersion("4.11.0")
public interface FeedableHttpEntity extends HttpEntity {

  /**
   * Writes a fragment of the body to this entity abstraction.
   *
   * @param data buffer containing the data.
   * @throws IOException if an error occurs while writing the data. For example, when the entity was already completed, both
   *                     normally or with an error.
   */
  void feed(ByteBuffer data) throws IOException;

  /**
   * Signals the entity with an error. If this method is called instead of completing normally (with {@link #complete()}),
   * implementors may discard the previously fed data. Future calls to {@link #feed(ByteBuffer)} must fail.
   *
   * @param exception the error.
   * @throws IllegalStateException if called after it was already completed, both normally or with an error.
   */
  void error(Exception exception);

  /**
   * After {@link #feed}ing all the data fragments, user must complete the entity by calling this method or
   * {@link #completeWithTrailers}.
   *
   * @throws IOException           if an error occurs during the completion of the entity.
   * @throws IllegalStateException if called after it was already completed, both normally or with an error.
   */
  default void complete() throws IOException {
    completeWithTrailers(emptyMultiMap());
  }

  /**
   * After {@link #feed}ing all the data fragments, user must complete the entity by calling this method or {@link #complete}.
   *
   * @param trailers the trailers.
   *
   * @throws IOException           if an error occurs during the completion of the entity.
   * @throws IllegalStateException if called after it was already completed, both normally or with an error.
   */
  void completeWithTrailers(MultiMap<String, String> trailers) throws IOException;
}
