/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.catalog.api;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataType;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * Default types that will be accessible when accessing the {@link TypeResolver#resolveType(String)} method.
 * <p/>
 * The complete list of primitive types can be read from the {@link #PRIMITIVE_TYPES} map's keys (each value represents the type
 * in the {@link MetadataType} model)
 *
 * @since 1.0
 */
public class PrimitiveTypesTypeLoader implements TypeLoader {

  /**
   * Unmodifiable map with the supported primitive types
   */
  final static public Map<String, MetadataType> PRIMITIVE_TYPES;

  public static final String STRING = "string";
  public static final String BOOLEAN = "boolean";
  public static final String NUMBER = "number";
  public static final String DATE = "date";
  public static final String DATETIME = "datetime";
  public static final String LOCALDATETIME = "localdatetime";
  public static final String TIME = "time";
  public static final String LOCALTIME = "localtime";
  public static final String TIMEZONE = "timezone";
  public static final String BINARY = "binary";
  public static final String ANY = "any";
  public static final String REGEX = "regex";

  static {
    Map<String, MetadataType> types = new HashMap<>();
    types.put(STRING, BaseTypeBuilder.create(JAVA).stringType().build());
    types.put(BOOLEAN, BaseTypeBuilder.create(JAVA).booleanType().build());
    types.put(NUMBER, BaseTypeBuilder.create(JAVA).numberType().build());
    types.put(DATE, BaseTypeBuilder.create(JAVA).dateType().build());
    types.put(DATETIME, BaseTypeBuilder.create(JAVA).dateTimeType().build());
    types.put(LOCALDATETIME, BaseTypeBuilder.create(JAVA).localDateTimeType().build());
    types.put(TIME, BaseTypeBuilder.create(JAVA).timeType().build());
    types.put(LOCALTIME, BaseTypeBuilder.create(JAVA).localTimeType().build());
    types.put(TIMEZONE, BaseTypeBuilder.create(JAVA).timeZoneType().build());
    types.put(BINARY, BaseTypeBuilder.create(JAVA).binaryType().build());
    types.put(ANY, BaseTypeBuilder.create(JAVA).anyType().build());
    types.put(REGEX, BaseTypeBuilder.create(JAVA).regexType().build());
    PRIMITIVE_TYPES = Collections.unmodifiableMap(types);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    return Optional.ofNullable(PRIMITIVE_TYPES.get(typeIdentifier));
  }
}
