/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.TupleType;
import org.mule.metadata.api.model.impl.DefaultTupleType;
import org.mule.metadata.internal.utils.LazyValue;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

public class TupleTypeBuilder extends AbstractBuilder<TupleType>
    implements TypeBuilder<TupleType>, WithAnnotation<TupleTypeBuilder> {

  private final List<TypeBuilder<?>> types = new LinkedList<>();
  private final LazyValue<TupleType> result = new LazyValue<>();

  protected TupleTypeBuilder(MetadataFormat format) {
    super(format);
  }

  public BaseTypeBuilder of() {
    final BaseTypeBuilder builder = new BaseTypeBuilder(format);
    this.types.add(builder);
    return builder;
  }


  public TupleTypeBuilder id(String typeIdentifier) {
    return with(new TypeIdAnnotation(typeIdentifier));
  }

  public TupleTypeBuilder of(TypeBuilder<?> typeBuilder) {
    this.types.add(typeBuilder);
    return this;
  }

  public TupleTypeBuilder description(String lang, String content) {
    return with(new DescriptionAnnotation(content, lang));
  }

  public TupleTypeBuilder description(String content) {
    return with(new DescriptionAnnotation(content));
  }

  public TupleTypeBuilder label(String label) {
    return with(new LabelAnnotation(label));
  }

  public TupleTypeBuilder with(TypeAnnotation extension) {
    this.addExtension(extension);
    return this;
  }

  @Override
  public TupleType build() {
    if (types.isEmpty()) {
      throw new RuntimeException("Tuple needs at least one type");
    }
    final ArrayList<MetadataType> types = new ArrayList<>();
    // This is to avoid stack overflow we create the object with no fields
    boolean needsInit = !result.isDefined();
    final TupleType tupleType = result.get(() -> new DefaultTupleType(types, format, annotations));
    if (needsInit) {
      final List<MetadataType> createdTypes = this.types.stream().map((builder) -> builder.build()).collect(Collectors.toList());
      types.addAll(createdTypes);
    }
    return tupleType;
  }

}
