/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.extension.api.annotation.ExpressionFunctions;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.Sources;
import org.mule.runtime.extension.api.annotation.connectivity.ConnectionProviders;
import org.mule.runtime.module.extension.api.loader.java.type.ConfigurationElement;
import org.mule.runtime.module.extension.api.loader.java.type.ConnectionProviderElement;
import org.mule.runtime.module.extension.api.loader.java.type.FunctionContainerElement;
import org.mule.runtime.module.extension.api.loader.java.type.OperationContainerElement;
import org.mule.runtime.module.extension.api.loader.java.type.SourceElement;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.TypeElement;

/**
 * {@link ConfigurationElement}
 *
 * @since 1.0
 */
public class ConfigurationASTElement extends ASTType implements ConfigurationElement {

  public ConfigurationASTElement(TypeElement typeElement, ProcessingEnvironment processingEnvironment) {
    super(typeElement, processingEnvironment);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConnectionProviderElement> getConnectionProviders() {
    return getValueFromAnnotation(ConnectionProviders.class).map(valueFetcher -> valueFetcher
        .getClassArrayValue(ConnectionProviders::value)
        .stream()
        .map(connElem -> (ConnectionProviderElement) new ConnectionProviderASTElement(((ASTType) connElem).getElement().get(),
                                                                                      processingEnvironment))
        .collect(toList())).orElse(emptyList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FunctionContainerElement> getFunctionContainers() {
    return getValueFromAnnotation(ExpressionFunctions.class).map(valueFetcher -> valueFetcher
        .getClassArrayValue(ExpressionFunctions::value)
        .stream()
        .map(functionElem -> (FunctionContainerElement) new FunctionContainerASTElement(((ASTType) functionElem).getElement()
            .get(),
                                                                                        processingEnvironment))
        .collect(toList())).orElse(emptyList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<SourceElement> getSources() {
    List<SourceElement> sources = new ArrayList<>();
    collectSourceElements(sources, Sources.class, Sources::value);
    collectSourceElements(sources, org.mule.sdk.api.annotation.Sources.class, org.mule.sdk.api.annotation.Sources::value);

    return sources;
  }

  private <A extends Annotation> void collectSourceElements(List<SourceElement> accumulator,
                                                            Class<A> annotationClass,
                                                            Function<A, Class[]> extractionFunction) {
    getValueFromAnnotation(annotationClass).ifPresent(valueFetcher -> valueFetcher
        .getClassArrayValue(extractionFunction)
        .forEach(type -> accumulator.add(new SourceElementAST(type.getElement().get(), processingEnvironment))));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<OperationContainerElement> getOperationContainers() {
    return getValueFromAnnotation(Operations.class).map(valueFetcher -> valueFetcher
        .getClassArrayValue(Operations::value)
        .stream()
        .map(operationElem -> (OperationContainerElement) new OperationContainerElementAST(((ASTType) operationElem)
            .getElement().get(),
                                                                                           processingEnvironment))
        .collect(toList())).orElse(emptyList());
  }
}
