/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.runtime.module.embedded.internal.classloading;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.lang3.builder.ToStringBuilder.reflectionToString;
import static org.apache.commons.lang3.builder.ToStringStyle.MULTI_LINE_STYLE;

import java.util.Arrays;
import java.util.Set;

/**
 * Determines if a given class or resource is exported in a plugin classloader
 */
// TODO MULE-11882 - Consolidate classloading isolation
public class ClassLoaderFilter {

  private static final String EMPTY_PACKAGE = "";
  private static final char RESOURCE_SEPARATOR = '/';
  public static final String CLASS_PACKAGE_SPLIT_REGEX = "\\.";
  public static final String RESOURCE_PACKAGE_SPLIT_REGEX = "/";
  private final Set<String> bootPackages;
  private Set<String> exportedResources;

  public ClassLoaderFilter(Set<String> bootPackages, Set<String> exportedResources) {
    this.bootPackages = bootPackages;
    this.exportedResources = exportedResources;
  }

  public boolean exportsClass(String name) {
    return isExportedBootPackage(name, CLASS_PACKAGE_SPLIT_REGEX);
  }

  public boolean exportsResource(String name) {
    checkArgument(name != null, "Resource name cannot be null");
    final String sanitizeResourceName = sanitizeResourceName(name);
    return exportedResources.contains(sanitizeResourceName);
  }

  // TODO(MULE-11882): copy of DefaultArtifactClassLoaderFilter#sanitizeResourceName -> Consolidate classloading isolation
  private String sanitizeResourceName(String resourceName) {
    String sanitizedResource = "";
    if (resourceName.length() > 0) {
      sanitizedResource = (resourceName.charAt(0) == RESOURCE_SEPARATOR) ? resourceName.substring(1) : resourceName;
      if (sanitizedResource.length() > 0) {
        sanitizedResource = sanitizedResource.charAt(sanitizedResource.length() - 1) == RESOURCE_SEPARATOR
            ? sanitizedResource.substring(0, sanitizedResource.length() - 1)
            : sanitizedResource;
      }
    }
    return sanitizedResource;
  }

  private boolean isExportedBootPackage(String name, String splitRegex) {
    boolean exported = false;
    final String[] splitName = name.split(splitRegex);
    final String[] packages = Arrays.copyOf(splitName, splitName.length - 1);
    String candidatePackage = "";

    for (String currentPackage : packages) {
      if (candidatePackage.length() != 0) {
        candidatePackage += ".";
      }
      candidatePackage += currentPackage;

      if (bootPackages.contains(candidatePackage)) {
        exported = true;
        break;
      }

    }
    return exported;
  }

  @Override
  public String toString() {
    return reflectionToString(this, MULTI_LINE_STYLE);
  }
}
