/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security;

import com.google.common.collect.ImmutableMap;
import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;

import static java.util.Optional.of;
import static org.apache.wss4j.common.ConfigurationConstants.ADD_USERNAMETOKEN_CREATED;
import static org.apache.wss4j.common.ConfigurationConstants.ADD_USERNAMETOKEN_NONCE;
import static org.apache.wss4j.common.ConfigurationConstants.USER;
import static org.apache.wss4j.common.ext.WSPasswordCallback.USERNAME_TOKEN;
import static org.apache.wss4j.dom.message.token.UsernameToken.PASSWORD_TYPE;

/**
 * Provides the capability to authenticate using Username and Password with a SOAP service by adding the UsernameToken element in
 * the SOAP request.
 *
 * @since 1.0
 */
public class WssUsernameTokenSecurityStrategy implements SecurityStrategy {

  /**
   * The username required to authenticate with the service.
   */
  private String username;

  /**
   * The password for the provided username required to authenticate with the service.
   */
  private String password;

  /**
   * A {@link PasswordType} which qualifies the {@link #password} parameter.
   */
  private PasswordType passwordType;

  /**
   * Specifies a if a cryptographically random nonce should be added to the message.
   */
  private boolean addNonce;

  /**
   * Specifies if a timestamp should be created to indicate the creation time of the message.
   */
  private boolean addCreated;

  public WssUsernameTokenSecurityStrategy(String user, String password, PasswordType passwordType, boolean addNonce,
                                          boolean addCreated) {
    this.username = user;
    this.password = password;
    this.passwordType = passwordType;
    this.addNonce = addNonce;
    this.addCreated = addCreated;
  }

  public SecurityStrategyType securityType() {
    return SecurityStrategyType.OUTGOING;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return of(new WSPasswordCallbackHandler(USERNAME_TOKEN,
                                            cb -> {
                                              if (cb.getIdentifier().equals(username)) {
                                                cb.setPassword(password);
                                              }
                                            }));
  }

  @Override
  public String securityAction() {
    return SecurityActions.USERNAME_TOKEN;
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    return ImmutableMap.<String, Object>builder()
        .put(USER, username)
        .put(PASSWORD_TYPE, passwordType.getType())
        .put(ADD_USERNAMETOKEN_NONCE, String.valueOf(addNonce))
        .put(ADD_USERNAMETOKEN_CREATED, String.valueOf(addNonce)).build();
  }
}
